#pragma once
/**
 *  @file socket.h
 *  @brief Socket-specific functions.
 *
 *  Object-like sockets representation.
 */

#include <errno.h>

#if (defined(__unix__) || defined(unix)) && !defined(USG)
#   include <sys/param.h> // For @c BSD macro
#endif

#if !defined(BSD) && !defined(__APPLE__) && !defined(__CYGWIN__)
#   define __LINUX
#endif

#ifdef __LINUX
#   define _GNU_SOURCE // For @c accept4(). See feature_test_macros(7)

#   include <features.h>
#   if !__GLIBC_PREREQ(2, 10)
#       define __GLIBC_LEGACY
#   endif
#endif

#include <sys/socket.h>

#include "util.h"


/** Socket base object. */
typedef struct {
    int                 fd;
    struct sockaddr_un  addr;
} Socket;

/** Creates new Unix-domain socket and starts listening on it. */
static inline Socket* newServerSocket(const char *path, int backlog) {
    Socket* sock = sfmalloc(sizeof(Socket));
#ifndef __LINUX
    // That's portability, dude!
    sock->fd = socket(AF_UNIX, SOCK_STREAM, 0);
    fcntl(sock->fd, F_SETFL, fcntl(sock->fd, F_GETFL) | FD_CLOEXEC);
#elif !defined(__GLIBC_LEGACY)
    sock->fd = socket(AF_UNIX, SOCK_STREAM | SOCK_CLOEXEC, 0);
#else
    // Lets close appropriate file descriptors on our own on legacy systems.
    sock->fd = socket(AF_UNIX, SOCK_STREAM, 0);
#endif
    if (sock->fd < 0)
        fatal("Unable to create a socket.");

    memset(&sock->addr, 0, sizeof(sock->addr));
    sock->addr.sun_family = AF_UNIX;
    strcpy(sock->addr.sun_path, path);
    if (bind(sock->fd, (struct sockaddr*)&sock->addr, sizeof(sock->addr)) < 0)
        fatal("Unable to bind a socket");

    listen(sock->fd, backlog);
    return sock;
}

/** Accepts new peer connection on the given server socket object.
 *  @param srv  Socket server object.
 *  @return     Peer socket object. The object should be released with @code{destroyPeerSocket}.
 */
static inline Socket* acceptPeer(Socket* srv) {
    Socket* peer = sfmalloc(sizeof(Socket));
    socklen_t addr_sz = sizeof(peer->addr);
#if defined(__LINUX) && !defined(__GLIBC_LEGACY)
    peer->fd = accept4(srv->fd, (struct sockaddr*)&peer->addr, &addr_sz, SOCK_CLOEXEC);
#else
    // BSD systems inherits flags from the main socket on @c accept().
    peer->fd = accept(srv->fd, (struct sockaddr*)&peer->addr, &addr_sz);
#endif
    if (peer->fd < 0)
        fatal("Error accepting new connection");
    return peer;
}

/** Sends the data given into the socket. Returns amount of sent data. */
static inline ssize_t socketSend(Socket* sock, uint8_t* data, size_t size) {
    while (true) {
        ssize_t rc = send(sock->fd, data, size, 0);
        if (rc >= 0)
            return rc;
        if (errno == EINTR)
            continue;
        if (errno == ECONNRESET || errno == ENOTCONN || errno == EPIPE)
            return -1;
        if (errno == EAGAIN || errno == EWOULDBLOCK)
            break;
        fatal("Unable to send data into the socket");
    }
    return 0;
}

/** Sends all the data given into the socket. In case of the socket will be closed,
 *  the function will terminate the program. */
static inline void socketSendAll(Socket* sock, uint8_t* data, size_t size) {
    size_t sent = 0;
    while (size > sent) {
        ssize_t rc = send(sock->fd, data + sent, size - sent, 0);
        if (rc < 1) {
            if (errno == EINTR)
                continue;
            else
                fatal("Unable to send data into the socket");
        }
        sent += rc;
    }
}

/** Receives data on given socket into given buffer of given size.
 *  @param sock Socket object to be used.
 *  @param data Buffer to be used.
 *  @param size Available buffer space.
 *  @return     Amount of data received.
 */
static inline ssize_t socketRecv(Socket* sock, uint8_t* data, size_t size) {
    if (!size) fatal("Socket read with zero buffer size requested");
    while (true) {
        ssize_t rc = recv(sock->fd, data, size, 0);
        if (rc >= 0)
            return rc;
        if (errno == EWOULDBLOCK || errno == EINTR)
            return -1;
        // Treat @c EINTR here as connection reset - it actually can be given on alarm signal
        // during the initial handshake or at the moment the process is signaled by some
        // external tool (master, for example). Anyway, in any case it should treat it
        // as socket connection reset.
        if (errno != ECONNRESET && errno != ENOTCONN)
            fatal("Unable to receive data from the socket");
        break;
    }
    return 0;
}

/** Close the peer socket and release the object. */
static inline void releasePeerSocket(Socket* sock) {
    if (sock->fd > 0) close(sock->fd);
    free(sock);
}

/** Close the server socket and release the object. */
static inline void releaseServerSocket(Socket* sock) {
    if (sock->fd > 0) close(sock->fd);
    unlink(sock->addr.sun_path);
    free(sock);
}
