#pragma once
/**
 *  @file util.h
 *  @brief Utility functions.
 *
 *  The file contains various miscellaneous utility functions.
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#if (defined(__unix__) || defined(unix)) && !defined(USG)
#include <sys/param.h> // For @c BSD macro
#endif

extern char **environ;


/** Minimum of two numerics 'function'. */
#define min(a, b) \
    ({ \
        __typeof__ (a) _a = (a); \
        __typeof__ (b) _b = (b); \
        _a < _b ? _a : _b; \
    })
/** Maximum of two numerics 'function'. */
#define max(a, b) \
    ({ \
        __typeof__ (a) _a = (a); \
        __typeof__ (b) _b = (b); \
        _a > _b ? _a : _b; \
    })


/** Forward declaration for string list entry structure type. */
typedef struct StringListItem StringListItem;
/** String list entry structure. */
struct StringListItem {
    /** Entry data. */
    char*           data;
    /** Entry data size. */
    size_t          size;
    /** Pointer to next list element. */
    StringListItem* next;
};

/** String list class. */
typedef struct {
    StringListItem* first;
    StringListItem* last;
    size_t          size;
} StringList;


/** Fatal error handler. Prints the given message with @code{errno} description and exists.
 *  @param msg  Message to print.
 *  @param code Process exit code.
 */
static inline void fatal(const char* msg) {
    perror(msg);
    exit(-1);
}

/** Simple safe wrapper around malloc with process termination on failure. */
static inline void* sfmalloc(size_t amount) {
    static char buf[0x100];
    void* ret = malloc(amount);
    if (ret) return ret;
    snprintf(buf, sizeof(buf), "Unable to allocate %zu bytes", amount);
    fatal(buf); // Never returns.
    return NULL;
}

/** Allocates a new string list object. */
static inline StringList* newStringList() {
    StringList* ret = sfmalloc(sizeof(StringList));
    ret->first = ret->last = NULL;
    ret->size = 0;
    return ret;
}

/** Appends new item to the list by copying it.
 *  @param list List to be updated.
 *  @param data Data to be appended.
 *  @param len  Amount of data to be copied.
 *  @return     New list cumulative data size.
 */
static inline size_t appendString2List(StringList* list, const char* data, size_t len) {
    StringListItem* itm = sfmalloc(sizeof(StringListItem));
    itm->data = strndup(data, len); assert(itm->data);
    itm->next = NULL;
    itm->size = len;
    if (!list->first) {
        list->first = list->last = itm;
    } else {
        list->last->next = itm;
        list->last = itm;
    }
    list->size += len;
    return list->size;
}

/** Releases the entire string list with data. */
static inline void releaseStringList(StringList* list) {
    if (!list) return;
    StringListItem* item = list->first;
    while (item) {
        free(item->data);
        StringListItem* next = item->next;
        free(item);
        item = next;
    }
    free(list);
}

/** Relases array of strings, including both pointed strings and the array itself. */
static inline void releaseArray(char** array) {
    char** ptr = array;
    while (ptr && *ptr) free(*ptr++);
    free(array);
}

/** Will point on `argv` area. */
static char* ps_buffer = NULL;
/** Available process name length. */
static size_t ps_buffer_size = 0;
/** Used to minimize length of clobber. */
static size_t last_status_len;


/** Releases allocated environment array. */
static inline void releaseEnvironment() {
    if (ps_buffer) releaseArray(environ);
}

/** Initializes process title changes. */
static inline void initProcessTitle(int argc, char** argv) {
#ifdef BSD
    // Nothing to do with BSD systems.
    return;
#endif
    // We're going to overwrite the `argv` area, count the available space.
    // Also move the environment to make additional room.

    // Check for contiguous `argv` strings.
    int i = 0;
    char* end_of_area = NULL;
    for (i = 0; i < argc; i++) {
        if (!i || end_of_area + 1 == argv[i])
            end_of_area = argv[i] + strlen(argv[i]);
    }

    if (!end_of_area) // Be a paranoid
        return;

    // Clobber environ.
    // Check for contiguous environ strings following `argv`.
    for (i = 0; environ[i] != NULL; i++) {
        if (end_of_area + 1 == environ[i])
            end_of_area = environ[i] + strlen(environ[i]);
    }

     // Move the environment out of the way.
    char** new_environ = (char**)sfmalloc((i + 1) * sizeof(char*));
    for (i = 0; environ[i] != NULL; i++)
        new_environ[i] = strdup(environ[i]);
    new_environ[i] = NULL;
    environ = new_environ;

    ps_buffer = argv[0];
    last_status_len = ps_buffer_size = end_of_area - argv[0];
    atexit(releaseEnvironment);
}

/** Sets process title.
 *  @param name     Process main name
 *  @param states   Process state(s) suffix.
 */
static inline void processTitle(const char* name, const char* const* states) {
    char title[0xFF];
    strcpy(title, "-liner- ");
    if (states && states[0]) {
        char* ptr = title + strlen(title) + 1;
        *(ptr - 1) = '[';
        for (int i = 0; ptr - title < 0xFD && states[i]; ++i) {
            if (i) {
                strncpy(ptr, ", ", 2);
                ptr += 2;
            }
            size_t len = min(strlen(states[i]), (size_t)(0xFD - (ptr - title)));
            strncpy(ptr, states[i], len);
            ptr += len;
        }
        strncpy(ptr, "] ", 3); // Including termination zero symbol.
    }
    strcpy(title + strlen(title), name);
#ifdef BSD
    setproctitle("%s", title);
#else // Linux? Its "just" enough to update `argv[0]`
    if (!ps_buffer) return;
    size_t len = min(strlen(title), ps_buffer_size - 1);
    strncpy(ps_buffer, title, len);
    // clobber remainder of old status string
    if (last_status_len > len)
        memset(ps_buffer + len, '\0', last_status_len - len);
    last_status_len = len;
#endif
}
