# we use it for cygwin only !!!
from __future__ import absolute_import
import os
import sys
import fcntl
import msgpack
import time
import errno


PROCTITLE_FILENAME = '/tmp/cygwin_proctitles'


class FileLockTimeoutException(Exception):
    pass


class FileLock(object):
    def __init__(self, file_name, timeout=10, delay=.1):
        self._file_name = file_name
        self._timeout = timeout
        self._delay = delay
        self._handle = None
        self._locked = False

    def acquire(self):
        if self._locked:
            return

        start_time = time.time()

        # Open file
        mode = 'r+b' if os.path.exists(self._file_name) else 'w+b'
        self._handle = open(self._file_name, mode)

        while True:
            try:
                fcntl.flock(self._handle, fcntl.LOCK_EX | fcntl.LOCK_NB)
                break
            except IOError as error:
                if error.errno not in [errno.EACCES, errno.EAGAIN]:
                    raise

                if (time.time() - start_time) >= self._timeout:
                    raise FileLockTimeoutException('Timeout occured')

                time.sleep(self._delay)

        self._locked = True

    def release(self):
        if self._locked:
            fcntl.flock(self._handle, fcntl.LOCK_UN)
            self._locked = False

        if self._handle:
            self._handle.close()
            self._handle = None

    @property
    def handle(self):
        return self._handle

    def __enter__(self):
        self.acquire()
        return self

    def __exit__(self, typ, value, traceback):
        self.release()

    def __del__(self):
        self.release()


def setproctitle(s):
    if sys.platform != 'cygwin':
        return

    try:
        with FileLock(PROCTITLE_FILENAME) as f:
            pid = os.getpid()
            data = msgpack.load(f.handle) or {}
            data[pid] = s
            f.handle.seek(0)
            msgpack.dump(data, f.handle)
    except FileLockTimeoutException:
        pass
    except TypeError:
        # failed to load msgpack, remove broken file
        os.remove(PROCTITLE_FILENAME)



def getproctitle():
    if sys.platform != 'cygwin':
        return ''

    try:
        with FileLock(PROCTITLE_FILENAME) as f:
            pid = os.getpid()
            data = msgpack.load(f.handle) or {}
            return data[pid]
    except FileLockTimeoutException:
        return ''
    except TypeError:
        # failed to load msgpack, remove broken file
        os.remove(PROCTITLE_FILENAME)
