#!/usr/bin/env bash

set -e

SKYNET_ROOT=$(realpath $(dirname $0)/..)
PACKAGES_PATH=$SKYNET_ROOT/packages
BUILDDIR=$(realpath "$PWD")/build
BIN_FN=skynet.bin
VERSION=
REVISION=
SKYCORE_REV=
URL=
UPDATE_SERVICES=1

# shellcheck source=build_v2/_utils.sh
source "${PACKAGES_PATH}"/build_v2/_utils.sh

# sort order depends on LANG and order is very important in skynet.bin tar
# so we set it here to be sure that everything will be ok
export LANG=C

help() {
    echo "Usage: ./make.sh [options]"
    echo "Options:"
    echo "  -s, --skynet-root       Skynet root directory (../ by default)"
    echo "  -b, --build-dir         Build directory to use (by default ./build)"
    echo "  -a, --arch              Force arch (detected as ${ARCH})"
    echo "  -o, --out               Result binary filename (by default ${BIN_FN})"
    echo "  --version               Set version to use"
    echo "  --svnurl                SVN Url to use"
    echo "  --svnrevision           SVN Revision to use"
    echo "  --no-service-update     Do not check out services from *.link"
}


parseargs() {
    while true; do
        case $1 in
            -s | --skynet-root)     SKYNET_ROOT=$2;     shift 2    ;;
            -a | --arch)            ARCH=$2;            shift 2    ;;
            -b | --build-dir)       BUILDDIR=$2;        shift 2    ;;
            -o | --out)             BIN_FN=$2;          shift 2    ;;
            --version)              VERSION=$2;         shift 2    ;;
            --svnurl)               URL=$2;             shift 2    ;;
            --svnrevision)          REVISION=$2;        shift 2    ;;
            --no-service-update)    UPDATE_SERVICES=0;  shift 1    ;;
            --no-old)                                   shift 1    ;;
            '')                     break                          ;;
            *)                      echo -e "Unknown option: $1\n"; help; exit 1;;
        esac
    done
}

check_skynet_sources() {
    if [ ! -e "${SKYNET_ROOT}"/startup/build.py ]; then
        echo "Cant find ${SKYNET_ROOT}/startup/build.py script!"
        exit 1
    fi
}

detect_version() {
    if [ -z "$URL" ] || [ -z "$REVISION" ]; then
        cd "${SKYNET_ROOT}"

        if [ -e .svn ]; then
            local CMD="svn info ."
        elif [ -e .hg ]; then
            local CMD="hg svn info ."
        else
            local found=0
            local part
            part=$(pwd)

            while true; do
                local npart
                npart="$(dirname "$part")"
                [ "$npart" == "$part" ] && break
                part="$npart"
                if [ -e "$part/.svn" ]; then
                    local CMD="svn info ."
                    found=1
                    break
                fi
            done

            [ $found -eq 0 ] && return
        fi

        [ ! "$CMD" == "" ] && ${CMD} > "${BUILDDIR}"/svninfo 2>&1
        [ ! -e "$BUILDDIR/svninfo" ] && return

        URL=$(grep '^URL:' "${BUILDDIR}/svninfo" | awk '{ print $2; }')
        URL=${URL%/skynet}  # strip /skynet at the end, see SKYNET-353

        # Strip user info from svn url if any. Also converts arcadia.yandex.net => arcadia.yandex.ru
        [[ $URL =~ (^.*)//.*@arcadia(.*$) ]] && \
            URL="${BASH_REMATCH[1]}//arcadia${BASH_REMATCH[2]}"

        REVISION=$(grep 'Last Changed Rev' "${BUILDDIR}/svninfo" | awk -F ': ' '{ print $2; }')

        cd - >/dev/null
    fi

    echo "svn: $URL@$REVISION"

    if [ -z "$VERSION" ]; then
        local TYPE
        TYPE=$(echo "${URL}" | cut -d / -f 5)
        if [ "$TYPE" = "trunk" ]; then
            VERSION=trunk
        elif [ "$TYPE" = "branches" ] || [ "$TYPE" = "tags" ]; then
            VERSION=$(echo "${URL}" | cut -d / -f 7)
        fi
    fi

    SKYCORE_REV="$URL@$REVISION"

    echo "version: $VERSION"
    echo "skycore revision: $SKYCORE_REV"
}

prepare_builddir() {
    rm -rf "${BUILDDIR}"
    mkdir -p "${BUILDDIR}"
}


unpack_python() {
    local archive=${PACKAGES_PATH}/python-dists/skynet-deps_${ARCH}.tar.gz

    if [ ! -e "${archive}" ]; then
        local SANDBOX_RESOURCE=2362567336  # 18.0 branch

        echo "There is no python dist archive for your architecture ${ARCH}"
        echo "Trying to download sandbox-resource:$SANDBOX_RESOURCE..."

        local dstpath="${PACKAGES_PATH}/python-dists"
        rm -rf "$dstpath" && mkdir -p "$dstpath" && cd "$dstpath"

        echo "Prepared python-dists folder, executing sky get..."

        sky get -u -w sandbox-resource:$SANDBOX_RESOURCE
        echo "Downloaded with return code $?"

        mv skynet-deps/* .
        rmdir skynet-deps
        cd - >/dev/null

        echo "Touching sandbox resource..."
        resp=$(curl \
            -s https://sandbox.yandex-team.ru/sandbox/xmlrpc \
            -X POST \
            -d "<methodCall><methodName>touch_resource</methodName><params><param><value>$SANDBOX_RESOURCE</value></param></params></methodCall>" \
            -H "Content-Type: text/xml"
        )
        if echo "$resp" | grep "<boolean>1</boolean>" >/dev/null; then
            echo "   touched OK"
        else
            echo "   failed"
            echo "$resp" | grep value
        fi

        if [ ! -e "${archive}" ]; then
            echo "Failed to grab python dist (searching for ${archive})"
            echo "We have these:"
            ls -l "$dstpath"
            exit 1
        fi
    fi

    echo "Unpacking ${archive}"
    tar -C "${BUILDDIR}" -xzf "${archive}" || exit 1


    PYTHON=${BUILDDIR}/python/bin/python
    if (which realpath >/dev/null 2>&1); then
        PYTHON=$(realpath "${PYTHON}")
    else
        PYTHON=$(readlink -e "${PYTHON}")
    fi

    if [[ "$(uname)" = "CYGWIN"* ]]; then
        export PATH=${BUILDDIR}/python/bin:$PATH
    fi
}


build_skynet_bundles() {
    echo "Running skynet build script..."
    local excl=()
    if [[ "$(uname)" = "CYGWIN"* ]]; then
        excl=(-e skybone-mds -e lacmus -e portoshell -e skybone-coord -e netstate -e kqueue -e heartbeat -e heartbeat-server -e logrotated -e resolver -e shardstates)
    fi

    if [ "$(uname -p)" = aarch64 ]; then
        excl=(-e skybone-coord)
    fi

    if [ ${UPDATE_SERVICES} = 1 ]; then
        update=""
    else
        update="--no-service-update"
    fi

    ${PYTHON} -u "${SKYNET_ROOT}/startup/build.py" -b "${BUILDDIR}/skynet" --variant bundle "${excl[@]}" ${update} || exit 1
}


build_selftar() {
    echo "Building self-extracting binary..."
    rm -rf "${BUILDDIR}/star"
    mkdir -p "${BUILDDIR}/star"

    ln -sf $(realpath "${PACKAGES_PATH}/build_v2/install/skyinstall.py") "${BUILDDIR}/star/skyinstall.py"
    ln -sf $(realpath "${PACKAGES_PATH}/build_v2/install/selftest.py") "${BUILDDIR}/star/selftest.py"
    ln -sf $(realpath "${PACKAGES_PATH}/build_v2/install/tests") "${BUILDDIR}/star/tests"

    for f in ${PACKAGES_PATH}/python-dists/*.tar.gz; do
        ln -sf $(realpath "$f") "${BUILDDIR}"/star/$(basename "$f" .tar.gz).tgz
    done

    for f in ${BUILDDIR}/skynet/*; do
        ln -sf $(realpath "$f") "${BUILDDIR}"/star/$(basename "$f")
    done

    if [ -e "$BUILDDIR/svninfo" ]; then
        ln -sf $(realpath "${BUILDDIR}/svninfo") "${BUILDDIR}/star/svninfo"
    fi

    bin=${BUILDDIR}/${BIN_FN}

    {
        echo "#!/usr/bin/env bash"
        cat "${PACKAGES_PATH}/build_v2/install/md5checker.sh"

        echo "RELEASE_URL=\"${URL}\""
        echo "RELEASE_REVISION=\"${REVISION}\""
        echo "RELEASE_VERSION=\"${VERSION}\""
        echo "RELEASE_SKYCORE_REV=\"${SKYCORE_REV}\""

        cat "${PACKAGES_PATH}/build_v2/_utils.sh" "${PACKAGES_PATH}/build_v2/install/skyinstall.sh"
        echo __XTR__
        tar -C "${BUILDDIR}/star" \
            -chf - \
            --exclude "__pycache__" \
            $(ls "${BUILDDIR}/star" | sort -r)
    } > "$bin"

    # This will update stored md5
    bash "${PACKAGES_PATH}/build_v2/install/md5checker.sh" "$bin"

    chmod +x "$bin"

    echo "Binary ready at $bin"
}


cleanup() {
    echo "Cleaning up..."
    cd "${BUILDDIR}"
    for f in *; do
        [ "$f" = "${BIN_FN}" ] || rm -rf "$f";
    done
    cd - >/dev/null 2>&1
}

main() {
    detectarch
    parseargs "$@"
    check_skynet_sources
    prepare_builddir
    detect_version
    unpack_python
    build_skynet_bundles
    build_selftar
    cleanup
}

main "$@"
