#!/skynet/python/bin/python -u

from __future__ import print_function, division, absolute_import

import argparse
import errno
import fcntl
import os
import pwd
import select
import shutil
import socket
import subprocess as sp
import sys

__import__('pkg_resources').require('gevent')

import gevent
import gevent.queue
import gevent.select
import gevent.socket

from gevent import hub

from kernel import util


class Popen(object):
    def __init__(self, *args, **kwargs):
        # delegate to an actual Popen object
        self.__p = sp.Popen(*args, **kwargs)
        # make the file handles nonblocking
        if self.stdin is not None:
            fcntl.fcntl(self.stdin, fcntl.F_SETFL, os.O_NONBLOCK)
        if self.stdout is not None:
            fcntl.fcntl(self.stdout, fcntl.F_SETFL, os.O_NONBLOCK)
        if self.stderr is not None:
            fcntl.fcntl(self.stderr, fcntl.F_SETFL, os.O_NONBLOCK)

    def __getattr__(self, name):
        # delegate attribute lookup to the real Popen object
        return getattr(self.__p, name)

    def _write_pipe(self, f, input):
        # writes the given input to f without blocking
        if input:
            bytes_total = len(input)
            bytes_written = 0
            while bytes_written < bytes_total:
                try:
                    # f.write() doesn't return anything, so use os.write.
                    bytes_written += os.write(f.fileno(), input[bytes_written:])
                except IOError, ex:
                    if ex[0] != errno.EAGAIN:
                        raise
                    sys.exc_clear()
                socket.wait_write(f.fileno())
        f.close()

    def _read_pipe(self, f):
        # reads output from f without blocking
        # returns output
        chunks = []
        while True:
            try:
                chunk = f.read(4096)
                if not chunk:
                    break
                chunks.append(chunk)
            except IOError, ex:
                if ex[0] != errno.EAGAIN:
                    raise
                sys.exc_clear()
            socket.wait_read(f.fileno())
        f.close()
        return ''.join(chunks)

    def communicate(self, input=None):
        # Optimization: If we are only using one pipe, or no pipe at
        # all, using select() is unnecessary.
        if [self.stdin, self.stdout, self.stderr].count(None) >= 2:
            stdout = None
            stderr = None
            if self.stdin:
                self._write_pipe(self.stdin, input)
            elif self.stdout:
                stdout = self._read_pipe(self.stdout)
            elif self.stderr:
                stderr = self._read_pipe(self.stderr)
            self.wait()
            return stdout, stderr
        else:
            return self._communicate(input)

    def _communicate(self, input):
        # identical to original... all the heavy lifting is done
        # in gevent.select.select
        read_set = []
        write_set = []
        stdout = None  # Return
        stderr = None  # Return

        if self.stdin:
            # Flush stdin buffer.
            self.stdin.flush()
            if input:
                write_set.append(self.stdin)
            else:
                self.stdin.close()
        if self.stdout:
            read_set.append(self.stdout)
            stdout = []
        if self.stderr:
            read_set.append(self.stderr)
            stderr = []

        input_offset = 0
        while read_set or write_set:
            try:
                readList, writeList, extraList = select.select(read_set, write_set, [])
            except select.error, e:
                if e.args[0] == errno.EINTR:
                    continue
                raise

            if self.stdin in writeList:
                # When select has indicated that the file is writable,
                # we can write up to PIPE_BUF bytes without risk
                # blocking.  POSIX defines PIPE_BUF >= 512
                bytes_written = os.write(self.stdin.fileno(), buffer(input, input_offset, 512))
                input_offset += bytes_written
                if input_offset >= len(input):
                    self.stdin.close()
                    write_set.remove(self.stdin)

            if self.stdout in readList:
                data = os.read(self.stdout.fileno(), 1024)
                if data == "":
                    self.stdout.close()
                    read_set.remove(self.stdout)
                stdout.append(data)

            if self.stderr in readList:
                data = os.read(self.stderr.fileno(), 1024)
                if data == "":
                    self.stderr.close()
                    read_set.remove(self.stderr)
                stderr.append(data)

        # All data exchanged.  Translate lists into strings.
        if stdout is not None:
            stdout = ''.join(stdout)
        if stderr is not None:
            stderr = ''.join(stderr)

        # Translate newlines, if requested.  We cannot let the file
        # object do the translation: It is based on stdio, which is
        # impossible to combine with select (unless forcing no
        # buffering).
        if self.universal_newlines and hasattr(file, 'newlines'):
            if stdout:
                stdout = self._translate_newlines(stdout)
            if stderr:
                stderr = self._translate_newlines(stderr)

        self.wait()
        return stdout, stderr

    def wait(self, check_interval=0.1):
        # non-blocking, use hub.sleep
        try:
            while True:
                status = self.poll()
                if status is not None:
                    return status
                hub.sleep(check_interval)
        except OSError, e:
            if e.errno == errno.ECHILD:
                # no child process, this happens if the child process
                # already died and has been cleaned up
                return -1
            else:
                raise


def system(cmd, env=None, cwd=None, raise_ex=True):
    print("Executing %r" % cmd)
    rc = sp.Popen(cmd, env=env, cwd=cwd, shell=True).wait()

    if rc:
        if raise_ex:
            raise Exception("Command %r failed with exit code %r." % (cmd, rc))
        print('WARNING: command %r finished with non-zero exit code (%r).' % (cmd, rc))
    return rc


def symlink(src, dst):
    print('Symlink(ing) %r -> %r' % (src, dst))
    os.symlink(src, dst)


def workdir_prepare(skydeps_tgz, root, raw):
    workdir = os.path.join(root, 'build/tests')
    srcdir = os.path.join(root, 'tests')

    vedir = os.path.join(workdir, 'venv')
    vebin = os.path.join(vedir, 'bin')
    vepy = os.path.join(vebin, 'python')
    vetests = os.path.join(vedir, 'tests')
    pip = os.path.join(vebin, 'pip')

    if raw and os.path.exists(workdir):
        return vedir, vebin, vepy, vetests

    print("Preparing working directory at %r" % workdir)
    try:
        shutil.rmtree(workdir)
    except OSError:
        pass
    os.makedirs(workdir)

    system('tar zxf %s -C %s' % (skydeps_tgz, workdir))
    rootbin = os.path.join(workdir, 'python', 'bin')
    rootpy = os.path.join(rootbin, 'python')
    rootve = os.path.join(rootbin, 'virtualenv')

    system('%s %s --no-site-packages %s' % (rootpy, rootve, vedir))
    system('%s %s --relocatable %r' % (rootpy, rootve, vedir))

    env = os.environ.copy()
    env.update({
        'CC': 'gcc',
        'CXX': 'g++',
        'PYTHONHOME': vedir,
        'CFLAGS': '-I{py}/include -L{py}/lib'.format(py=os.path.join(workdir, 'python')),
    })

    system(
        '%s %s install --use-wheel --index http://pypi.yandex-team.ru/simple -r %r %s' % (
            vepy, pip,
            os.path.join(srcdir, 'requirements.txt'),
            os.path.join(srcdir, 'teamcity'),
        ),
        env=env,
    )
    system('%s %s --relocatable %r' % (rootpy, rootve, vedir))

    symlink(os.path.join(rootbin, 'liner'), os.path.join(vebin, 'liner'))
    symlink(os.path.join(rootbin, 'liner.debug'), os.path.join(vebin, 'liner.debug'))
    symlink(srcdir, vetests)

    return vedir, vebin, vepy, vetests


def local(skydeps_tgz, teamcity=False, raw=False):
    root = os.path.dirname(os.path.realpath(os.path.abspath(sys.argv[0])))
    vedir, vebin, vepy, vetests = workdir_prepare(skydeps_tgz, root, raw)

    pytest = os.path.join(vebin, 'py.test')
    teamcity_arg = ''
    if teamcity:
        teamcity_arg = '--teamcity-ex --teamcity-flow-prefix "%s" --teamcity-global-suite "%s"' % (
            '%s-pytest' % (socket.gethostname(), ), socket.gethostname()
        )

    if os.uname()[0].lower() == 'darwin':
        env = os.environ.copy()
        env['DYLD_LIBRARY_PATH'] = os.path.join(root, 'build/tests/python/lib')
    else:
        env = None

    system(
        '%s %s -rf -v --tb=short --basetemp=/tmp/skydepstest.%s %s %s' % (
            vepy, pytest, pwd.getpwuid(os.getuid())[0], teamcity_arg, vetests
        ),
        cwd=vedir,
        env=env
    )


def remote(host, skydeps_tgz, teamcity=False):
    print('Initializing remote session with %r. Sending test environment files with %r.' % (host, skydeps_tgz))

    user = 'robot-skybot'

    own_name = os.path.basename(sys.argv[0])
    p = sp.Popen(['tar', '-cf-', skydeps_tgz, 'tests', own_name], stdout=sp.PIPE)

    total = 0

    tmpdir = '/var/tmp/sdeptst_reme'

    proc = sp.Popen([
        'ssh', '-l', user,
        host,
        'rm -rf %s; mkdir -p %s && '
        'tar -C %s -xf-' % (tmpdir, tmpdir, tmpdir)
    ], stdin=sp.PIPE)

    while True:
        chunk = p.stdout.read(0x10000)
        total += len(chunk)
        proc.stdin.write(chunk)
        if len(chunk) != 0x10000:
            break

    print('Total amount of data sent: %s' % util.size2str(total))

    proc.stdin.close()
    proc.wait()

    if proc.returncode != 0:
        raise Exception('proc #1: return code %r' % (proc.returncode, ))

    p.wait()

    print('Running remote test script on %s' % (host, ))

    proc = Popen([
        'ssh', '-l', user,
        host,
        'cd %s && find -name "*.pyc" -delete && '
        './test.py --teamcity local %s 2>&1' % (tmpdir, skydeps_tgz)
    ], stdout=sp.PIPE)

    partline = None

    while True:
        gevent.socket.wait_read(proc.stdout.fileno())
        stdout = proc.stdout.read(8192)

        if not stdout:
            break

        lines = stdout.split('\n')

        if len(lines) == 1:
            if partline:
                partline += lines[0]
            else:
                partline = lines[0]

            continue

        if partline is not None:
            lines[0] = partline + lines[0]
            partline = None

        if lines[-1]:
            partline = lines.pop(-1)
        else:
            lines.pop(-1)

        if not stdout:
            break

        for line in lines:
            print('%-12s: %s' % (host, line))

    proc.wait()
    print('Remote script finished with returncode %s' % (proc.returncode, ))

    return proc.returncode


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '-t', '--teamcity', action='store_true',
        help='Output teamcity messages.'
    )

    subparsers = parser.add_subparsers()
    lgrp = subparsers.add_parser('local', help='local tests execution')
    lgrp.add_argument(
        'skydeps_tgz',
        action='store',
        help='Skynet thirdparty binary dependencies compressed tarball.'
    )
    lgrp.add_argument(
        '--raw', '-r',
        action='store_true',
        help='Do not perform cleanup (and prepare) in case of virtual environment exists.'
    )

    rgrp = subparsers.add_parser('remote', help='remote tests execution')
    rgrp.add_argument(
        'hosts_files_list',
        action='store',
        nargs='+',
        help='Comma-separated list of hosts '
             '(with name of skynet dependencies tarball, separated with colon) '
             'to be processed remotely.'
    )

    args = parser.parse_args()
    if 'skydeps_tgz' in args:
        local(args.skydeps_tgz, teamcity=args.teamcity, raw=args.raw)

    if 'hosts_files_list' in args:
        rc = 0
        gths = [
            gevent.spawn(remote, *hf.split(':'), teamcity=args.teamcity)
            for hf in args.hosts_files_list
        ]
        gevent.joinall(gths)
        for gt in gths:
            if gt.exception:
                rc |= 1
            elif gt.value:
                rc |= gt.value
        return rc

    return 0

if __name__ == '__main__':
    sys.exit(main())
