#!/skynet/python/bin/python

import argparse
import os
import sys
import shutil
import hashlib
import tarfile
import itertools
import subprocess
import imp
import py_compile

import py


# Size to string formatter (from `kernel.util.misc`)
def size2str(size):
    size = float(size)
    mods = ['byte(s)'] + ['%siB' % c for c in 'KMGTPEZY']
    for mod in mods:
        if size < 0x400 or mod == mods[-1]:
            return '%.2f%s' % (size, mod)
        size /= 0x400


def _checkout(args, dir, item, force_checkout=False):
    should_checkout = not args.no_service_update or force_checkout
    uri = open(os.path.join(dir, item), 'rb').read().strip()
    item = item.rsplit('.', 1)[0]
    printChunk('{0: >{1}}'.format(item, 37), suffix='Checkout ')
    if should_checkout:
        item_full_path = os.path.join(dir, item)
        if os.path.exists(item_full_path):
            if os.path.islink(item_full_path):
                os.unlink(item_full_path)
            else:
                shutil.rmtree(item_full_path)
        proc = subprocess.Popen(['svn', 'co', '-q', uri, item_full_path])
        proc.wait()
        if proc.returncode != 0:
            raise SystemExit(proc.returncode)
    printChunk('skipped' if not should_checkout else 'done', nl=True)


def getServices(args, vcs):
    servicesPath = os.path.join(args.skynetroot, 'services')
    if not os.path.exists(servicesPath):
        printChunk('There are no services in this bundle. Skip it.', nl=True)
        return

    for item in os.listdir(servicesPath):
        if item.startswith('.'):
            # Skip .svn, .hg and so on
            continue

        if item.endswith('.link'):
            _checkout(args, servicesPath, item)
            item = item.rsplit('.', 1)[0]
        else:
            if os.path.exists(os.path.join(servicesPath, item + '.link')):
                continue

        fullPath = os.path.join(servicesPath, item)
        if os.path.isdir(fullPath) and not os.path.islink(fullPath):
            modCtlFile = os.path.join(fullPath, 'ctl.py')
            if os.path.exists(modCtlFile):
                mod = imp.load_module('%s_ctl' % item, open(modCtlFile, 'U'), modCtlFile, ('.py', 'U', imp.PY_SOURCE))
                yield mod.Ctl().name(), fullPath


def pywalk(initialPath, skipDirs=set(), skipExts=('pyc', 'pyo')):
    for path, dirs, files in os.walk(initialPath):
        # If current directory starts with '.' -- ignore everything in it
        if os.path.basename(path).startswith('.'):
            skipDirs.add(path)
            continue

        # Skip if our dir is below some dir in skip list
        if any((path.startswith(d) for d in skipDirs)):
            continue

        for filename in sorted(files):
            filepath = os.path.join(path, filename)

            # Skip files with extensions from skipExts list
            if os.path.splitext(filepath)[1][1:] in skipExts:
                continue

            # Skip .* and *~ files
            if (filename.startswith('.') and not filename.startswith('.version')) or filepath.endswith('~'):
                continue

            yield filepath, filepath[len(initialPath):]


def printChunk(prefix=None, suffix=None, align=0, nl=False):
    if prefix:
        sys.stdout.write(prefix)
    if suffix:
        sys.stdout.write(' {0: <{align}}'.format(suffix, align=align))
    if not nl:
        sys.stdout.flush()
    else:
        sys.stdout.write('\n')


def srcdirChecksum(dirs, relto=None, vcs=None):
    # Obtain versioned files list firstly
    files = []
    env = os.environ.copy()
    env.pop('LD_LIBRARY_PATH', None)
    relto = py.path.local(relto) if isinstance(relto, basestring) else relto
    relpath = py.path.local().bestrelpath(relto) if relto else '.'

    for d in dirs if isinstance(dirs, (list, set)) else [dirs]:
        if relpath:
            d = os.path.join(relpath, d) if d != '.' else relpath
        if vcs == 'svn' or os.path.isdir(os.path.join(d, '.svn')):
            cmd = 'svn', 'ls', '--recursive', d
        elif vcs == 'hg':
            cmd = 'hg', 'status', '-ncma', d
        elif vcs == 'git':
            cmd = 'git', 'ls-files', d
        elif vcs == 'arc':
            cmd = 'arc', 'ls-files', d
        else:
            assert 0, 'Unknown vcs: %r' % (vcs, )

        stdout = subprocess.check_output(cmd, env=env)
        files += [os.path.join(d, x) if vcs != 'arc' else x for x in stdout.strip().split('\n')]
    files = filter(lambda x: os.path.isfile(x), files)

    # Compile checksums of them
    assert files
    return hashlib.sha1(subprocess.check_output(['sha1sum'] + files, env=env))


def chksumWDeps(root, deps, chksum):
    if deps:
        root = py.path.local(root) if isinstance(root, basestring) else root
        chksum.update(''.join(root.join(dep, '.version').read().strip() for dep in deps))
    return chksum


def buildbase(args, vcs):
    BINDIRS = ['startup', 'tools']
    LIBDIRS = ['api', 'ext', 'kernel', 'library']

    rootpath = py.path.local(args.skynetroot)
    if args.variant == 'devel':
        os.symlink(rootpath.strpath, os.path.join(args.builddir, 'skynet'))
        printChunk('{0: <{1}}'.format('(BASE)', 31), suffix='Done. Symlink created.', nl=True)
    if args.variant != 'bundle':
        return

    tmpdir = os.path.join(args.builddir, 'temp')
    if os.path.exists(tmpdir):
        shutil.rmtree(tmpdir)
    os.makedirs(tmpdir)

    # Fist of all, compile checksum of library and tools directories
    checksums = []

    print_offset = 31

    # First, checkout .link files if needed
    for i, dir in enumerate(LIBDIRS):
        dirpath = rootpath.join(dir)
        for item in dirpath.listdir():
            if item.basename.endswith('.link'):
                uri = item.read(mode='rb').strip()

                checkout_as = item.basename.rsplit('.', 1)[0]
                nitem = item.dirpath().join(checkout_as)

                printChunk('{0: >{1}}'.format(
                    '(%s)' % (os.path.join(dir, checkout_as), ),
                    print_offset), suffix='Checkout '
                )
                print_offset = 37

                if nitem.check(exists=1):
                    nitem.remove()

                proc = subprocess.Popen(['svn', 'export', '-q', uri, nitem.strpath])
                proc.wait()

                if proc.returncode != 0:
                    raise SystemExit(proc.returncode)

                printChunk('done', nl=True)

    # Checksum everything
    for dir in LIBDIRS + BINDIRS:
        dirpath = rootpath.join(dir)
        deps = dirpath.join('.dependencies').read().strip().split()
        chksum = chksumWDeps(rootpath, deps, srcdirChecksum(dir, relto=rootpath, vcs=vcs))
        checksums.append(chksum)
        chksum = chksum.hexdigest()
        dirpath.join('.version').write(chksum, 'wb')
        printChunk('{0: >{1}}'.format('(%s)' % dir, print_offset), suffix='Checksum %s' % chksum, nl=True)
        print_offset = 37

    # Create cummulative checksum for 'BASE'.
    checksums[0].update(''.join(chksum.hexdigest() for chksum in checksums[1:]))
    rootpath.join('.version').write(checksums[0].hexdigest(), 'wb')
    printChunk('{0: >{1}}'.format('[CUMMULATIVE] (BASE)', 37), suffix='Checksum %s' % checksums[0].hexdigest())

    sizes = [0, 0]
    printChunk(' Tarball')
    tmptar = os.path.join(tmpdir, 'skynet.tgz')
    tar = tarfile.open(None, 'w:gz', open(tmptar, 'wb'))
    files = pywalk(rootpath.strpath, skipDirs=set(
        rootpath.join(x).strpath
        for x in 'python packages services startup/supervisor startup/build'.split()
    ))
    chkFiles = (files.next(), )
    # As mocksoul@ says, we SHOULD check the first tarball entries
    # should be dot-version* files.
    assert ['/.version'] == map(lambda x: x[1], chkFiles), repr(chkFiles)

    for fullpath, relpath in itertools.chain(chkFiles, files):

        # Dont add .link files from LIBDIR dirs
        if fullpath.endswith('.link'):
            skip = False
            for dir in LIBDIRS:
                if fullpath.startswith(rootpath.join(dir).strpath):
                    skip = True
                    break
            if skip:
                continue

        tar.add(fullpath, arcname=relpath)
        sizes[0] += os.stat(fullpath).st_size
        if fullpath.endswith('.py'):
            try:
                py_compile.compile(fullpath)
                tar.add(fullpath[:-3] + '.pyc', relpath[:-3] + '.pyc')
                sizes[0] += os.stat(fullpath).st_size
            except:
                pass

    tar.close()

    targettar = os.path.join(args.builddir, 'skynet.tgz')
    if os.path.exists(targettar):
        os.unlink(targettar)
    os.rename(tmptar, targettar)
    sizes[1] = os.stat(targettar).st_size
    printChunk(suffix='/'.join('{0: >9}'.format(size2str(s)) for s in sizes), nl=True)
    return True


def build_skycore(args, vcs):
    rootpath = py.path.local(args.skynetroot)
    skpath = rootpath.join('skycore')

    if skpath.check(exists=1) and args.variant != 'devel':
        shutil.rmtree(skpath.strpath)

    if rootpath.join('skycore.link').check(exists=1, file=1) and not skpath.check(exists=1):
        _checkout(args, rootpath.strpath, 'skycore.link', True)

    if skpath.check(exists=1):
        build2(args, 'skycore', skpath.strpath, skpath.join('build.sh').strpath, vcs)
        if args.variant == 'bundle':
            os.rename(os.path.join(args.builddir, 'services', 'skycore.tgz'), os.path.join(args.builddir, 'skycore.tgz'))
        elif args.variant == 'devel':
            os.rename(os.path.join(args.builddir, 'services', 'skycore'), os.path.join(args.builddir, 'skycore'))


def build2(args, name, path, buildscript, vcs):
    if args.variant == 'bundle':
        tmpdir = os.path.join(args.builddir, 'temp')
        if os.path.exists(tmpdir):
            shutil.rmtree(tmpdir)
        os.makedirs(tmpdir)

    cwd = os.path.abspath(os.curdir)
    os.chdir(path)

    buildscript = '.' + buildscript[len(path):]
    if buildscript.endswith('.py'):
        buildargs = [sys.executable, buildscript]
    else:
        buildargs = [buildscript]

    def _call(cmd):
        args = buildargs + [cmd]
        env = {
            'PYTHON': sys.executable,
        }
        for evar in ('PATH', 'SSH_AUTH_SOCK', 'SVN_SSH'):
            value = os.getenv(evar)
            if value:
                env[evar] = value

        proc = subprocess.Popen(
            args,
            stdout=subprocess.PIPE, stderr=subprocess.STDOUT,
            env=env,
        )
        stdout = proc.communicate()[0]
        if proc.returncode:
            print '%s: failed in call %r: %s' % (name, args, stdout)
        return stdout if not proc.returncode else None

    if _call('clean') is None or _call(args.variant) is None:
        return
    deps = _call('dependencies')
    if deps is None:
        return
    deps = deps.strip().split()

    snapshotDir = os.path.join(path, 'snapshot')
    rootpath = py.path.local(args.skynetroot)
    if args.variant == 'bundle':
        # Create checksum file first
        printChunk(' Checksum')
        chksum = chksumWDeps(
            rootpath, deps, srcdirChecksum(
                '.',
                vcs=vcs
            )
        )
        chksum = hashlib.sha1('%s:%s' % (name, chksum.hexdigest()))
        py.path.local(snapshotDir).join('.version').write(chksum.hexdigest(), 'wb')
        printChunk(suffix=chksum.hexdigest())

        sizes = [0, 0]
        printChunk(' Tarball')
        tmptar = os.path.join(tmpdir, name + '.tgz')
        tar = tarfile.open(None, 'w:gz', open(tmptar, 'wb'))
        for fullpath, relpath in pywalk(snapshotDir, skipExts=()):
            sizes[0] += os.stat(fullpath).st_size
            tar.add(fullpath, arcname=relpath)
        tar.close()

        targettar = os.path.join(args.builddir, 'services', name + '.tgz')
        if os.path.exists(targettar):
            os.unlink(targettar)
        os.rename(tmptar, targettar)
        sizes[1] = os.stat(targettar).st_size
        printChunk(suffix='/'.join('{0: >9}'.format(size2str(s)) for s in sizes), nl=True)
    elif args.variant == 'devel':
        os.symlink(snapshotDir, os.path.join(args.builddir, 'services', name))
        printChunk(suffix='Done. Symlink created.', nl=True)

    if args.variant == 'bundle':
        shutil.rmtree(tmpdir)

    os.chdir(cwd)
    return True


def build(args, vcs):
    print 'Using python:   ', sys.executable
    print 'Build type:     ', args.variant
    print 'Skynet root:    ', args.skynetroot
    print 'Build directory:', os.path.abspath(args.builddir)
    print 'VCS:            ', vcs
    print '-' * 80

    if os.path.exists(args.builddir):
        shutil.rmtree(args.builddir)
    os.makedirs(args.builddir)
    os.makedirs(os.path.join(args.builddir, 'services'))

    if 'base' in args.components:
        printChunk('Build ')
        buildbase(args, vcs)
        build_skycore(args, vcs)

    if 'services' in args.components:
        failed = []
        for serviceName, path in sorted(getServices(args, vcs=vcs)):
            if args.services is not None and serviceName not in args.services:
                continue
            if args.excludeServices is not None and serviceName in args.excludeServices:
                continue

            buildsh = os.path.join(path, 'build.sh')
            buildpy = os.path.join(path, 'build.py')

            if os.path.exists(buildsh):
                printChunk('Build v2.sh', 'service %s' % serviceName, 25)
                rc = build2(args, serviceName, path, buildsh, vcs)
            elif os.path.exists(buildpy):
                printChunk('Build v2.py', 'service %s' % serviceName, 25)
                rc = build2(args, serviceName, path, buildpy, vcs)
            else:
                raise Exception('Build v1 is not supported anymore')
            if not rc:
                failed.append(serviceName)
        if failed:
            print('Following component(s) build failed: %r' % failed)
            sys.exit(1)


def main():
    skynetroot = os.path.abspath(__file__)
    for i in range(2):
        skynetroot = os.path.split(skynetroot)[0]

    parser = argparse.ArgumentParser(description='Prebuild skynet.')
    parser.add_argument(
        '--variant',
        dest='variant', action='store',
        choices=('devel', 'bundle'),
        default='bundle',
        help='Build type'
    )
    parser.add_argument(
        '-b', '--builddir',
        dest='builddir', action='store',
        default='build',
        help='Build directory relative to current cwd'
    )
    parser.add_argument(
        '-r', '--skynet-root',
        dest='skynetroot', action='store',
        default=skynetroot,
        help='Root of skynet sources (detected at: %(default)s)'
    )
    parser.add_argument(
        '-s', '--service',
        dest='services', action='append',
        help='Add services to build (by default -- all)'
    )
    parser.add_argument(
        '-e', '--exclude-service',
        dest='excludeServices', action='append',
        help='Exclude services from build (by default - no)'
    )
    parser.add_argument(
        '-c', '--components',
        dest='components', action='store',
        type=lambda x: tuple([_.strip() for _ in x.split(',')]),
        default=('base', 'services'),
        help='Components to build (%(default)s)'
    )
    parser.add_argument(
        '--no-service-update',
        dest='no_service_update', action='store_true', default=False,
        help='Do not check out services from service.link. Keep current version'
    )

    args = parser.parse_args()
    args.builddir = os.path.abspath(args.builddir)

    sys.path.append(args.skynetroot)

    vcs = None
    for part in py.path.local(skynetroot).parts(reverse=True):
        if part.join('.svn').check(dir=1):
            vcs = 'svn'
            break
        elif part.join('.hg').check(dir=1):
            vcs = 'hg'
            break
        elif part.join('.arcadia.root').check(exists=1):
            vcs = 'arc'
            break
        elif part.join('.git').check(dir=1):
            vcs = 'git'
            break

    if vcs is None:
        sys.stderr.write('Unable to detect VCS system!\n')
        raise SystemExit(1)

    build(args, vcs=vcs)


if __name__ == '__main__':
    main()
