"""
Tests for api.semaphorer.
"""
import unittest
import threading
import Queue

from api.semaphorer import Semaphore, TimingSemaphore, SemaphoreBusyError
from api.semaphorer import ftok


class ApiSemaphorerTest(unittest.TestCase):
    def test_bad_args(self):
        self.assertRaises(ValueError, Semaphore, 0, 0)
        self.assertRaises(ValueError, Semaphore, 1, 0)
        self.assertRaises(ValueError, Semaphore, 0, 1)
        self.assertRaises(ValueError, Semaphore, 1, -1)

    def test_acquire_simple(self):
        s = Semaphore('name', 1)
        self.assertTrue(s.acquire())
        s.release()
        # there are different code paths
        # if timeout exists
        self.assertTrue(s.acquire(10))
        s.release()

    def test_acquire_failed(self):
        queue = Queue.Queue()

        def acquire(sem, queue):
            queue.put(sem.acquire(0.1))

        s = Semaphore('name', 1)
        s1 = Semaphore('name', 1)

        self.assertEqual(s.value, 1)

        self.assertTrue(s.acquire())

        self.assertEqual(s.current(), 0)

        thr = threading.Thread(target=acquire, args=(s1, queue))
        thr.start()
        thr.join()

        failed_res = queue.get(timeout=1)
        self.assertFalse(failed_res)

        def acquire(sem, queue):
            queue.put(sem.acquire(10))

        thr = threading.Thread(target=acquire, args=(s1, queue))
        thr.start()

        s.release()

        thr.join()
        res = queue.get(timeout=1)

        self.assertTrue(res)
        s1.release()

    def test_timing_acquire(self):
        s = TimingSemaphore('name', 1, 1)
        self.assertTrue(s.acquire())
        s.release()
        # there are different code paths
        # if timeout exists
        self.assertTrue(s.acquire())
        self.assertEqual(s.current(), 0)
        s.release()
        self.assertEqual(s.current(), 1)

    def test_context_manager(self):
        with TimingSemaphore('name', 1, 0.1) as s:
            self.assertEqual(s.current(), 0)
            s1 = TimingSemaphore('name', 1, 0.1)
            self.assertRaises(SemaphoreBusyError, s1.acquire)
        self.assertEqual(s1.current(), 1)

    def test_current_check(self):
        # check that current works
        # without acquiring a semaphore
        s = Semaphore('aaaa', 1)
        self.assertEqual(s.current(), 1)
        s = Semaphore('aaaa', 100)
        self.assertEqual(s.current(), 100)


class FtokTest(unittest.TestCase):
    def test_bad_args(self):
        self.assertRaises(OSError, ftok, '__not_exist__', 100)

    def test_ok_args(self):
        # should return int > 0
        self.assertTrue(ftok('.', 1))
