from __future__ import print_function
import sys
import optparse

from api.skycore import ServiceManager, NotRunningError, NamespaceError


def available_services():
    try:
        return set(ServiceManager().list_services('skynet'))
    except (NamespaceError, ImportError, NotRunningError, EnvironmentError, RuntimeError):
        return set()


class Cmd(object):
    mode = None
    description = None
    ptype = None
    usage = None
    example = None
    related_services = ''

    class __metaclass__(type):
        available_services = None

        def __new__(mcs, *args):
            cls = type.__new__(mcs, *args)
            cls._commands = {}

            if mcs.available_services is None:
                mcs.available_services = available_services()

            for service in cls.related_services.split():
                any_of_services = service.split('|')
                found = False
                for s in any_of_services:
                    if s in mcs.available_services:
                        found = True
                        break

                if not found:
                    return cls

            if cls.mode is not None:
                if cls.ptype is None:
                    cls.ptype = Cmd
                cls.ptype._commands.setdefault(cls.mode, cls)
            return cls

    @classmethod
    def get_command(cls, name):
        return cls._commands.get(name, None)

    def __init__(self, options=None, help='h', argvreqd=True):
        super(Cmd, self).__init__()

        self.parser = optparse.OptionParser(add_help_option=False)
        self.parser.disable_interspersed_args()

        if help is not None:
            self.parser.add_option(
                '-' + help, '--help', action='store_true', default=False,
                help='print help'
            )

        if options:
            for option in options:
                option.add_opt(self.parser)

        self.papa = None
        self.pref = None
        self.code = 1
        self.options = None
        self.argv = None
        self.argvreqd = argvreqd

        usage_pref = '%prog'
        usage = [usage_pref]
        if self.parser.option_list:
            usage.append('[options]')
        if self.usage:
            usage.append(self.usage.strip())
        elif self.__doc__:
            usage.append(self.__doc__.strip())
        else:
            usage.append('...')
        self.parser.usage = ' '.join(usage)

        if self.example:
            self.epilog = ['Example:', usage_pref, self.example]
        else:
            self.epilog = None

    def parse(self, argv):
        if not self.pref:
            self.pref = [sys.argv[0]]
        self.parser.prog = ' '.join(self.pref)
        if self.epilog:
            self.parser.epilog \
                = self.parser.expand_prog_name(' '.join(self.epilog))
        self.options, self.argv = self.parser.parse_args(argv)
        if getattr(self.options, 'help', False) or self.argvreqd and not argv:
            self.help()
            sys.exit(self.code)

    def pre_run(self):
        return 0

    def run(self):
        return 0

    def help(self):
        self.parser.print_help(sys.stderr)

        if 1 < len(self._commands):  # just 'help' doesn't count
            print('\nPossible commands are:', file=sys.stderr)
            for command in sorted(self._commands):
                cls = self.get_command(command)
                sys.stderr.write(
                    "  %s%s\t%s\n" % (
                        command,
                        " " * (16 - len(command)),
                        cls.description if cls.description else ""
                    )
                )

        print(__import__('textwrap').dedent('''
            You can get more help here:
             - General Info : https://doc.yandex-team.ru/Search/skynet-dg/concepts/about.xml
             - cqudp Docs   : https://doc.yandex-team.ru/Search/skynet-api/cqueue/index.html
             - Blinov Calc  : https://doc.yandex-team.ru/Search/skynet-api/sky/blinovcalc.html
             - Common FAQ   : https://wiki.yandex-team.ru/Skynet/FAQ2
        '''), file=sys.stderr)

    def Run(self, *args, **kwargs):
        try:
            return self._Run(*args, **kwargs)
        except (SystemExit, GeneratorExit, KeyboardInterrupt):
            raise
        except BaseException:
            ei = sys.exc_info()
            try:
                from kernel.util.errors import formatException
            except:
                raise ei[1], None, ei[2]
            print('Got unhandled error while executing command:', file=sys.stderr)
            print(formatException(), file=sys.stderr)
            print((
                'Got stuck? Read about known possible errors '
                'here: https://wiki.yandex-team.ru/Skynet/FAQ2/'
            ), file=sys.stderr)
            return 1

    def _Run(self, argv=None, pref=None, pop=True, papa=None):
        self.papa = papa
        if not pref:
            pref = []
        self.pref = pref
        if argv is None:
            argv = sys.argv
        if pop:
            self.pref.append(argv[0])
            argv = argv[1:]
        self.parse(argv)

        while True:  # one-pass loop
            self.code = self.pre_run()
            if self.code:
                break

            if not self.argv:
                break

            if not self._commands:
                break

            cmd = self.argv[0]
            cls = self.get_command(cmd)
            if cls is None:
                if 1 == len(self._commands):
                    break
                print("Unsupported mode:", cmd)
                self.help()
                sys.exit(1)

            self.code = cls()._Run(self.argv, self.pref, papa=self)
            break

        if not self.code:
            self.code = self.run()

        return self.code
