import logging
from typing import List

from smarttv.alice.mm_backend.common.mm import Form
from smarttv.alice.mm_backend.common.constants import is_ya_tv

logger = logging.getLogger(__name__)


class SlotsParser:
    def __init__(self, data):
        self._slots = {}
        for item in data:
            self._slots[item['name']] = item['value']

    def get_value(self, name):
        return self._slots.get(name)


class NoDataException(Exception):
    pass


class InvalidClient(Exception):
    pass


class IAliceTvRequest:
    def check(self, scenario_name, slots: List[Form.Slot]):
        raise NotImplementedError


class AliceTvRequest(IAliceTvRequest):
    def __init__(self, request_object):
        self.ro = request_object
        self.slots = {}
        self.tv_set = None

    def check(self, scenario_name, slots: List[Form.Slot], section: str):
        app_id = self.ro['base_request']['client_info']['app_id']
        if not is_ya_tv(app_id):
            raise InvalidClient('Incompatible client with id: :s', app_id)

        try:
            for item in self.ro['base_request']['device_state']['tv_set'][section]:
                item['list']
        except KeyError as ex:
            logger.warning(
                'some keys [(input, base_request.device_state.tv_set.%s.list) are not presented in request' % section)
            raise NoDataException('input request absent field: %s' % ex)

        try:
            self.ro['input']['semantic_frames']
        except KeyError:
            msg = 'input.semantic_frames not found'
            logger.warning(msg)
            raise NoDataException(msg)

        if 'semantic_frames' not in self.ro['input']:
            raise NoDataException('input.semantic_frames not found')

        self.tv_set = self.ro['base_request']['device_state']['tv_set']

        frames = list(filter(lambda x: x['name'] == scenario_name, self.ro['input']['semantic_frames']))
        if len(frames) == 0:
            raise NoDataException('no semantic_frames for %s', scenario_name)

        uninitialized_slots = []
        for slot in slots:
            value = SlotsParser(frames[0]['slots']).get_value(slot.name)
            if value is None and slot.cant_be_none:
                uninitialized_slots.append(slot.name)
                continue

            self.slots[slot.name] = value

        if uninitialized_slots:
            raise NoDataException('slots have to be filled: %s', ', '.join(uninitialized_slots))
