from abc import ABCMeta, abstractmethod

from smarttv.alice.tv_proxy.proxy.saas_defaults import default_prefix


class DocumentItemTypes:
    literal_attr = 'l'
    grouping_attr = 'g'
    int_attr = 'i'
    property = 'p'
    zone = 'z'


class BaseMessage(object):
    __metaclass__ = ABCMeta

    @abstractmethod
    def json(self):
        pass

    @property
    @abstractmethod
    def action(self):
        pass


class ModifyMessage(BaseMessage):
    def __init__(self, url, prefix):
        self.document = self._empty_document(url)
        self.prefix = prefix

    @property
    def action(self):
        return 'modify'

    # noinspection PyMethodMayBeStatic
    def _empty_document(self, url):
        return {
            'options': {
                'mime_type': 'text/plain',
                'charset': 'utf8',
                'language': 'ru'
            },
            'url': url
        }

    # noinspection PyMethodMayBeStatic
    def _get_child_item(self, value, types):
        assert len(types) > 0, 'at least one type must be specified'
        # noinspection PyPep8
        assert DocumentItemTypes.zone not in types or len(
            types) == 1, 'incorrect syntax: using #z with other item types (#lgp)'

        return {
            'value': value,
            'type': '#' + ''.join(types)
        }

    def add_zone(self, name, content):
        assert name not in self.document, 'zone "%s" already presented in the document' % name
        self.document[name] = self._get_child_item(content, [DocumentItemTypes.zone])

    def add_attr(self, name, value, types):
        if name not in self.document:
            self.document[name] = []

        self.document[name].append(self._get_child_item(value, types))

    def json(self):
        return {
            'prefix': self.prefix,
            'action': self.action,
            'docs': [
                self.document
            ]
        }


class DeleteMessage(BaseMessage):
    def __init__(self, url):
        self.url = url

    @property
    def action(self):
        return 'delete'

    def json(self):
        return {'prefix': default_prefix, 'action': self.action, 'docs': [{'url': self.url}]}


class DeleteAllMessage(BaseMessage):
    @property
    def action(self):
        return 'delete'

    def json(self):
        return {
            'prefix': default_prefix,
            'action': self.action,
            'request': '$remove_all$'
        }
