# -*- coding: utf-8 -*-

import concurrent.futures
import logging
from abc import ABCMeta, abstractmethod

import requests
from furl import furl

from smarttv.alice.tv_proxy.proxy.index_messages import DocumentItemTypes, DeleteMessage, DeleteAllMessage, \
    ModifyMessage
from smarttv.alice.tv_proxy.proxy.pq_delivery import message_writer, SaasWriterError, FormatError
from smarttv.alice.tv_proxy.proxy.saas_defaults import default_prefix
from smarttv.alice.tv_proxy.proxy.search_proxy import saas_search_client

logger = logging.getLogger(__name__)

non_significant_arg = 'device_id'
non_significant_arg2 = 'ts'

CUMMULATIVE_SENDING_TIMEOUT = 10


def patch_url(url, arg_name, arg_value):
    url = furl(url)
    if arg_name not in url.args:
        url.args[arg_name] = arg_value

    return url.url


class IndexingError(Exception):
    pass


class ChannelData(object):
    __metaclass__ = ABCMeta

    extra = None

    def __init__(self, device_id):
        self.device_id = device_id

    @property
    @abstractmethod
    def title(self):
        pass

    @property
    @abstractmethod
    def uri(self):
        pass

    @property
    @abstractmethod
    def number(self):
        pass

    @property
    @abstractmethod
    def region_ids(self):
        pass


class CabelChannelData(ChannelData):
    region_ids = ()

    def __init__(self, data, device_id, timestamp):
        self.data = data
        self.timestamp = timestamp
        super(CabelChannelData, self).__init__(device_id)

    @property
    def uri(self):
        uri = self.data['uri']
        uri = patch_url(uri, non_significant_arg, self.device_id)
        uri = patch_url(uri, non_significant_arg2, self.timestamp)

        return uri

    @property
    def title(self):
        return self.data['name']

    @property
    def number(self):
        return self.data['number']


class Cleaner(object):
    def __init__(self, device_id):
        self.device_id = device_id

    def validate_device_id(self):
        if self.device_id and not self.device_id.isalnum():
            raise ValueError('Device id validation error: {}'.format(self.device_id))

    def rm_docs_earlier_than(self, timestamp, exclude=None):
        self.validate_device_id()

        query = 's_device_id:%s' % self.device_id
        try:
            docs = saas_search_client.search(query)
            logger.info('saas return %d documents', len(docs))
        except requests.exceptions.BaseHTTPError:
            logger.exception('saas search proxy request failed')
            docs = set()

        if not docs:
            return

        remove_me = []
        exclude = exclude or set()

        for doc in docs:
            if not doc.ts or not doc.device_id:
                logger.debug('skipping entry without ts or device id: %s', doc)
                continue
            if doc.device_id != self.device_id:
                logger.debug('device id mismatch (strange), skipping doc %s', doc)
                continue
            if 0 < doc.ts < timestamp:
                if doc.url not in exclude:
                    remove_me.append(doc.url)

        self.delete_urls(remove_me)

    @staticmethod
    def rm_docs_by(query):
        logger.info('removing docs by query: %s', query)
        try:
            docs = saas_search_client.search(query)
            logger.info('saas return %d documents', len(docs))
        except requests.exceptions.BaseHTTPError:
            logger.exception('saas search proxy request failed')
            docs = set()

        if not docs:
            return

        urls = [d.url for d in docs]

        Cleaner.delete_urls(urls)

    @staticmethod
    def delete_urls(urls, raise_on_fail=False):
        logger.info('sending %d delete messages', len(urls))
        index([DeleteMessage(url) for url in urls], raise_on_fail)

    @staticmethod
    def clear_collection():
        index([DeleteAllMessage()])


def make_indexer_message(channel_data, timestamp=None):
    msg = ModifyMessage(channel_data.uri, prefix=default_prefix)
    msg.add_zone('title', channel_data.title)
    msg.add_attr('s_device_id', channel_data.device_id, [DocumentItemTypes.literal_attr, DocumentItemTypes.property])

    for region_id in channel_data.region_ids:
        msg.add_attr('i_region_id', region_id, [DocumentItemTypes.int_attr, DocumentItemTypes.property])

    if channel_data.number is not None:
        msg.add_zone('z_number', u'%s канал программа кнопка' % channel_data.number)
        msg.add_attr('number', str(channel_data.number), [DocumentItemTypes.property])

    if timestamp is not None:
        msg.add_attr('i_ts', timestamp, [DocumentItemTypes.int_attr, DocumentItemTypes.property])

    extra = channel_data.extra
    if extra is not None:
        msg.add_zone('z_extra', extra)

    return msg


def index(messages, raise_on_fail=True):
    if not messages:
        return

    def send(msg):
        message_writer.send_message(msg.json())

    features = []
    with concurrent.futures.ThreadPoolExecutor(max_workers=min(20, len(messages))) as executor:
        for msg in messages:
            future = executor.submit(send, msg)
            features.append(future)

        done, not_done = concurrent.futures.wait(features, timeout=CUMMULATIVE_SENDING_TIMEOUT)
        for feature in done:
            try:
                feature.result()
            except (SaasWriterError, FormatError):
                raise IndexingError()

        if len(not_done) > 0:
            logger.warning('%i indexing messages failed', len(not_done))
            if raise_on_fail:
                raise IndexingError()
