# -*- coding: utf-8 -*-

import logging
from time import time, sleep

import requests
from furl import furl

from django.core.management.base import BaseCommand
from smarttv.alice.tv_proxy.proxy import thesaurus
from smarttv.alice.tv_proxy.proxy.indexer import make_indexer_message, index, ChannelData, Cleaner, non_significant_arg

logger = logging.getLogger(__name__)

fake_device_id = 'f' * 8


def efir_uri(channel_id):
    url = furl('live-tv://android.media.tv/channel/vh')
    url.path.add(str(channel_id))
    url.args[non_significant_arg] = fake_device_id
    return url.url


class DroidekaChannelData(ChannelData):
    def __init__(self, data):
        self.data = data
        super(DroidekaChannelData, self).__init__(fake_device_id)

    @property
    def title(self):
        return self.data['title']

    @property
    def uri(self):
        return efir_uri(self.data['channel_id'])

    @property
    def region_ids(self):
        return []  # todo: add region_ids to serializer in droideka

    @property
    def extra(self):
        synonims = thesaurus.synonims.get(self.data['channel_id'])
        if synonims is None:
            return None

        return ' '.join(synonims)

    @property
    def number(self):
        try:
            return self.data['channel_smarttv_number']
        except KeyError:
            logger.info('ya.efir channel with title=%s has no channel_smarttv_number field', self.title)
            return None


class MyEfirData(ChannelData):
    # noinspection SpellCheckingInspection
    uri = 'ya-vh-content://4461546c4debdcffbab506fd75246e19?my_efir_mode=true&restriction_age=18'
    title = 'Мой эфир'
    region_ids = ()
    number = None


class YaEfirChannels(object):
    tv_agent = 'com.yandex.tv.input.efir/1.4.0 (DEXP U50E9100Q; Android 7.1.1)'

    def __init__(self, url):
        self.channels_url = url

    # noinspection PyMethodMayBeStatic
    def _only_with_number(self, channels):
        result = []
        for item in channels:
            if 'channel_smarttv_number' in item:
                result.append(item)
            else:
                logger.warning('channel with channel_id=%s has no attribute channel_smarttv_number', item['channel_id'])

        return result

    def get_channels(self):
        response = requests.get(self.channels_url, headers={'User-Agent': self.tv_agent})
        response.raise_for_status()
        return self._only_with_number(response.json()['channels'])


def index_my_efir(timestamp):
    msg = make_indexer_message(MyEfirData(fake_device_id), timestamp)
    index([msg])


def index_efir_channels(data_url, timestamp):
    channels = YaEfirChannels(data_url).get_channels()
    messages = [make_indexer_message(DroidekaChannelData(item), timestamp) for item in channels]
    index(messages)
    logger.info('Index %d yandex efir channels', len(messages))


class Command(BaseCommand):
    help = 'Index ya efir channels'

    def add_arguments(self, parser):
        parser.add_argument('--device-id', default=fake_device_id)
        parser.add_argument('--delete', action='store_true', default=False,
                            help='delete old docs after indexing for current device_id')
        parser.add_argument('--my-efir', action='store_true', default=False, help='add to index "My efir" channel')
        parser.add_argument('--ya-efir', action='store_true', default=False, help='add to index yandex efir channels')
        parser.add_argument('--purge-request', default=None,
                            help='search request, all founded docs will be deleted before indexing')
        parser.add_argument('--data-url', default='https://droideka.smarttv.yandex.net/api/v6/channels',
                            help='droideka channels endpoint url')

    def handle(self, *args, **options):
        logger.info('Starting...')

        query = options['purge_request']
        if query is not None:
            Cleaner.rm_docs_by(query)

        ts = int(time())
        if options['ya_efir']:
            index_efir_channels(options['data_url'], ts)

        if options['my_efir']:
            index_my_efir(ts)

        if options['delete']:
            sleep(10)  # wait until new index messages will be merged
            Cleaner(options['device_id']).rm_docs_earlier_than(ts, exclude={MyEfirData.uri})

        logger.info('Done')
