import threading
import logging

from saas.library.persqueue.writer.python import writer as saas_writer
from django.conf import settings

import retry
from smarttv.alice.tv_proxy.proxy import unistat

broker_logger = logging.getLogger('logbroker')
# noinspection PyArgumentList
lb_logger = saas_writer.Logger(broker_logger)

logger = logging.getLogger(__name__)
writer_init_lock = threading.Lock()

topics_directory = 'saas/services/{name}/{ctype}/topics'.format(name=settings.SAAS_SERVICE_NAME,
                                                                ctype=settings.SAAS_CTYPE)


class FormatError(Exception):
    pass


class SaasWriterError(Exception):
    pass


class LogBrokerResult(object):
    def __init__(self, raw_result):
        self.result = raw_result

    @property
    def is_ok(self):
        return self.result.get('written', False)

    @property
    def error_details(self):
        return self.result.get('comment', 'no details')

    @property
    def is_user_error(self):
        return self.result.get('user_error', False)

    def __str__(self):
        return str(self.result)


class MessageWriter(unistat.UnistatMixin):
    unistat_suffix = unistat.LB_SERVICE

    class NotReadyError(Exception):
        pass

    def __init__(self):
        super(MessageWriter, self).__init__()

        self._saas_writer = None
        self._initialized = False

    # noinspection PyMethodMayBeStatic
    def _get_searchmap_settings(self):
        search_map_settings = saas_writer.SearchMapInputSettings()

        search_map_settings.dm_host = settings.SAAS_DM_HOST
        search_map_settings.statica_host = settings.SAAS_STATICA_HOST
        search_map_settings.statica_query = settings.SAAS_CTYPE
        search_map_settings.ctype = settings.SAAS_CTYPE

        return search_map_settings

    def _get_saas_writer_settings(self):
        writer_settings = saas_writer.Settings()
        writer_settings.set_logger(lb_logger)
        writer_settings.set_service_info(self._get_searchmap_settings(), settings.SAAS_SERVICE_NAME)
        writer_settings.set_tvm(settings.ALICE_INDEXER_TVM_CLIENT_ID, settings.LOGBROKER_TVM_CLIENT_ID,
                                settings.ALICE_INDEXER_TVM_SECRET)
        writer_settings.set_persqueue_settings(settings.LOGBROKER_HOST, topics_directory)
        writer_settings.set_max_attempts(3)

        return writer_settings

    @retry.retry(tries=3, delay=1, backoff=2)
    def _create_saas_writer(self):
        writer = saas_writer.Writer()
        writer.init(self._get_saas_writer_settings())
        logger.info('writer settings is set')
        return writer

    def is_ready(self):
        try:
            if self.get_saas_writer():
                return True
        except MessageWriter.NotReadyError:
            pass

        return False

    def get_saas_writer(self):
        if not self._initialized:
            with writer_init_lock:
                if not self._initialized:  # еще раз намеренно
                    logger.debug('Initializing writer')
                    # noinspection PyPep8,PyBroadException
                    try:
                        self._saas_writer = self._create_saas_writer()
                        self._initialized = True
                        logger.info('Saas persqueue writer created')
                    except Exception:
                        self._saas_writer = None
                        logger.exception('Saas persqueue writer creation error')
                else:
                    logger.debug('Writer already initialized (from lock)')
        else:
            logger.debug('Writer already initialized')

        if not self._saas_writer:
            raise MessageWriter.NotReadyError('Saas writer is not initialized')

        return self._saas_writer

    def _write_json(self, msg):
        future = self.get_saas_writer().write_json(msg)
        return LogBrokerResult(future.result())

    def send_message(self, msg):
        result = self._write_json(msg)
        logger.info('Send message result: %s', result)

        if result.is_ok:
            self.increment_unistat_counter(unistat.Status._2xx)
            return

        logger.error('Indexing error, details: %s', result.error_details)
        if result.is_user_error:
            self.increment_unistat_counter(unistat.Status._4xx)
            logger.error('Wrong message format, see details above')
            raise FormatError(msg)
        else:
            self.increment_unistat_counter(unistat.Status.error)
            raise SaasWriterError()


message_writer = MessageWriter()
