from collections import namedtuple
import logging

import requests
from django.conf import settings
from furl import furl

from smarttv.alice.tv_proxy.proxy import unistat

from smarttv.alice.tv_proxy.proxy.saas_defaults import default_prefix

logger = logging.getLogger(__name__)

session = requests.Session()

READ_TIMEOUT_SECONDS = 2.0
DEFAULT_TIMEOUT = (0.1, READ_TIMEOUT_SECONDS)

FoundDocument = namedtuple('FoundDocument', ['url', 'ts', 'device_id'])


class SaasSearchClient(unistat.UnistatMixin):
    unistat_suffix = unistat.SAAS_SERVICE

    def __init__(self):
        super(SaasSearchClient, self).__init__()

        self.base_url = furl(settings.SAAS_SEARCH_PROXY_URL)
        self.base_url.args['service'] = settings.SAAS_SERVICE_NAME
        self.base_url.args['kps'] = default_prefix
        self.base_url.args['format'] = 'json'
        self.base_url.args['relev'] = 'attr_limit=99999999'
        self.base_url.args['numdoc'] = 200
        self.base_url.args['timeout'] = int(READ_TIMEOUT_SECONDS) * 1000000  # required in microseconds

    def search(self, text):
        url = self.base_url.copy()
        url.args['text'] = text
        logger.info('saas search url: %s', url.url)

        found_docs = set()
        try:
            response = session.get(url.url, timeout=DEFAULT_TIMEOUT)
            self.collect_unistat_metrics(response)

            response.raise_for_status()
            r = response.json()['response']
            try:
                if r.get('results') and int(r['results'][0]['found']['all']) > 0:
                    for group in r['results'][0]['groups']:
                        for doc in group['documents']:
                            found_docs.add(build_found_document(doc))
            except (KeyError, IndexError, TypeError):
                self.increment_unistat_counter(unistat.Status.parsing_error)
                logger.exception('Unexpected response structure: %s', r)
        except requests.exceptions.RequestException:
            self.increment_unistat_counter(unistat.Status.error)
            logger.exception('Http request error')

        return found_docs


def build_found_document(saas_doc):
    url = saas_doc['url']
    ts = None
    device_id = None

    if 'properties' in saas_doc:
        if saas_doc['properties'].get('i_ts'):
            ts = int(saas_doc['properties']['i_ts'])
        if saas_doc['properties'].get('s_device_id'):
            device_id = saas_doc['properties']['s_device_id']

    return FoundDocument(url, ts, device_id)


saas_search_client = SaasSearchClient()
