import json
import logging

from django.http import HttpResponse
from django.views.decorators.http import require_GET
from rest_framework import status
from rest_framework.views import APIView

from smarttv.alice.tv_proxy.proxy import unistat
from smarttv.alice.tv_proxy.proxy.headers import QUASAR_DEVICE_ID_HEADER
from smarttv.alice.tv_proxy.proxy.http import get_http_header
from smarttv.alice.tv_proxy.proxy.indexer import index, IndexingError, make_indexer_message, CabelChannelData, Cleaner
from smarttv.alice.tv_proxy.proxy.mixins import RequestValidatorMixin
from smarttv.alice.tv_proxy.proxy.pq_delivery import message_writer
from smarttv.alice.tv_proxy.proxy.validators import CabelChannelsTvMsgValidator

logger = logging.getLogger(__name__)


@require_GET
def ping(_):
    if not message_writer.is_ready():
        return HttpResponse(status=status.HTTP_503_SERVICE_UNAVAILABLE)

    return HttpResponse(status=status.HTTP_200_OK)


@require_GET
def unistat_handler(_):
    metrics = unistat.manager.get_counter_metrics()
    return HttpResponse(json.dumps(metrics), content_type='text/json', status=status.HTTP_200_OK)


class IndexerView(RequestValidatorMixin, APIView):
    validator_class = CabelChannelsTvMsgValidator
    required_headers = (QUASAR_DEVICE_ID_HEADER,)

    def post(self, request):
        msg = self.get_validated_data(request)
        quasar_device_id = get_http_header(request, QUASAR_DEVICE_ID_HEADER)
        try:
            ts = msg['timestamp']
            messages = []
            for item in msg['channels']:
                data = CabelChannelData(item, quasar_device_id, ts)
                msg = make_indexer_message(data, ts)
                messages.append(msg)

            index(messages)

            cleaner = Cleaner(quasar_device_id)
            cleaner.rm_docs_earlier_than(ts)
        except IndexingError:
            return HttpResponse(
                content='indexing error, please retry later',
                status=status.HTTP_503_SERVICE_UNAVAILABLE
            )

        return HttpResponse(status=status.HTTP_200_OK)
