import logging
import os

import yenv

SAAS_SERVICE_NAME = 'smart_tv_channels'

SAAS_CTYPE = yenv.choose_key_by_type({
    'production': 'stable',
    'development': 'prestable',
}, fallback=True)

SAAS_DM_HOST = 'saas-dm-proxy.n.yandex-team.ru'
SAAS_STATICA_HOST = 'saas-searchmap.s3.mds.yandex.net'

ALICE_INDEXER_TVM_CLIENT_ID = 2020635
ALICE_INDEXER_TVM_SECRET = os.getenv('ALICE_INDEXER_TVM_SECRET', '')

LOGBROKER_TVM_CLIENT_ID = yenv.choose_key_by_type({
    'production': 2001059,
    'development': 2001147,
}, fallback=True)

LOGBROKER_HOST = yenv.choose_key_by_type({
    'production': 'logbroker.yandex.net',
    'development': 'logbroker-prestable.yandex.net',
}, fallback=True)

SAAS_SEARCH_PROXY_URL = yenv.choose_key_by_type({
    'production': 'http://saas-searchproxy.yandex.net:17000',
    'development': 'http://saas-searchproxy-prestable.yandex.net:17000',
}, fallback=True)

BASE_DIR = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
os.environ.setdefault('REQUESTS_CA_BUNDLE', '/etc/ssl/certs')

DEBUG = yenv.type == 'development' or os.environ.get('YA_TEST_RUNNER')

if DEBUG:
    os.environ.setdefault('SECRET_KEY', '_________')

# SECURITY WARNING: keep the secret key used in production secret!
SECRET_KEY = os.environ['SECRET_KEY']

ALLOWED_HOSTS = ['*']

# Application definition

INSTALLED_APPS = [
    'django.contrib.admin',
    'django.contrib.auth',
    'django.contrib.contenttypes',
    'django.contrib.sessions',
    'django.contrib.messages',
    'django.contrib.staticfiles',
    'smarttv.alice.tv_proxy.proxy',
    'rest_framework',
]

REST_FRAMEWORK = {
    'DEFAULT_RENDERER_CLASSES': [
        'rest_framework.renderers.JSONRenderer'
    ],
    'EXCEPTION_HANDLER': 'smarttv.alice.tv_proxy.proxy.exceptions.custom_exception_handler',
}

MIDDLEWARE = [
    'smarttv.alice.tv_proxy.middleware.FixEmptyHostMiddleware',
    'smarttv.alice.tv_proxy.middleware.LoggingContextMiddleware',
    'csp.middleware.CSPMiddleware',
    'django.middleware.common.CommonMiddleware',
    'django.middleware.security.SecurityMiddleware',
    'django.contrib.sessions.middleware.SessionMiddleware',
    'django.middleware.common.CommonMiddleware',
    'django.middleware.csrf.CsrfViewMiddleware',
    'django.contrib.messages.middleware.MessageMiddleware',
    'django.middleware.clickjacking.XFrameOptionsMiddleware',
    'django.middleware.http.ConditionalGetMiddleware',
]

CORS_ORIGIN_ALLOW_ALL = False
CORS_HOSTS = [
    'alice.smarttv.yandex.net',
    'alice.tst.smarttv.yandex.net',
]

if DEBUG:
    DEBUG_CORS_HOST = os.environ.get('DEBUG_CORS_HOST')
    if DEBUG_CORS_HOST:
        CORS_HOSTS.append(DEBUG_CORS_HOST)

if not DEBUG:
    INSTALLED_APPS.append('raven.contrib.django.raven_compat')

ROOT_URLCONF = 'smarttv.alice.tv_proxy.urls'

TEMPLATES = [
    {
        'BACKEND': 'library.python.django.template.backends.arcadia.ArcadiaTemplates',
        'OPTIONS': {
            'debug': DEBUG,
            'loaders': [
                'library.python.django.template.loaders.resource.Loader',
                'library.python.django.template.loaders.app_resource.Loader',
            ],
            'context_processors': [
                'django.template.context_processors.debug',
                'django.template.context_processors.request',
                'django.contrib.auth.context_processors.auth',
                'django.contrib.messages.context_processors.messages',
            ],
        },
    },
]

FORM_RENDERER = 'library.python.django.template.backends.forms_renderer.ArcadiaRenderer'

LANGUAGE_CODE = 'en-us'
USE_I18N = False

###########################
# Date and Time settings
TIME_ZONE = 'Europe/Moscow'
USE_TZ = True

# setting USE_L10N=False to use DATE_FORMAT and DATETIME_FORMAT for formatting instead of local formatting
USE_L10N = False
DATE_FORMAT = 'Y-m-d'
DATETIME_FORMAT = 'Y-m-d H:i:s'
###########################


# Database

DATABASES = {
    'default': {
        'ENGINE': 'django.db.backends.dummy',
        'NAME': 'mydatabase',
    }
}

# Logging
RAVEN_CONFIG = {
    'dsn': os.environ.get('SENTRY_DSN', ''),
    'string_max_length': 8192,
    'list_max_length': 256,
    'environment': yenv.type,
}


UNIFIED_AGENT_LOGS_URI = os.getenv('UNIFIED_AGENT_LOGS_URI')


def get_log_level():
    log_level = os.getenv('LOGGING_LEVEL')
    if log_level not in ('CRITICAL', 'FATAL', 'ERROR', 'WARNING', 'WARN', 'INFO', 'DEBUG'):
        log_level = yenv.choose_key_by_type({
            'production': logging.INFO,
            'testing': logging.DEBUG,
            'stress': logging.INFO,
        }, fallback=True)
    return log_level


def get_log_handlers():
    handlers = {
        'console': {
            'class': 'logging.StreamHandler',
            'formatter': 'default',
        },
        'qloud': {
            'class': 'logging.StreamHandler',
            'formatter': 'json',
        },
        'sentry': {
            'level': logging.ERROR,
            'class': 'raven.contrib.django.raven_compat.handlers.SentryHandler',
        },
    }

    if UNIFIED_AGENT_LOGS_URI:
        handlers['logbroker'] = {
            'class': 'logbroker.unified_agent.client.python.UnifiedAgentYdHandler',
            'logger_name': 'stdout',
            'formatter': 'json',
            'uri': UNIFIED_AGENT_LOGS_URI,
        }

    return handlers


def get_log_handlers_list():
    handlers = yenv.choose_key_by_type({
        'production': ['qloud', 'sentry'],
        'stress': ['qloud'],
        'development': ['console'],
    }, fallback=True)

    if UNIFIED_AGENT_LOGS_URI:
        handlers.append('logbroker')

    return handlers


LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'formatters': {
        'default': {
            'format': '[%(asctime)s] [%(process)d] [%(levelname)s] [%(name)s] %(message)s',
            'datefmt': '%Y-%m-%d %H:%M:%S',
            'class': 'logging.Formatter',
        },
        'raw': {
            'format': '%(message)s',
            'datefmt': '',
            'class': 'logging.Formatter',
        },
        'json': {
            '()': 'ylog.format.QloudJsonFormatter',
        },
    },
    'handlers': get_log_handlers(),
    'loggers': {
        '': {
            'level': get_log_level(),
            'handlers': get_log_handlers_list(),
            'propagate': True,
        },
        'gunicorn.access': {
            'handlers': get_log_handlers_list(),
        },
        'gunicorn.error': {
            'handlers': get_log_handlers_list(),
        },
    },
}

# Static files (CSS, JavaScript, Images)
# https://docs.djangoproject.com/en/1.11/howto/static-files/
STATIC_URL = '/static/'

STATICFILES_FINDERS = [
    'django.contrib.staticfiles.finders.FileSystemFinder',
    'django.contrib.staticfiles.finders.AppDirectoriesFinder',
    'library.python.django.contrib.staticfiles.finders.ArcadiaAppFinder'
]

SECURE_CONTENT_TYPE_NOSNIFF = True
SECURE_PROXY_SSL_HEADER = ('HTTP_X_FORWARDED_PROTO', 'https')

###################################

# Content Secure Policy (django-csp)
CSP_DEFAULT_SRC = ("'none'",)
CSP_SCRIPT_SRC = ("'unsafe-inline'", "'self'")
CSP_STYLE_SRC = ("'unsafe-inline'", "'self'")
CSP_IMG_SRC = ("'self'",)
