import json
import logging

import pytest
from django.test import Client
from smarttv.droideka.functional_tests.endpoint_info import EndpointInfo, Programs, Channels, Categories, Carousels, \
    Carousel, CardDetail, CardDetailThin, CardDetailRatings, Doc2doc, Search, ScreenSaver as ScreenSaverInfo, \
    SuggestHistory
from smarttv.droideka.functional_tests.smart_tv_client import Android, SmartTvClient
from smarttv.droideka.proxy.models import Versions, ScreenSaver, Category2
from smarttv.droideka.proxy.models import PlatformModel
from smarttv.droideka.utils import PlatformType
import mock

required_android_headers = {
    'HTTP_USER_AGENT': 'com.yandex.tv.home/1.2.773 (Realtek SmartTV; Android 7.1.1)',
    'HTTP_X_YAUUID': 'd68c1bca4efa403313837b12f5cdcd26',
    'HTTP_X_YADEVICEID': 'ca9b68da30474d5db0bbd1e8a25565bb',
}

http_client = Client(content_type='application.json')

api_version = 6


logger = logging.getLogger(__name__)


@mock.patch('smarttv.droideka.proxy.views.card.ThinCardDetailView.load_memento_configs',
            mock.Mock(return_value=None))
@mock.patch('smarttv.droideka.proxy.views.card.CardDetailV4View.load_memento_configs',
            mock.Mock(return_value=None))
@mock.patch('smarttv.droideka.proxy.views.card.RatingsCardDetail.load_memento_configs',
            mock.Mock(return_value=None))
@mock.patch('smarttv.droideka.proxy.views.doc2doc.PlatformDoc2DocView.load_memento_configs',
            mock.Mock(return_value=None))
@mock.patch('smarttv.droideka.proxy.views.carousels.CarouselV5View.load_memento_configs',
            mock.Mock(return_value=None))
@mock.patch('smarttv.droideka.proxy.views.carousels.CommonCarouselsView.load_memento_configs',
            mock.Mock(return_value=None))
@mock.patch('smarttv.droideka.proxy.views.channels.ChannelsView.load_memento_configs',
            mock.Mock(return_value=None))
@mock.patch('smarttv.droideka.proxy.views.programs.ProgramsView.load_memento_configs',
            mock.Mock(return_value=None))
@mock.patch('smarttv.droideka.proxy.views.search.SearchView.load_memento_configs',
            mock.Mock(return_value=None))
@mock.patch('smarttv.droideka.proxy.views.search.SuggestHistoryView.load_memento_configs',
            mock.Mock(return_value=None))
@mock.patch('smarttv.droideka.proxy.views.search.BaseSearchView.load_memento_configs',
            mock.Mock(return_value=None))
@mock.patch('smarttv.droideka.proxy.views.screensaver.ScreenSaverView.load_memento_configs',
            mock.Mock(return_value=None))
@mock.patch('smarttv.droideka.proxy.views.categories.CategoriesView.load_memento_configs',
            mock.Mock(return_value=None))
@mock.patch('smarttv.droideka.proxy.views.categories.CategoriesView.get_user_country_region_id', mock.Mock(return_value=225))  # region: Russio
@mock.patch('smarttv.droideka.proxy.views.categories.CategoriesView.is_from_russia', mock.Mock(return_value=True))
@pytest.mark.parametrize('client', [Android()], ids=lambda x: x.__class__.__name__)
class TestApi:
    endpoint_classes = (
        Programs, Channels, Carousels, Carousel, CardDetail, CardDetailThin, CardDetailRatings, Doc2doc, Search,
        SuggestHistory
    )

    proxy_only_endpoints = [endpoint(api_version) for endpoint in endpoint_classes]

    # noinspection PyMethodMayBeStatic
    def basic_checks(self, view_info, client):
        response = http_client.get(path=view_info.endpoint,
                                   data={**view_info.params_200},
                                   **client.required_headers)

        assert response.status_code in (200, 204), response.content

        if response.status_code != 204:
            obj = json.loads(response.content)
            assert obj  # not empty

    @pytest.mark.django_db
    @pytest.mark.parametrize('view_info', proxy_only_endpoints, ids=lambda x: x.endpoint_name)
    def test_ok(self, view_info: EndpointInfo, client: SmartTvClient):
        self.basic_checks(view_info, client)

    @pytest.mark.django_db
    def test_categories_ok(self, client: SmartTvClient):
        category = Category2(id=0, category_id='123', title='test category', rank=1, parent_category_id=None)
        category.save()
        category.above_platforms.add(PlatformModel.objects.filter(platform_type=PlatformType.NONE).first())
        category.below_platforms.add(PlatformModel.objects.filter(platform_type=PlatformType.NONE).first())
        category.exclude_platforms.add(PlatformModel.objects.filter(platform_type=PlatformType.NONE).first())
        category.include_platforms.add(PlatformModel.objects.filter(platform_type=PlatformType.ANY).first())

        self.basic_checks(view_info=Categories(api_version), client=client)

    @pytest.mark.django_db
    def test_screensaver_ok(self, client: SmartTvClient, view_info: EndpointInfo = ScreenSaverInfo(api_version)):
        version = 1
        Versions(entity=ScreenSaver.get_entity(), version=version).save()
        ScreenSaver(
            title='title',
            type=view_info.params_200['type'],
            resolution=view_info.params_200['resolutions'],
            rank=1,
            version=version
        ).save()

        self.basic_checks(view_info=view_info, client=client)


@mock.patch('smarttv.droideka.proxy.views.carousels.CarouselV5View.load_memento_configs',
            mock.Mock(return_value=None))
class TestApiValidation:
    @pytest.mark.parametrize('update_params', [{'tag': 'movie'}, {'filter': 'somefilter'}, {'tag': 'movie', 'filter': 'somefilter'}])
    def test_carousel_400(self, update_params):
        params = {
            'carousel_id': 'not_filterable_carousel_id',
            'limit': 10,
        }
        params.update(update_params)
        response = http_client.get(
            path='/api/v6/carousel',
            data=params,
            **required_android_headers
        )

        assert 400 == response.status_code


@mock.patch('smarttv.droideka.proxy.views.card.RatingsCardDetail.load_memento_configs',
            mock.Mock(return_value=None))
class TestVHNonStandartResponses:
    """
    На некоторые запросы VH отвечает нестандартно - закрывает соединение
    или выдает 404. Этот тест проверяет такие случаи.
    """
    def test_404(self):
        """
        При ответе 404 от CardInfo, мы отдаем клиенту 404, а не 500
        """
        response = http_client.get(
            path='/api/v6/card_detail/ratings',
            data={'content_id': '0000c1c6f25d18e581187f4ef0eaddbe'},
            **required_android_headers
        )

        assert response.status_code == 404

    def test_closed_connection(self):
        """
        Если VH отвечает пустым ответом и закрывает соединение, это все равно,
        что он сказал 404
        """
        response = http_client.get(
            path='/api/v6/card_detail/ratings',
            data={'content_id': 'pleasecloseconnectiontest'},
            **required_android_headers
        )

        assert response.status_code == 404
