import logging
from typing import Optional

import atexit
import collections
import signal
import time
import gevent

from rest_framework.views import APIView as BaseAPIView
from rest_framework.response import Response
from rest_framework import status as http_status

from werkzeug.serving import BaseWSGIServer, WSGIRequestHandler
from werkzeug.wrappers import Request as FlaskRequest, Response as FlaskResponse

from smarttv.droideka.proxy.serializers import serializers


logger = logging.getLogger(__name__)


class StackSampler:
    def __init__(self, sampling_interval=0.005):
        self.sampling_interval = sampling_interval
        self._started = None
        self._stack_counts = collections.defaultdict(int)

    def start(self):
        self._started = time.time()
        try:
            signal.signal(signal.SIGVTALRM, self._sample)
        except ValueError:
            raise ValueError('Can sample only on the main thread')

        signal.setitimer(signal.ITIMER_VIRTUAL, self.sampling_interval, self.sampling_interval)
        atexit.register(self.stop)

    def _sample(self, _, frame):
        stack = []
        while frame is not None:
            stack.append(self._format_frame(frame))
            frame = frame.f_back

        stack = ';'.join(reversed(stack))
        self._stack_counts[stack] += 1

    def _format_frame(self, frame):
        return '{}({})'.format(frame.f_code.co_name,
                               frame.f_globals.get('__name__'))

    def output_stats(self):
        if self._started is None:
            return ''
        elapsed = time.time() - self._started
        lines = ['elapsed {}'.format(elapsed),
                 'granularity {}'.format(self.sampling_interval)]
        ordered_stacks = sorted(self._stack_counts.items(),
                                key=lambda kv: kv[1], reverse=True)
        lines.extend(['{} {}'.format(frame, count)
                      for frame, count in ordered_stacks])
        return '\n'.join(lines) + '\n'

    def reset(self):
        self._started = time.time()
        self._stack_counts = collections.defaultdict(int)

    def stop(self):
        self.reset()
        signal.setitimer(signal.ITIMER_VIRTUAL, 0)

    def __del__(self):
        self.stop()


class _QuietHandler(WSGIRequestHandler):
    def log_request(self, *args, **kwargs):
        """Do not to pollute application logs."""
        pass


class StoppableServer(BaseWSGIServer):
    stopped = False

    def serve_forever(self):
        while not self.stopped:
            self.handle_request()

    def force_stop(self):
        self.server_close()
        self.stopped = True


class Emitter:
    def __init__(self, sampler, port=16384):
        self.sampler = sampler
        self.port = port
        self.server = StoppableServer('0.0.0.0', port, self.handle_request, _QuietHandler)
        self.server.log = lambda *args, **kwargs: None
        self.started = False

    def handle_request(self, environ, start_response):
        stats = self.sampler.output_stats()
        request = FlaskRequest(environ)
        if request.args.get('reset') in ('1', 'true'):
            self.sampler.reset()
        response = FlaskResponse(stats)
        return response(environ, start_response)

    def stop(self):
        if self.started:
            self.server.force_stop()
            self.sampler.stop()
            logger.info('Stopped emitter')

    def start(self):
        if not self.started:
            logger.info('Serving profiles on port {}'.format(self.port))
            self.started = True
            self.sampler.start()
            self.server.serve_forever()
        else:
            logger.info('Attempt to start already started server')


class EmitterHolder:
    emitter: Optional[Emitter]

    def __init__(self):
        self.emitter = None
        self.started_worker = None

    def _start(self):
        self.emitter = Emitter(StackSampler())
        self.started = True
        logger.info('Starting emitter')
        self.emitter.start()

    def start_emitter(self):
        if not self.started_worker:
            self.started_worker = gevent.spawn(self._start)
        else:
            logger.info('Attempt to start emitter twice')

    def stop_emitter(self):
        if self.started_worker:
            self.emitter.stop()
            self.emitter = None
            if self.started_worker.dead:
                self.started_worker.kill(block=False)
            self.started_worker = None
        else:
            logger.info('No emitter to stop')


emitter_holder = EmitterHolder()


class ProfilingControlView(BaseAPIView):
    def post(self, request):
        validator = serializers.ProfilingControlValidator(data=request.data)
        validator.is_valid(raise_exception=True)
        if validator.validated_data['state']:
            emitter_holder.start_emitter()
        else:
            emitter_holder.stop_emitter()
        return Response(status=http_status.HTTP_204_NO_CONTENT)
