import logging
import json

from collections import deque

import yenv
from django import forms
from django.conf import settings
from django.contrib import admin, messages
from django.contrib.auth.admin import UserAdmin
from django.contrib.auth.models import Group, User
from django.utils.html import format_html
from django.db import transaction
from django.conf.urls import re_path
from django.shortcuts import render, redirect
from rest_framework import exceptions
from smarttv.droideka.proxy.constants.carousels import KpCarousel
from blackbox import JsonBlackbox

from plus.utils.yauth import YaAuthAdminSite

from smarttv.droideka.proxy import tvm
from smarttv.droideka.proxy.identifiers import IdentifiersImporter
from smarttv.droideka.proxy.models import Versions, ScreenSaver, Device, ValidIdentifier, SharedPreferences, \
    PlatformModel, CategoryExtendedEditable, Category2, Category2Editable, SupportDeviceProxy, CategoryExperiment, \
    Promo, RecommendedChannel, SmotreshkaChannel, Cinema
from smarttv.droideka.proxy.forms import UploadIdentifiersForm
from smarttv.droideka.proxy.s3mds import s3_client
from smarttv.droideka.utils import PlatformType, PlatformInfo, is_only_one_true
from smarttv.utils.machelpers import normalize_mac


logger = logging.getLogger(__name__)

blackbox = JsonBlackbox(blackbox_client=settings.BLACKBOX_CLIENT)


class DroidekaAdminSite(YaAuthAdminSite):
    site_title = f'Droideka admin [{yenv.type}]'
    site_header = site_title


admin_site = DroidekaAdminSite()
admin_site.register(Group)


def ensure_version_for_entity_created(entity):
    """
    If record in 'Versions' table exists - return the same entity
    If record doesn't exist - create and return newly created entity
    """
    actual_version = Versions.objects.filter(entity=entity).first()
    if not actual_version:
        actual_version = Versions(entity=entity, version=0)
        actual_version.save()
    return actual_version


def copy_data_with_actual_version_to_new_version(actual_version, new_version, objects):
    """
    Copies all the data with version equal to current version and sets them new version

    This action required to edit new version of data without affecting existing
    This action performed only if there is no data with new version yet, otherwise - does nothing
    """
    if objects.filter(version=new_version).count() == 0:
        models = list(objects.filter(version=actual_version))
        if models:
            for model in models:
                model.pk = None
                model.version = new_version
            objects.bulk_create(objs=models)


class CategoryPublisher:
    def __init__(self):
        self.editable_categories_map = {}

    def _remove_old_categories(self):
        Category2.exclude_platforms.through.objects.all().delete()
        Category2.include_platforms.through.objects.all().delete()
        Category2.above_platforms.through.objects.all().delete()
        Category2.below_platforms.through.objects.all().delete()
        Category2.category_experiments.through.objects.all().delete()
        Category2.category_disable_experiments.through.objects.all().delete()
        Category2.objects.all().delete()

    def _get_category_groups(self) -> list:
        """
        Category model is an ancestral tree like structure
        This function does bread-first-search for obtaining every level of the tree(from top to bottom)
        """
        first_category_group = Category2Editable.objects.filter(visible=True, parent_category_id__isnull=True)
        category_groups = []

        categories_to_process = deque()
        categories_to_process.append(first_category_group)

        while len(categories_to_process) > 0:
            editable_categories = categories_to_process.popleft()
            group = []
            for categ in editable_categories:
                self.editable_categories_map[categ.id] = categ
                children = Category2Editable.objects.filter(visible=True, parent_category_id=categ.id)
                categories_to_process.append(children)
                group.append(categ.to_publishable_model())
            category_groups.append(group)
        return category_groups

    def _publish_categories_by_levels(self, category_groups: list):
        for group in category_groups:
            Category2.objects.bulk_create(group)

    def _update_platforms(self):
        for category in Category2.objects.all():
            editable_category = self.editable_categories_map.get(category.id)
            if not editable_category:
                logger.warning('No matched editable category for published category')
                continue
            for platform in editable_category.exclude_platforms.all():
                category.exclude_platforms.add(platform)
            for platform in editable_category.include_platforms.all():
                category.include_platforms.add(platform)
            for platform in editable_category.above_platforms.all():
                category.above_platforms.add(platform)
            for platform in editable_category.below_platforms.all():
                category.below_platforms.add(platform)

    def _update_experiments(self):
        for category in Category2.objects.all():
            editable_category = self.editable_categories_map.get(category.id)
            if not editable_category:
                logger.warning('No matched editable category for published category')
                continue
            for experiment in editable_category.category_experiments.all():
                category.category_experiments.add(experiment)
            for experiment in editable_category.category_disable_experiments.all():
                category.category_disable_experiments.add(experiment)

    def _do_publish(self):
        self._remove_old_categories()
        # It's required to publish categories level by level, since may be the case, when parent category has
        # 'visible=False', but it's children have 'visible=True', and in case of attempt to publish these children -
        # Integrity error will occur, since will be attempt to publish categories, which parent not in DB
        # So, BFS guarantees that on every level there are only children, which parent's already published
        groups = self._get_category_groups()
        self._publish_categories_by_levels(groups)
        self._update_platforms()
        self._update_experiments()

    def publish(self):
        with transaction.atomic():
            self._do_publish()


def publish_categories2(modeladmin, request, queryset):
    CategoryPublisher().publish()


class VersionedModelAdminMixin(admin.ModelAdmin):
    """
    Base class for model admin to safely edit data
    On every query new version data is generated(if necessary), and then this new version can be edited without
    affecting clients
    """

    @property
    def model_class(self):
        raise NotImplementedError

    def get_queryset(self, request):
        """
        General method for generating new version for entity when querying entities
        :param clazz: entity class
        :return: queryset for displaying in django admin
        """
        actual_version = ensure_version_for_entity_created(self.model_class.get_entity())
        new_version = actual_version.version + 1
        copy_data_with_actual_version_to_new_version(actual_version.version, new_version, self.model_class.objects)
        return self.model_class.objects.filter(version=new_version)

    def save_model(self, request, obj, form, change):
        actual_version = ensure_version_for_entity_created(self.model_class.get_entity())
        new_version = actual_version.version + 1
        obj.version = new_version
        super().save_model(request, obj, form, change)


class Category2ModelForm(forms.ModelForm):

    def clean_rank(self):
        content_type = self.cleaned_data.get('content_type')
        rank = self.cleaned_data.get('rank')
        is_side_menu_item = not content_type
        if is_side_menu_item and not rank:
            raise forms.ValidationError("'rank' must be positive integer for side menu item")
        return rank

    def clean_position(self):
        content_type = self.cleaned_data.get('content_type')
        position = self.cleaned_data.get('position')
        is_kp_carousel = content_type == KpCarousel.TYPE
        if is_kp_carousel and not position:
            raise forms.ValidationError("'position' must be positive integer for KP carousel")
        return position

    class Meta:
        model = Category2Editable
        fields = '__all__'


@admin.register(CategoryExtendedEditable, site=admin_site)
class CategoryExtendedAdmin(admin.ModelAdmin):
    list_display = ('title', 'get_image_preview')
    search_fields = ('category_id', 'title')
    fields = ('category_id', 'title', 'content_type', 'rank', 'position', 'icon_s3_key', 'description',
              'exclude_platforms', 'include_platforms', 'above_platforms', 'below_platforms', 'parent_category',
              'visible')
    ordering = ('position', 'rank')
    empty_value_display = '-empty-'

    def get_image_url(self, icon_s3_key):
        return s3_client.get_url(icon_s3_key)

    def get_image_preview(self, obj):
        if not obj.icon_s3_key:
            return None
        return format_html('<img src="{}" width="24" height="24" style="background-color:black;" />',
                           self.get_image_url(obj.icon_s3_key))


@admin.register(Category2Editable, site=admin_site)
class Category2Admin(admin.ModelAdmin):
    form = Category2ModelForm

    list_display = ('title', 'get_image_preview')
    search_fields = ('category_id', 'title')
    fields = ('id', 'category_id', 'title', 'content_type', 'rank', 'position', 'icon_s3_key', 'thumbnail_s3_key',
              'logo_s3_key', 'banner_S3_key', 'description', 'exclude_platforms', 'include_platforms',
              'above_platforms', 'below_platforms', 'parent_category', 'authorization_required', 'show_in_tandem',
              'category_experiments', 'category_disable_experiments', 'persistent_client_category_id', 'visible',
              'carousel_type')
    readonly_fields = ('id',)
    ordering = ('position', 'rank')
    empty_value_display = '-empty-'
    actions = (publish_categories2,)

    def get_image_url(self, icon_s3_key):
        return s3_client.get_url(icon_s3_key)

    def get_image_preview(self, obj):
        if not obj.icon_s3_key:
            return None
        return format_html('<img src="{}" width="24" height="24" style="background-color:black;" />',
                           self.get_image_url(obj.icon_s3_key))


@admin.register(Versions, site=admin_site)
class VersionsAdmin(admin.ModelAdmin):
    """
    Manages versions of data visible to client
    All changes here will directly affect the clients
    """
    list_display = ('entity',)
    fields = ('entity', 'version')
    ordering = ('entity',)
    search_fields = ('entity',)
    empty_value_display = '-empty-'


@admin.register(ScreenSaver, site=admin_site)
class ScreenSaverModelAdmin(VersionedModelAdminMixin):
    list_display = ('title', 's3_key')
    search_fields = ('title', 's3_key')
    fields = ('title', 'type', 'resolution', 's3_key', 'rank', 'visible', 'version')
    readonly_fields = ('version',)
    ordering = ('rank',)
    empty_value_display = '-empty-'
    list_filter = ('type', 'resolution', 'visible')

    @property
    def model_class(self):
        return ScreenSaver


@admin.register(Device, site=admin_site)
class DeviceAdmin(admin.ModelAdmin):
    list_display = ('hardware_id', 'serial_number', 'wifi_mac', 'ethernet_mac',
                    'subscription_puid', 'promocode', 'created_at')
    search_fields = ('hardware_id', 'serial_number', 'wifi_mac', 'ethernet_mac', 'subscription_puid')
    empty_value_display = '-empty-'


def gift_given(obj):
    return bool(obj.subscription_puid) or obj.kp_gifts_given
gift_given.short_description = 'Gift given?'


def get_login(obj):
    if yenv.type not in ('production', 'prestable') or not obj or not obj.subscription_puid or \
            not settings.DEPLOY_POD_IP_ADDRESS:
        return '-'
    response = blackbox.userinfo(
        uid=obj.subscription_puid,
        userip=settings.DEPLOY_POD_IP_ADDRESS,
        headers=tvm.add_service_ticket(settings.BLACKBOX_CLIENT_ID))
    try:
        return response['users'][0]['login']
    except (TypeError, KeyError, IndexError):
        return '-'
get_login.short_description = 'Login'


@admin.register(SupportDeviceProxy, site=admin_site)
class SupportDeviceAdmin(admin.ModelAdmin):
    search_fields = ('wifi_mac', 'ethernet_mac')
    list_display = (gift_given, 'created_at', get_login, 'subscription_puid')
    list_display_links = None

    change_list_template = 'activation_info/get.html'

    def get_search_results(self, request, queryset, search_term):
        if not search_term or len(search_term) < 2:
            return SupportDeviceProxy.objects.none(), False
        return super().get_search_results(request, queryset, normalize_mac(search_term))

    def changelist_view(self, request, extra_context=None):
        search_query = request.GET.get('q')
        extra_context = {'title': 'Lookup for the device by MAC address', 'has_search_query': bool(search_query)}
        return super().changelist_view(request, extra_context=extra_context)

    def has_add_permission(self, request):
        return False

    def has_delete_permission(self, request, obj=None):
        return False

    def has_change_permission(self, request, obj=None):
        return False

    def save_model(self, request, obj, form, change):
        pass

    def delete_model(self, request, obj):
        pass

    def save_related(self, request, form, formsets, change):
        pass


@admin.register(ValidIdentifier, site=admin_site)
class ValidIdentifierAdmin(admin.ModelAdmin):
    list_display = ('type', 'value')
    list_filter = ('type',)
    search_fields = ('value',)

    change_list_template = 'valid_identifier/changelist.html'

    def get_urls(self):
        my_urls = [
            re_path('import/', self.import_from_file),
        ]
        return my_urls + super().get_urls()

    def import_from_file(self, request):
        if request.method == "POST":
            logger.info('Start uploading identifiers to database')
            try:
                raw_identifiers = IdentifiersImporter.read_file(request.FILES["identifiers_file"])
            except IdentifiersImporter.InvalidMacType:
                logger.exception('Invalid mac type')
                self.message_user(request, 'File has invalid mac type in the first line', level=messages.ERROR)
                return redirect('..')
            except IdentifiersImporter.NoMacs:
                logger.exception('File has no any mac')
                self.message_user(request, 'There are no macs in the file', level=messages.WARNING)
                return redirect('..')

            try:
                models = IdentifiersImporter.create_models(raw_identifiers)
            except exceptions.ValidationError:
                logger.exception('File contains invalid mac addresses')
                self.message_user(request, 'File contains invalid mac addresses', level=messages.ERROR)
                return redirect('..')
            IdentifiersImporter.save_identifiers(models)
            logger.info('Successfully uploaded %s identifiers', len(models))

            return redirect('..')
        form = UploadIdentifiersForm()
        payload = {"form": form}
        return render(
            request, 'valid_identifier/upload.html', payload
        )


class SharedPreferencesModelForm(forms.ModelForm):

    @staticmethod
    def validate_only_one_field_selected(data):
        """
        Checks that at time, only one field of SharedPreferences has been selected
        Since it has no sense when, for example 'int' field and 'string' field both have values

        The idea of preferences - it's key-value storage with next format: <key[string]: value[type]>
        'key' can't have multiple values. It should have only 1 value, and this value must always have only 1 type
        'value' should never change it's type!
        """
        int_value = data.get('int_value')
        bool_value = data.get('bool_value')
        char_value = data['char_value'] if data.get('char_value') else None  # force '' -> None
        datetime_value = data.get('datetime_value')
        if not is_only_one_true(int_value, bool_value, char_value, datetime_value):
            raise forms.ValidationError("Field of only one type can be set")

    @staticmethod
    def validate_json_field(data):
        is_json = data.get('is_json')
        if not is_json:
            return
        char_value = data.get('char_value')
        if not char_value:
            raise forms.ValidationError('Empty string can not be interpreted as JSON')
        try:
            json.loads(char_value)
        except json.decoder.JSONDecodeError as e:
            raise forms.ValidationError(f'Provided JSON is invalid: {e.msg}')

    def clean(self):
        cleaned_data = super(SharedPreferencesModelForm, self).clean()
        self.validate_only_one_field_selected(cleaned_data)
        self.validate_json_field(cleaned_data)
        return cleaned_data

    class Meta:
        model = SharedPreferences
        fields = '__all__'


@admin.register(SharedPreferences, site=admin_site)
class SharedPreferencesAdmin(admin.ModelAdmin):
    form = SharedPreferencesModelForm
    list_display = ('key',)
    fields = ('key', 'tag', 'int_value', 'bool_value', 'char_value', 'is_json', 'datetime_value')
    ordering = ('key',)
    search_fields = ('key',)
    empty_value_display = '-empty-'


@admin.register(PlatformModel, site=admin_site)
class PlatformAdmin(admin.ModelAdmin):
    list_display = (PlatformType.KEY, PlatformInfo.KEY_PLATFORM_VERSION,
                    PlatformInfo.KEY_APP_VERSION, PlatformInfo.KEY_DEVICE_MANUFACTURER, PlatformInfo.KEY_DEVICE_MODEL)
    search_fields = (PlatformType.KEY, PlatformInfo.KEY_PLATFORM_VERSION,
                     PlatformInfo.KEY_APP_VERSION)
    fields = (PlatformType.KEY, PlatformInfo.KEY_PLATFORM_VERSION,
              PlatformInfo.KEY_APP_VERSION, PlatformInfo.KEY_DEVICE_MANUFACTURER,
              PlatformInfo.KEY_DEVICE_MODEL, PlatformInfo.KEY_QUASAR_PLATFORM)
    empty_value_display = '-empty-'
    list_filter = (PlatformType.KEY,)


@admin.register(CategoryExperiment, site=admin_site)
class ExperimentAdmin(admin.ModelAdmin):
    list_display = ('value', 'description')
    search_fields = ('value',)
    fields = ('value', 'description')
    empty_value_display = '-empty-'


@admin.register(User, site=admin_site)
class CustomUserAdmin(UserAdmin):
    actions = ['make_superuser']

    def make_superuser(self, request, queryset):
        queryset.update(is_staff=True, is_active=True, is_superuser=True)

    make_superuser.short_description = "Make selected users superuser"


@admin.register(Promo, site=admin_site)
class CustomPromoAdmin(admin.ModelAdmin):
    list_display = ('promo_id', 'promo_type', 'enabled')
    search_fields = ('promo_id',)
    list_filter = ('promo_type',)
    fields = ('enabled', 'promo_id', 'promo_type', 'thumbnail', 'title', 'subtitle', 'content_type',
              'action_value', 'fallback_action')
    empty_value_display = '-empty-'


@admin.register(RecommendedChannel, site=admin_site)
class RecommendedChannelAdmin(admin.ModelAdmin):
    list_display = ('title', 'rank', 'visible')
    ordering = ('rank',)


@admin.register(SmotreshkaChannel, site=admin_site)
class SmotreshkaChannelAdmin(admin.ModelAdmin):
    list_display = ('number', 'title', 'enabled')
    ordering = ('number',)


@admin.register(Cinema, site=admin_site)
class CinemaAdmin(admin.ModelAdmin):
    list_display = ('code', 'enabled')
    readonly_fields = ('created_at',)
