import logging
from dataclasses import dataclass

from smarttv.utils.headers import (
    OBJECT_RESPONSE_CLIENT,
    OTT_SERVICE_ID,
    OTT_SERVICE_NAME,
    STRM_FROM,
    VIDEOPOISK_CLIENT,
)

logger = logging.getLogger(__name__)


@dataclass
class AccessConfig:
    """
    Contains parameters for accessing third party services: Videohosting, OTT(Kinopoisk backend), Videopoisk

    P.S.
    Do not try to generalize these parameters somehow. Their values can be the same sometimes, but in fact, they are
    absolutely different, and every parameter should be used in specific place.
    """

    vh_service: str
    vh_from: str
    ott_service_id: int
    videopoisk_client: str
    object_response_client: str


def tv_default():
    return AccessConfig(
        vh_service='ya-tv-android',
        vh_from='tvandroid',
        ott_service_id=167,
        videopoisk_client='tvandroid',
        object_response_client='tvandroid',
    )


def module_default():
    return AccessConfig(
        vh_service='ya-tv-android',
        vh_from='tvandroid',
        ott_service_id=234,
        videopoisk_client='tvandroid',
        object_response_client='tvandroid',
    )


class AccessConfigGetter:
    def __call__(self, request):
        return self.get_access_config(request)

    def fill_from_headers(self, config, headers):
        # header -> attribute_name
        map = {
            OTT_SERVICE_NAME: 'vh_service',
            OTT_SERVICE_ID: 'ott_service_id',
            STRM_FROM: 'vh_from',
            VIDEOPOISK_CLIENT: 'videopoisk_client',
            OBJECT_RESPONSE_CLIENT: 'object_response_client',
        }

        for key, attr in map.items():
            if key in headers:
                setattr(config, attr, headers[key])
                logger.debug('%s from client headers: %s', attr, headers[key])

    def is_module(self, request):
        if hasattr(request, 'platform_info'):
            return request.platform_info.is_module()
        return False

    def is_service_headers_broken(self, request):
        if hasattr(request, 'platform_info'):
            return request.platform_info.is_service_headers_broken()
        return False

    def get_access_config(self, request):
        """
        Возвращает конфиг сервисных параметров на основе значений, переданных с клиента
        в заголовках, либо дефолтов
        """
        config = tv_default()

        if not request:
            # при запуске manage-команды, например
            logger.info('No request provided, using default tv config')
            return config

        if self.is_module(request):
            logger.info('Using module default config')
            config = module_default()

        if self.is_service_headers_broken(request):
            # see SMARTTVBACKEND-1104, ALICE-18018
            logger.info('Skip parsing service headers because of broken client version')
            return config

        logger.info('Overwriting service params from client headers')
        self.fill_from_headers(config, request.headers)

        return config


tv_config = tv_default()
get_access_config = AccessConfigGetter()
