import logging

import requests
from rest_framework import status

from plus.utils.api import BaseAPIService
from plus.utils.mediabilling import settings

logger = logging.getLogger(__name__)

# Класс скопирован из plus/utils/mediabilling
# todo: убрать зависимость от кода плюса в этом месте


class MediaBillingAPI(BaseAPIService):
    base_url = settings.MEDIABILLING_API_URL

    allowed_methods = ('GET', 'POST')
    default_timeout = settings.DEFAULT_TIMEOUT
    max_retries = settings.DEFAULT_RETRIES

    use_tvm = False

    dst_tvm_id = settings.MEDIABILLING_CLIENT_ID

    class UnavailableForLegalReasons(Exception):
        pass

    def prepare_headers(self, headers=None):
        headers = super(MediaBillingAPI, self).prepare_headers(headers)

        headers = {
            'Accept': 'application/json',
            **headers
        }
        return headers

    def process_http_error(self, response):
        if response.status_code == status.HTTP_451_UNAVAILABLE_FOR_LEGAL_REASONS:
            raise self.UnavailableForLegalReasons()
        try:
            response.raise_for_status()
        except requests.exceptions.HTTPError:
            msg = f'Request to {self.__class__.__name__} failed due to HTTP error {response.status_code}.'

            json_response = super(MediaBillingAPI, self).parse_response(response)
            if json_response and 'error' in json_response:
                error = json_response['error']
                msg += f" Error name='{error.get('name', '')}', message='{error.get('message')}'"

            logger.warning(msg, exc_info=True)
            raise self.RequestError(msg, status_code=response.status_code, response_body=response.text)

    def consume_promocode(self, uid, promocode, user_ip, source=None,
                          timeout=settings.TIMEOUT_CONSUME_PROMOCODE):
        """
        Activates promocode
        """
        params = {
            '__uid': uid,
            'ip': user_ip,
            'code': promocode,
            'source': source,
        }

        response = self.request(
            'POST',
            'billing/promo-code/consume',
            params=params,
            timeout=timeout
        )
        return response['result']

    def clone_promocode(self, prototype, timeout=None):
        params = {
            'prototype': prototype,
        }

        response = self.request(
            'POST',
            'billing/promo-code/clone',
            params=params,
            timeout=timeout
        )

        promocode = response['result'].get('code')
        if not promocode:
            raise self.APIError('Failed to issue promocode by prototype: empty response from MediaBilling API')

        return promocode
