import logging
from copy import copy
from typing import Optional

from django.conf import settings
from smarttv.droideka.proxy.api import base
from smarttv.droideka.proxy.api.access_config import get_access_config
from smarttv.droideka.proxy.constants.carousels import DROIDEKA_INTERNAL_OTT_WINDOW_ID
from smarttv.droideka.utils import RequestInfo
from smarttv.utils import headers as headers_keys

logger = logging.getLogger(__name__)

KEY_SELECTION_WINDOW_ID = 'selection_window_id'
KEY_SELECTION_ID = 'selection_id'

KEY_AVAILABILITY = 'availability'
KEY_STATUS = 'status'

UNAVAILABILITY_REASONS = [
    'UNAVAILABLE_FOR_USER_AGENT',
    'UNAVAILABLE_IN_CURRENT_REGION',
]

OTT_NOT_FOUND_STATUS = 'NOT_FOUND'


class OttContentApi(base.BaseJsonApi):
    unistat_suffix = 'ott'
    authorization_failed_codes = frozenset((401, 403))

    API_VERSION = settings.OTT_API_VERSION

    def _fill_device_id_if_exists(self, request_info: RequestInfo, headers: dict):
        if request_info.quasar_device_id:
            headers[headers_keys.OTT_DEVICE_ID] = request_info.quasar_device_id

    def metadata(self, content_id, initial_request, headers=None):
        path = f'{self.API_VERSION}/hd/content/{content_id}/metadata'
        return self._request(endpoint=path, headers=headers, request=initial_request)

    def films_to_watch(self, content_id, initial_request, headers=None):
        path = f'{self.API_VERSION}/films-to-watch/{content_id}'
        return self._request(endpoint=path, headers=headers, request=initial_request)

    def children(self, content_id, initial_request, headers=None):
        """
        :return: Series structure. seasons + episodes
        """
        path = f'{self.API_VERSION}/hd/content/{content_id}/children'
        return self._request(endpoint=path, headers=headers, request=initial_request)

    def selections(
            self, selection_window_id, selection_id, headers, initial_request, offset=0, limit=10,
            session_id=None) -> dict:
        headers = copy(headers)
        path = f'{self.API_VERSION}/selections/{selection_id}'
        params = {
            'serviceId': self.get_service_id(initial_request),
            'offset': offset,
            'limit': limit,
        }
        if selection_window_id != DROIDEKA_INTERNAL_OTT_WINDOW_ID:
            params['selectionWindowId'] = selection_window_id
        if session_id:
            params['sessionId'] = session_id
        self._fill_device_id_if_exists(initial_request.request_info, headers)
        return self._request(endpoint=path, params=params, headers=headers, request=initial_request)

    def multi_selections(self, selection_window_id, headers, initial_request, items_limit=10,
                         selections_limit=10, selections_offset=0, session_id=None):
        headers = copy(headers)
        path = f'{self.API_VERSION}/selections'
        params = {
            'serviceId': self.get_service_id(initial_request),
            'itemsLimit': items_limit,
            'selectionsLimit': selections_limit,
            'selectionsOffset': selections_offset,
            'pageId': 'multiselection',
        }
        if selection_window_id != DROIDEKA_INTERNAL_OTT_WINDOW_ID:
            params['selectionWindowId'] = selection_window_id
        if session_id:
            params['sessionId'] = session_id
        self._fill_device_id_if_exists(initial_request.request_info, headers)
        return self._request(endpoint=path, params=params, headers=headers, request=initial_request)

    def purchases(self, headers, initial_request, offset: int = 0, limit: int = 10,
                  available_only: bool = False, content_id=None):
        service_id = self.get_service_id(initial_request)
        if not content_id:
            path = f'{self.API_VERSION}/purchases'
            params = {'serviceId': service_id, 'offset': offset, 'limit': limit, 'availableOnly': available_only}
        else:
            path = f'{self.API_VERSION}/purchases/{content_id}'
            params = {'serviceId': service_id}
        return self._request(endpoint=path, params=params, headers=headers, request=initial_request)

    def create_kp_profile(self, headers, initial_request):
        path = f'{self.API_VERSION}/profiles'
        params = {'kpProfileAutoCreation': True}
        return self._request(method='POST', endpoint=path, params=params, headers=headers, request=initial_request)

    def get_filmography(self, person_id, headers, initial_request):
        path = f'{self.API_VERSION}/persons/{person_id}/filmography'
        return self._request(endpoint=path, headers=headers, request=initial_request)

    def handle_status(self, response, **kwargs):
        exception_cls = None
        msg = None
        msg_args = ()
        if response.status_code in self.authorization_failed_codes:
            msg = f'OTT API authorization failed with code {response.status_code}'
            exception_cls = self.PermissionDeniedError
        elif response.status_code == 400:
            msg = f'OTT API: Bad request (code {response.status_code}), response body: %s'
            msg_args = (response.content,)
            exception_cls = self.BadRequestError

        if exception_cls is not None:
            base.log_and_raise(exception_cls, msg, log_msg_args=msg_args, logger=logger)

        super().handle_status(response)

    def handle_response(self, json_response, params, headers):
        if json_response and KEY_AVAILABILITY in json_response and KEY_STATUS in json_response[KEY_AVAILABILITY]:
            status = json_response[KEY_AVAILABILITY][KEY_STATUS]
            args = (headers.get(headers_keys.USER_AGENT_HEADER), params)
            if status in UNAVAILABILITY_REASONS:
                base.log_and_raise(self.PermissionDeniedError, 'Not allowed for %s, %s', args, 'Not allowed')
            elif status == OTT_NOT_FOUND_STATUS:
                base.log_and_raise(self.PermissionDeniedError, 'Not found for %s, %s', args, 'Not found')

    def add_from(self, params: Optional[dict]):
        if params is not None:
            params['from'] = 'ya-tv-android'
            params['droideka'] = '1'

    def get_service_id(self, request) -> str:
        return str(get_access_config(request).ott_service_id)

    def _request(self, method='GET', endpoint='', headers=None, params=None, timeout=None, data=None, **make_request_kwargs):
        self.add_from(params)
        return super()._request(method, endpoint, headers, params, timeout, data, **make_request_kwargs)


client = OttContentApi(
    url=settings.OTT_CONTENT_API_URL,
    timeout=settings.OTT_CONTENT_DEFAULT_TIMEOUT,
    retries=settings.OTT_CONTENT_DEFAULT_RETRIES,
)
