import logging
from typing import Optional

from django.conf import settings

from smarttv.utils.headers import GO_ZORA_USER_AGENT
from smarttv.droideka.proxy.api import base
from smarttv.droideka.proxy.tvm import tvm_client

logger = logging.getLogger(__name__)


class SmotreshkaApi(base.BaseJsonApi):
    unistat_suffix = 'smotreshka'
    backend_user_agent = 'YandexTv backend'
    pbs_limit_message = 'play back session limit reached'

    class PbsLimitReached(base.BaseJsonApi.Forbidden):
        pass

    class BadRequest(base.BaseJsonApi.RequestError):
        pass

    def handle_status(self, response, **kwargs):
        if response.status_code == 403:
            if self.pbs_limit_message in response.text:
                raise self.PbsLimitReached('Pbs limit reached')
            raise self.Forbidden(f'Http 403: {response.text}')
        elif response.status_code == 400:
            raise self.BadRequest(f'Http 400: {response.text}')

        super().handle_status(response, **kwargs)

    @property
    def proxies(self) -> Optional[dict]:
        source = settings.GO_ZORA_SOURCE_NAME
        tvm_ticket = tvm_client.get_service_ticket(settings.GO_ZORA_TVM_CLIENT_ID)
        zora_url = f'http://{source}:{tvm_ticket}@{settings.GO_ZORA_URL}:{settings.GO_ZORA_PORT}/'
        return {
            'http': zora_url,
            'https': zora_url,
        }

    @property
    def verify_ssl(self) -> bool:
        return False

    def _get_headers(self, user_agent=None):
        headers = {
            'Content-Type': 'application/json',
        }
        if user_agent:
            headers[GO_ZORA_USER_AGENT] = user_agent
        return headers

    def begin(self, raw_data: dict):
        endpoint = '/playback/tv/v2/begin-for-client'
        headers = self._get_headers(self.backend_user_agent)
        logger.debug('Sending %r', raw_data)
        return self._request('POST', endpoint=endpoint, headers=headers, json=raw_data)

    def end(self, playback_session_id, initial_request=None):
        endpoint = '/playback/tv/v2/end-for-client'
        headers = self._get_headers(self.backend_user_agent)
        data = {'pbsId': playback_session_id}
        return self._request('POST', endpoint=endpoint, headers=headers, json=data)

    def get_tv_asset_token_for_account(self, account_id=None, region=None):
        endpoint = '/user/v2/tv-asset-token-for-account'
        headers = self._get_headers()
        params = {'accountId': account_id}
        if region:
            params['region'] = region

        return self._request('GET', endpoint=endpoint, headers=headers, params=params)

    def get_playback_sessions(self, account_id):
        endpoint = '/playback-session/v2/list-for-account'
        params = {'accountId': account_id}
        return self._request('GET', endpoint=endpoint, params=params)

    def get_channels(self, tv_asset_token):
        endpoint = '/tv/v2/channels'
        headers = self._get_headers()
        params = {'tv-asset-token': tv_asset_token}
        return self._request('GET', endpoint=endpoint, headers=headers, params=params)

    def get_medias(self, tv_asset_token):
        """
        Получить список медиа для пользователя
        Обратите внимание, смотрешка возвращает 400: {"id":"bad request","message":"bad request"},
        если ключ неправильный
        """
        endpoint = '/tv/v2/medias'
        headers = self._get_headers()
        params = {'tv-asset-token': tv_asset_token}
        return self._request('GET', endpoint=endpoint, headers=headers, params=params)


class BillingApi(base.BaseJsonApi):
    unistat_suffix = 'smotreshka_billing'

    class ApiError(Exception):
        pass

    class Forbidden(ApiError):
        pass

    def handle_status(self, response, **kwargs):
        msg = f'[{response.status_code}] {response.text}'
        if response.status_code == 403:
            raise self.Forbidden(msg)
        else:
            raise self.ApiError(msg)

    @property
    def proxies(self) -> Optional[dict]:
        source = settings.GO_ZORA_SOURCE_NAME
        tvm_ticket = tvm_client.get_service_ticket(settings.GO_ZORA_TVM_CLIENT_ID)
        zora_url = f'http://{source}:{tvm_ticket}@{settings.GO_ZORA_URL}:{settings.GO_ZORA_PORT}/'
        return {
            'http': zora_url,
            'https': zora_url,
        }

    @property
    def verify_ssl(self) -> bool:
        return False

    def _get_headers(self, user_agent=None):
        headers = {
            'Content-Type': 'application/json',
        }
        if user_agent:
            headers[GO_ZORA_USER_AGENT] = user_agent
        return headers

    def create_account(self, username, email, purchases=None, password=None, region=None, info=None):
        data = {
            'username': username,
            'email': email,
        }

        if purchases:
            data['purchases'] = purchases

        if password:
            data['password'] = password

        if region:
            data['region'] = region

        if info:
            data['info'] = info

        headers = self._get_headers()
        result = self._request('POST', '/v2/accounts', headers=headers, json=data)

        return result

client = SmotreshkaApi(
    url=settings.SMOTRESHKA_URL,
    timeout=settings.SMOTRESHKA_DEFAULT_TIMEOUT,
    retries=settings.SMOTRESHKA_DEFAULT_RETRIES,
)

billing_client = BillingApi(
    url=settings.SMOTRESHKA_BILLING_URL,
    timeout=settings.SMOTRESHKA_BILLING_DEFAULT_TIMEOUT,
    retries=settings.SMOTRESHKA_BILLING_DEFAULT_RETRIES,
)
