import base64
import enum
import logging
import json
from collections import defaultdict
from typing import Iterable, Optional

from django.conf import settings

from smarttv.utils import add_log_context, headers as header_keys

from smarttv.droideka.proxy.api import base
from smarttv.droideka.proxy.models import SharedPreferences
from smarttv.droideka.proxy.constants.reserved_shared_pref_keys import PREFERENCES_TEST_IDS
from smarttv.droideka.utils import YANDEX_MODULE_QUASAR_PLATFORM
from ylog.context import LogContext

logger = logging.getLogger(__name__)

HOMEAPP_HANDLER = 'SMARTTV-HOMEAPP'
PLAYER_HANDLER = 'SMARTTV-PLAYER'
BACKEND_HANDLER = 'SMARTTV-BACK'

KNOWN_HANDLERS = {
    HOMEAPP_HANDLER,
    PLAYER_HANDLER,
    BACKEND_HANDLER,
}


class DeviceModel(enum.Enum):
    MODULE = 'yandexmodule_2'
    UNKNOWN = 'Unknown'


KNOWN_QUASAR_PLATFORMS_MAPPING = defaultdict(lambda: DeviceModel.UNKNOWN, {
    YANDEX_MODULE_QUASAR_PLATFORM: DeviceModel.MODULE
})


class Experiment:
    def __init__(self, key: str, value: str, handler: str):
        self.key = key
        self.value = value
        self.handler = handler

    def send_to_client(self):
        return self.handler != BACKEND_HANDLER

    def serialize(self):
        return {
            'key': self.key,
            'value': self.value,
            'handler': self.handler,
        }


class StubExperiments:
    """ Empty experiments for endpoints not providing proper identifiers """

    def get_value(self, key):
        return None

    def get_value_or_default(self, key, default=None):
        return None


class EmptyIcookieException(Exception):
    message = 'Experiments icookie is empty'


class LazyExperiments:
    """ Lazily goes to UaaS for experiments """
    _experiments = None
    _experiments_dict = None
    _log_data = None

    def __init__(self, test_ids, user_ip, app_version, auth_header, request):
        self.test_ids = test_ids or []
        self.icookie = request.request_info.experiments_icookie
        self.auth_header = auth_header
        self.user_ip = user_ip
        self.app_version = app_version
        self.quasar_platform = request.platform_info.quasar_platform
        self.quasar_device_id = request.request_info.quasar_device_id

    def load_additional_test_ids(self):
        """
        Workaround for test_ids override in backend experiments
        Delete when client will send test_ids to all backend requests
        """

        # TODO: Delete this code when TVANDROID-3127 be ready

        if not settings.USAAS_TESTIDS_FROM_PREFERENCES:
            return
        additional_test_ids = SharedPreferences.get_string(PREFERENCES_TEST_IDS)
        if not additional_test_ids:
            return
        for test_id in additional_test_ids.split(','):
            if test_id not in self.test_ids:
                self.test_ids.append(test_id)

    def load_experiments(self):
        if self.icookie is None:
            # This exception will be caught, experiment_errors counter will raise.
            # Client will get results without experiments
            raise EmptyIcookieException()

        self.load_additional_test_ids()
        result = client.get_split(
            icookie=self.icookie,
            auth_header=self.auth_header,
            ip=self.user_ip,
            test_ids=self.test_ids,
            app_version=self.app_version,
            quasar_platform=self.quasar_platform,
            quasar_device_id=self.quasar_device_id
        )
        self._experiments = list(result['experiments'])
        self._experiments_dict = {exp.key: exp.value for exp in self._experiments}
        self._log_data = result['log_data']
        add_log_context(experiments_data=self._log_data)
        logger.info('Experiments loaded, boxes: %s', self._log_data['exp_boxes'])

    @property
    def all(self):
        if self._experiments is None:
            self.load_experiments()
        return self._experiments

    def get_value(self, key):
        if self._experiments_dict is None:
            self.load_experiments()
        return self._experiments_dict.get(key)

    def get_value_or_default(self, key, default=None):
        try:
            return self.get_value(key)
        except EmptyIcookieException:
            return default

    @property
    def log_data(self):
        if self._log_data is None:
            self.load_experiments()
        return self._log_data


class USaaSApi(base.BaseJsonApi):
    unistat_suffix = 'uaas'

    VERSION = 'version'
    DEVICE_TYPE = 'deviceType'
    DEVICE_MODEL = 'deviceModel'

    def get_experiments(self, experiments_data: dict):
        try:
            handlers = experiments_data['all']['CONTEXT']['MAIN']
        except KeyError:
            base.log_and_raise(self.RequestError, 'Failed to parse handlers')
        for handler_name, flags in handlers.items():
            if handler_name not in KNOWN_HANDLERS:
                continue
            for key, value in flags.items():
                yield Experiment(
                    key=key,
                    value=value,
                    handler=handler_name
                )

    def get_log_data(self, experiments_data: dict) -> dict:
        try:
            return {
                'exp_boxes': experiments_data['exp_boxes']
            }
        except KeyError:
            base.log_and_raise(self.RequestError, 'Failed to parse exp_boxes')

    def fill_app_info_if_necessary(self, headers: dict, app_version: Optional[str], quasar_platform):
        data = {}

        if app_version:
            data[self.VERSION] = app_version

        mapped_device_model = KNOWN_QUASAR_PLATFORMS_MAPPING[quasar_platform]
        if mapped_device_model != DeviceModel.UNKNOWN:
            data[self.DEVICE_MODEL] = mapped_device_model.value

        if not data:
            return
        data[self.DEVICE_TYPE] = 'tv'

        headers[header_keys.EXP_APP_INFO] = base64.b64encode(json.dumps(data).encode()).decode()

    def fill_host_if_necessary(self, headers: dict):
        if hasattr(settings, 'USAAS_DROIDEKA_ENVIRONMENT_HOST') and settings.USAAS_DROIDEKA_ENVIRONMENT_HOST:
            headers[header_keys.HOST_HEADER] = settings.USAAS_DROIDEKA_ENVIRONMENT_HOST

    def fill_auth_token_if_necessary(self, headers: dict, auth_token: Optional[str]):
        if auth_token:
            headers[header_keys.AUTHORIZATION_HEADER] = auth_token

    def log_usaas_request(self, cgi_params: dict, headers: dict, usaas_response: dict):
        if not settings.LOG_USAAS_REQUESTS:
            return
        headers.pop(header_keys.AUTHORIZATION_HEADER, None)  # do not log oauth token if it exists
        logging_usaas_info = {
            'api_url': self.api_url,
            'cgi_params': cgi_params,
            'headers': headers,
            'response': usaas_response
        }

        with LogContext(usaas_log_request=True):
            logger.info(json.dumps(logging_usaas_info, ensure_ascii=False))

    # noinspection PyUnusedLocal
    def get_split(self,
                  icookie: str,
                  auth_header: str,
                  test_ids: Iterable[str],
                  ip: str,
                  app_version: Optional[str],
                  quasar_platform: str,
                  quasar_device_id) -> dict:
        headers = {
            header_keys.EXP_SERVICE: settings.USAAS_SERVICE,
            header_keys.EXP_HANDLER: settings.USAAS_SERVICE,
            header_keys.VH_ICOOKIE_HEADER: icookie,
            header_keys.FORWARDED_Y_HEADER: ip,
        }
        self.fill_app_info_if_necessary(headers, app_version, quasar_platform)
        self.fill_host_if_necessary(headers)
        self.fill_auth_token_if_necessary(headers, auth_header)

        params = {'deviceid': quasar_device_id}
        if test_ids:
            # https://wiki.yandex-team.ru/jandekspoisk/kachestvopoiska/abt/uaas/#kakjamogupopastvtrebuemyevyborki
            params['test-id'] = '_'.join(test_ids)

        experiments_data = self._request(params=params, headers=headers)
        self.log_usaas_request(params, headers, experiments_data)
        return {
            'experiments': self.get_experiments(experiments_data),
            'log_data': self.get_log_data(experiments_data),
        }


client = USaaSApi(
    url=settings.USAAS_URL,
    timeout=settings.USAAS_TIMEOUT,
    retries=settings.USAAS_RETRIES,
)
