import inspect
import logging
import uuid
from collections import defaultdict
from enum import Enum
import yenv

import requests
from django.conf import settings
from requests.exceptions import ConnectionError
from urllib3.util.retry import Retry

from ylog.context import LogContext
from smarttv.utils import headers as header_keys

from smarttv.droideka.proxy.api import base
from smarttv.droideka.proxy.api.access_config import get_access_config
from smarttv.droideka.proxy.api.es import OO_CONTENT_TYPE_SERIES, OO_CONTENT_TYPE_EPISODE
from smarttv.droideka.utils import dict_to_string
from smarttv.droideka.unistat import manager
from smarttv.droideka.proxy.constants.reserved_shared_pref_keys import PREF_VH_BETA_ETHERNET_MACS
from smarttv.droideka.proxy.models import SharedPreferences

GENERAL_VH_RESPONSE_LOG_PARAMS = ('user_data', 'reqid', 'apphost-reqid')

SERVICE_PROD = 'ya-tv-android'

CONTENT_TYPE_SERIES = 'vod-library'
CONTENT_TYPE_EPISODE = 'vod-episode'

GOOD_CONTENT_ID = '4ab7044baa2c2dc599c244196429245f'
# "Ну, погоди" content_id. Здесь нужен идшник, который наверняка есть в vh,
# чтобы сервис, если доступен, всегда отвечал на такой запрос 200

experiment_errors_counter = manager.get_counter('experiment_errors')


class OntoCategory(Enum):
    FILM = 'film'
    ANIM_FILM = 'anim_film'
    SERIES = 'series'
    ANIM_SERIES = 'anim_series'


OTT_CONTENT_TYPES = frozenset(
    (CONTENT_TYPE_SERIES, CONTENT_TYPE_EPISODE, OO_CONTENT_TYPE_SERIES, OO_CONTENT_TYPE_EPISODE))

VH_CONTENT_TYPES = (CONTENT_TYPE_SERIES, CONTENT_TYPE_EPISODE, 'episode', 'season')

logger = logging.getLogger(__name__)

_TRACKING_PARAM_REQ_ID = 'reqid'
_TRACKING_PARAM_APP_HOST_REQ_ID = 'apphost-reqid'
_TRACKING_PARAM_USER_DATA = 'user_data'
_TRACKING_PARAMS = (_TRACKING_PARAM_REQ_ID, _TRACKING_PARAM_APP_HOST_REQ_ID, _TRACKING_PARAM_USER_DATA)

# TODO: remove when all parameters will have new format (SMARTTVBACKEND-670)
OLD_VH_PARAMS_EXPERIMENTS_KEY = 'vh_additional_params'
VH_PARAMS_EXPERIMENTS_KEY = 'vh_params'


def log_vh_request_params(request_fields: list = None, response_fields: list = None):
    def vh_request_decorator(func):
        # noinspection PyPep8
        def func_wrapper(*args, **kwargs):
            func_args = inspect.signature(func).bind(*args, **kwargs).arguments
            func_args['func_name'] = func.__name__
            args_to_log = {}
            if request_fields:
                for field_name in request_fields:
                    args_to_log[field_name] = func_args.get(field_name, 'No such field')
            with LogContext(request_params=args_to_log):
                # noinspection PyBroadException
                try:
                    result = func(*args, **kwargs)
                except:
                    logger.warning('Error performing request request params: %s', dict_to_string(args_to_log))
                    raise
                result_params_to_log = {}
                if response_fields and result:
                    for field_name in response_fields:
                        if field_name in result:
                            result_params_to_log[field_name] = result[field_name]
                if request_fields or response_fields:
                    with LogContext(result_params=result_params_to_log):
                        logger.info('Request params: %s, '
                                    'response params: %s',
                                    dict_to_string(args_to_log),
                                    dict_to_string(result_params_to_log))
            return result

        return func_wrapper

    return vh_request_decorator


def is_ott_content(content_type):
    return content_type in OTT_CONTENT_TYPES


def propagate_vh_tracking_params(origin_vh_response, result_response):
    """
    Propagates VH's tracking labels to client, in order to let the client propagate these params to player for
    better recommendations
    """
    for param in _TRACKING_PARAMS:
        if param in origin_vh_response:
            result_response[param] = origin_vh_response[param]


def fake_stream_url(vh_response):
    """
    Add fake values for missed stream.url for back-compatibility

    One day, VH decided not include url for stream if that stream is not
    allowed for user to be watched. But old clients require this field. So we're
    faking these urls for them.
    """
    try:
        for item in vh_response['set']:
            for stream_item in item.get('streams', []):
                if stream_item.get('url'):
                    # stream already has address - skip it
                    continue
                if stream_item.get('stream_type') == 'DASH':
                    stream_item['url'] = '.mpd'
                elif stream_item.get('stream_type') == 'HLS':
                    stream_item['url'] = '.m3u8'
                else:
                    stream_item['url'] = ''
    except (KeyError, TypeError) as err:
        logger.info('VH response has incorrect format, can not add stream url. Error: %s', err)


class VhIdLogger:
    RESULT_KEY_AGGREGATED_IDS = 'aggregated_ids'

    KEY_FEED = 'feed'
    KEY_CAROUSEL = 'carousel'
    KEY_DOC2DOC = 'doc2doc'
    KEY_PROGRAMS = 'programs'

    KEY_USER_DATA = 'user_data'
    KEY_REQID = 'reqid'
    KEY_APPHOST_REQID = 'apphost-reqid'
    KEY_SEARCH_REQID = 'search-reqid'

    KEY_CONTENT_ID = 'content_id'
    KEY_PARENT_ID = 'parent_id'
    KEY_TITLE = 'title'
    KEY_INCLUDES = 'includes'
    KEY_SET = 'set'
    KEY_ITEMS = 'items'
    KEY_CAROUSEL_ID = 'carousel_id'
    KEY_CACHE_HASH = 'cache_hash'
    KEY_DOCS_CACHE_HASH = 'docs_cache_hash'

    KEY_END_TIME = 'end_time'
    KEY_PROGRAM_ID = 'program_id'
    KEY_START_TIME = 'start_time'
    KEY_CHANNEL_ID = 'channel_id'
    KEY_BANNED = 'banned'

    @classmethod
    def _get_log_info_for_document(cls, document):
        result = {
            cls.KEY_CONTENT_ID: document.get(cls.KEY_CONTENT_ID),
            cls.KEY_PARENT_ID: document.get(cls.KEY_PARENT_ID),
            cls.KEY_TITLE: document.get(cls.KEY_TITLE)
        }
        if cls.KEY_INCLUDES in document:
            # carousel / doc2doc / feed can contain only first episode in the response
            first_episode = document[cls.KEY_INCLUDES][0]
            result[cls.KEY_INCLUDES] = cls._get_log_info_for_document(first_episode)
        return result

    @classmethod
    def _get_log_info_for_carousel(cls, carousel, **kwargs):
        content_list = carousel.get(cls.KEY_INCLUDES) or carousel.get(cls.KEY_SET)
        carousel_id = carousel.get(cls.KEY_CAROUSEL_ID) or kwargs.get(cls.KEY_CAROUSEL_ID)
        docs_cache_hash = carousel.get(cls.KEY_DOCS_CACHE_HASH) or carousel.get(cls.KEY_CACHE_HASH)
        banned = carousel.get(cls.KEY_BANNED, False)
        if banned:
            return carousel
        result = []
        for item in content_list:
            result.append(cls._get_log_info_for_document(item))
        return {cls.KEY_CAROUSEL_ID: carousel_id, cls.KEY_DOCS_CACHE_HASH: docs_cache_hash, 'content': result}

    @classmethod
    def _get_log_info_for_doc2doc(cls, response):
        documents = response.get(cls.KEY_SET)
        if not documents:
            return None
        result = []
        for document in documents:
            result.append(cls._get_log_info_for_document(document))
        return result

    @classmethod
    def _get_log_info_for_feed(cls, response):
        carousels = response.get(cls.KEY_ITEMS)
        cache_hash = response.get(cls.KEY_CACHE_HASH)
        if not carousels:
            logger.warning('Empty feed response, nothing to log')
            return None
        result = []
        for item in carousels:
            log_info = cls._get_log_info_for_carousel(item)
            if log_info:
                result.append(log_info)
        return {cls.KEY_CACHE_HASH: cache_hash, 'carousels': result}

    @classmethod
    def _get_program_content(cls, program):
        banned = program.get(cls.KEY_BANNED)
        if banned:
            return program
        end_time = program.get(cls.KEY_END_TIME)
        program_id = program.get(cls.KEY_PROGRAM_ID)
        start_time = program.get(cls.KEY_START_TIME)
        channel_id = program.get(cls.KEY_CHANNEL_ID)
        title = program.get(cls.KEY_TITLE)
        if not end_time or not program_id or not start_time or not channel_id or not title:
            logger.warning('Strange program: %s', repr(program))
        result = {}
        if end_time:
            result[cls.KEY_END_TIME] = end_time
        if program_id:
            result[cls.KEY_PROGRAM_ID] = program_id
        if start_time:
            result[cls.KEY_START_TIME] = start_time
        if channel_id:
            result[cls.KEY_CHANNEL_ID] = channel_id
        if title:
            result[cls.KEY_TITLE] = title
        return result

    @classmethod
    def _get_log_info_for_programs(cls, response):
        if not response or not response.get(cls.KEY_SET):
            return None
        programs = response['set']
        return [cls._get_program_content(program) for program in programs]

    @classmethod
    def get_log_info(cls, response, key, **kwargs):
        info = {
            cls.KEY_USER_DATA: response.get(cls.KEY_USER_DATA),
            cls.KEY_REQID: response.get(cls.KEY_REQID),
            cls.KEY_APPHOST_REQID: response.get(cls.KEY_APPHOST_REQID),
            cls.KEY_SEARCH_REQID: response.get(cls.KEY_SEARCH_REQID)
        }

        if key == cls.KEY_FEED:
            info[cls.RESULT_KEY_AGGREGATED_IDS] = cls._get_log_info_for_feed(response)
        if key == cls.KEY_CAROUSEL:
            info[cls.RESULT_KEY_AGGREGATED_IDS] = cls._get_log_info_for_carousel(response, **kwargs)
        if key == cls.KEY_DOC2DOC:
            info[cls.RESULT_KEY_AGGREGATED_IDS] = cls._get_log_info_for_doc2doc(response)
        if key == cls.KEY_PROGRAMS:
            info[cls.RESULT_KEY_AGGREGATED_IDS] = cls._get_log_info_for_programs(response)
        return info

    @classmethod
    def log_content_ids(cls, response, key, **kwargs):
        if not settings.LOG_CONTENT_ID:
            return None
        info = cls.get_log_info(response, key, **kwargs)
        if info:
            with LogContext(response_content=info):
                logger.info(info)


class FrontendVHApi(base.BaseJsonApi):
    unistat_suffix = 'vh'

    SERVICE_MAIN = 'ya-main'

    _KEY_USER_AGENT = 'User-Agent'

    CAROUSEL_ITEM_GRAPHQL_FIELDS = """
    {
        content_id
        content_type_name
        parent_id
        streams
        adConfig
        ottParams
        release_date_ut
        channel_id
        subtitle
        description
        original_source
        auto_fields
        title
        program_title
        release_year
        genres
        onto_id
        onto_category
        duration
        thumbnail
        onto_poster
        percentage_score
        rating_kp
        start_time
        end_time
        can_play_till
        blogger
        blogger_id
        restriction_age
        series
        season
        episode_number
        parent_vod_lib_id
        is_special_project
        meta
        skippableFragments
        sport
        sport_event
        supertag
        likes
        dislikes
        progress
        can_play_on_station
    }
    """

    FILTERABLE_CAROUSEL_GRAPHQL_QUERY = """
    {{
        carousel({data})
        {fields}
    }}
    """

    LOG_ERROR_RESPONSE_HEADERS = ('x-yandex-req-id', 'x-forwardtouser-y')

    def __init__(self, **kwargs):
        super().__init__(**kwargs)

    def get_service_params(self, request) -> dict:
        config = get_access_config(request)
        return {
            'service': config.vh_service,
            'from': config.vh_from,
        }

    def get_beta_ethernet_macs(self):
        beta_macs = SharedPreferences.get_json(PREF_VH_BETA_ETHERNET_MACS)
        if not isinstance(beta_macs, list):
            logger.debug('Beta ethernet macs must be valid JSON list of strings')
            return None
        return beta_macs

    def get_url_part_by_user_type(self, initial_request, ordinary_part, beta_part):
        if yenv.type != 'testing' or not initial_request:
            return ordinary_part
        beta_macs = self.get_beta_ethernet_macs()
        if not beta_macs or initial_request.request_info.ethernet_mac not in beta_macs:
            return ordinary_part
        return beta_part

    @staticmethod
    def get_graphql_params(params: dict) -> str:
        if not params:
            raise ValueError('Can not generate params from empty dict')
        str_list = []
        for key, value in params.items():
            if value:
                actual_value = f'{value}' if isinstance(value, (int, float)) else f'"{value}"'
                str_list.append(f'{key}:{actual_value}')
        return ','.join(str_list)

    def gen_graphql_request(self, data: dict) -> str:
        graphql_params = self.get_graphql_params(data)
        return self.FILTERABLE_CAROUSEL_GRAPHQL_QUERY.format(data=graphql_params,
                                                             fields=self.CAROUSEL_ITEM_GRAPHQL_FIELDS)

    def get_graphql_cgi_params(self, graphql_data: dict) -> dict:
        result = {}
        for key, value in graphql_data.items():
            if '-' in key:
                result[key] = graphql_data[key]
        for key in result.keys():
            del graphql_data[key]
        return result

    def is_graphql_response_ok(self, status_code):
        return 200 <= status_code <= 299

    def make_request(self, method, url, headers, params, timeout, data, **kwargs):
        try:
            return super().make_request(method, url, headers, params, timeout, data, **kwargs)
        except requests.exceptions.ConnectionError:
            if not kwargs.get('is_connection_check') and self.is_connection_available(headers):
                msg = 'ConnectionError while requesting url %s with params %s, assuming Not Found',
                msg_args = (url, params)
                base.log_and_raise(
                    exception_cls=self.NotFoundError,
                    log_msg=msg,
                    log_msg_args=msg_args,
                    exception_msg='Not found',
                    logger=logger,
                )
            else:
                raise

    def get_default_params(self, initial_request):
        params = self.get_service_params(initial_request)
        if getattr(initial_request, 'platform_info', None) is not None:
            app_version = initial_request.platform_info.app_version
            params['client_version'] = str(app_version)
        else:
            logger.warning('No version info, not sending it to vh')

        self.update_experimental_params(initial_request, params)
        return params

    def update_experimental_params(self, initial_request, params):
        if initial_request is None:
            logger.debug('No request: no additional params from experiments')
            return

        new_params = defaultdict(set)
        try:
            additional_params = initial_request.request_info.experiments.get_value(VH_PARAMS_EXPERIMENTS_KEY)
        except Exception:
            logger.warning('Failed to load experiments', exc_info=True)
            experiment_errors_counter.increment()
            return

        if not additional_params:
            logger.debug('Additional params from experiments are empty')
            additional_params = None
            return

        if not isinstance(additional_params, list):
            logger.warning('Wrong additional params format')
            experiment_errors_counter.increment()
            additional_params = None

        sep = '='
        if additional_params is not None:
            for param in additional_params:
                if sep not in param:
                    logger.warning("CGI parameter %s doesn't contain %s, skipping", param, sep)
                    continue
                key, value = param.split(sep, 1)
                new_params[key].add(value)

        params.update(new_params)
        logger.debug('Additional params from experiments: %s', additional_params)

    # noinspection PyMethodMayBeStatic
    def fill_optional_param(self, params, key, value, extra=None):
        if value:
            params[key] = value
            if extra:
                params.update(extra)

    def update_headers(self, headers):
        headers = super().update_headers(headers)
        headers[header_keys.INTERNAL_REQUEST] = '1'
        return headers

    @log_vh_request_params(request_fields=('region_id',),
                           response_fields=GENERAL_VH_RESPONSE_LOG_PARAMS)
    def channels_regions(self, initial_request, headers, region_id=None, project_alias=None,
                         with_filtration=1):
        api_version = self.get_url_part_by_user_type(initial_request, settings.VH_API_VERSION,
                                                     settings.VH_BETA_API_VERSION)
        endpoint = f'{api_version}/channels_regions'
        params = self.get_default_params(initial_request)

        self.fill_optional_param(params, 'project_alias', project_alias)
        if region_id:
            self.fill_optional_param(params, 'region', region_id, {'with_filtration': with_filtration})

        return self._request('GET', endpoint=endpoint, headers=headers, params=params)

    @log_vh_request_params(request_fields=('from_time', 'to_time', 'kwargs'),
                           response_fields=GENERAL_VH_RESPONSE_LOG_PARAMS)
    def episodes(self, initial_request, headers, from_time=None, to_time=None, **kwargs):
        api_version = self.get_url_part_by_user_type(initial_request, settings.VH_API_VERSION,
                                                     settings.VH_BETA_API_VERSION)
        endpoint = f'{api_version}/episodes.json'
        params = self.get_default_params(initial_request)
        params.update(kwargs)

        params['limit'] = params.get('limit', 20)
        params['offset'] = params.get('offset', 0)
        self.fill_optional_param(params, 'end_date__from', from_time)
        self.fill_optional_param(params, 'start_date__to', to_time)

        response = self._request('GET', endpoint=endpoint, headers=headers, params=params)
        VhIdLogger.log_content_ids(response, VhIdLogger.KEY_PROGRAMS)
        return response

    def get_uuid(self, request):
        if not request:
            return None
        uuid_str = request.headers.get(header_keys.UUID_HEADER)
        if uuid_str:
            try:
                uuid.UUID(uuid_str)  # validation
                return uuid_str
            except ValueError:
                logger.warning('Incorrect uuid: %s', uuid_str)

    @log_vh_request_params(request_fields=('content_id',),
                           response_fields=GENERAL_VH_RESPONSE_LOG_PARAMS)
    def content_detail(self, initial_request, headers, content_id, method='GET'):
        api_version = self.get_url_part_by_user_type(initial_request, settings.VH_API_VERSION,
                                                     settings.VH_BETA_API_VERSION)
        endpoint = f'{api_version}/player'
        params = self.get_default_params(initial_request)

        uuid = self.get_uuid(initial_request)
        if uuid:
            params['uuid'] = uuid

        path = '{}/{}.json'.format(endpoint, content_id)
        return self._request(method=method,
                             endpoint=path,
                             headers=headers,
                             params=params,
                             is_connection_check=method == 'HEAD')

    @log_vh_request_params(request_fields=('series_id',),
                           response_fields=GENERAL_VH_RESPONSE_LOG_PARAMS)
    def seasons(self, initial_request, headers, series_id, limit=20, offset=0):
        api_version = self.get_url_part_by_user_type(initial_request, settings.VH_API_VERSION,
                                                     settings.VH_BETA_API_VERSION)
        endpoint = f'{api_version}/series_seasons.json'
        params = self.get_default_params(initial_request)

        params['series_id'] = series_id
        params['limit'] = limit
        params['offset'] = offset

        return self._request('GET', endpoint=endpoint, headers=headers, params=params)

    @log_vh_request_params(request_fields=('season_id',),
                           response_fields=GENERAL_VH_RESPONSE_LOG_PARAMS)
    def series_episodes(self, initial_request, headers, season_id, limit=20, offset=0):
        api_version = self.get_url_part_by_user_type(initial_request, settings.VH_API_VERSION,
                                                     settings.VH_BETA_API_VERSION)
        endpoint = f'{api_version}/series_episodes.json'
        params = self.get_default_params(initial_request)

        params['season_id'] = season_id
        params['limit'] = limit
        params['offset'] = offset

        return self._request('GET', endpoint=endpoint, headers=headers, params=params)

    @log_vh_request_params(request_fields=('tag', 'cache_hash'),
                           response_fields=GENERAL_VH_RESPONSE_LOG_PARAMS)
    def carousels_videohub(self, initial_request, headers, tag=None, limit=10, offset=0, enable_channels=1, num_docs=10,
                           cache_hash=None, restriction_age=None):

        api_version = self.get_url_part_by_user_type(initial_request, settings.VH_API_VERSION,
                                                     settings.VH_BETA_API_VERSION)
        endpoint = f'{api_version}/carousels_videohub.json'
        params = self.get_default_params(initial_request)

        params['enable_channels'] = enable_channels
        params['limit'] = limit
        params['num_docs'] = num_docs
        params['delete_filtered'] = 1
        params['stream_options'] = 'hires'

        self.fill_optional_param(params, 'tag', tag)
        self.fill_optional_param(params, 'cache_hash', cache_hash)
        self.fill_optional_param(params, 'offset', offset)
        if restriction_age is not None:
            self.fill_optional_param(params, 'relev', f'age_limit={restriction_age}')

        return self._request('GET', endpoint=endpoint, headers=headers, params=params)

    @log_vh_request_params(request_fields=('tag', 'cache_hash'),
                           response_fields=GENERAL_VH_RESPONSE_LOG_PARAMS)
    def feed(self, initial_request, headers, tag=None, limit=10, offset=0, num_docs=10, delete_filtered=0,
             locale='ru', param_filter='carousels', cache_hash=None, restriction_age=None,
             block_continue_watching=True, block_recommendations=True):

        api_version = self.get_url_part_by_user_type(initial_request, settings.VH_API_VERSION,
                                                     settings.VH_BETA_API_VERSION)
        endpoint = f'{api_version}/feed.json'
        params = self.get_default_params(initial_request)

        params['limit'] = limit
        params['num_docs'] = num_docs
        params['delete_filtered'] = delete_filtered
        params['locale'] = locale
        params['filter'] = param_filter

        # Continue watching carousels comes when user not finished watching some film
        # by default this carousel is enabled, and to disable it, it's required
        # to explicitly specify '&build_delayed=0' in query params
        if block_continue_watching:
            params['build_delayed'] = 0

        # recommendations carousels comes at first position in every tag. It's enabled by default.
        # To disable it, it's required to explicitly specify '&mixed_carousel=0' in query params
        if block_recommendations:
            params['mixed_carousel'] = 0

        params['rearr'] = ['scheme_Local/VideohubMiddle/AssocMinGroupSize=5']

        self.fill_optional_param(params, 'tag', tag)
        self.fill_optional_param(params, 'cache_hash', cache_hash)
        self.fill_optional_param(params, 'offset', offset)
        if restriction_age is not None:
            self.fill_optional_param(params, 'relev', f'age_limit={restriction_age}')

        response = self._request('GET', endpoint=endpoint, headers=headers, params=params)
        VhIdLogger.log_content_ids(response, VhIdLogger.KEY_FEED)
        return response

    def carousel(self, initial_request, headers, carousel_id, tag, filters, offset=0, limit=12, cache_hash=None,
                 restriction_age=None):
        endpoint = self.get_url_part_by_user_type(initial_request, 'graphql', f'{settings.VH_BETA_API_VERSION}/graphql')
        data = {
            'carousel_id': carousel_id,
            'limit': limit,
            'offset': offset,
            'tag': tag,
            'filters': filters,
            'docs_cache_hash': cache_hash,
        }
        default_params = self.get_default_params(initial_request)
        data.update(default_params)
        cgi_params = self.get_graphql_cgi_params(data)
        data = self.gen_graphql_request(data)

        response = self._request('POST', endpoint=endpoint, headers=headers, params=cgi_params, data=data)
        carousel_response = response['carousel']
        if not self.is_graphql_response_ok(carousel_response['status_code']):
            raise self.BadGatewayError()
        try:
            carousel = carousel_response['content']
        except KeyError:
            return None
        VhIdLogger.log_content_ids(carousel, VhIdLogger.KEY_CAROUSEL, carousel_id=carousel_id)
        return carousel

    @log_vh_request_params(request_fields=('carousel_id', 'docs_cache_hash'),
                           response_fields=GENERAL_VH_RESPONSE_LOG_PARAMS)
    def carousel_videohub(self, initial_request, headers, carousel_id, offset=0, limit=10, docs_cache_hash=None,
                          tag=None, restriction_age=None):

        api_version = self.get_url_part_by_user_type(initial_request, settings.VH_API_VERSION,
                                                     settings.VH_BETA_API_VERSION)
        endpoint = f'{api_version}/carousel_videohub.json'
        params = self.get_default_params(initial_request)

        params['carousel_id'] = carousel_id
        params['limit'] = limit
        if tag:
            params['tag'] = tag

        self.fill_optional_param(params, 'docs_cache_hash', docs_cache_hash)
        params['offset'] = offset
        if restriction_age is not None:
            self.fill_optional_param(params, 'relev', f'age_limit={restriction_age}')

        if carousel_id == 'ChFoaG9nd3dra3h5dWF1c3RoaBIJdHZhbmRyb2lkIAAoAA==':
            params['rearr'] = 'scheme_Local/VideohubMiddle/RequestType=tv_new_search_promo'

        response = self._request('GET', endpoint=endpoint, headers=headers, params=params)
        VhIdLogger.log_content_ids(response, VhIdLogger.KEY_CAROUSEL, carousel_id=carousel_id)
        return response

    @log_vh_request_params(request_fields=('content_id', 'rvb'),
                           response_fields=GENERAL_VH_RESPONSE_LOG_PARAMS)
    def doc2doc(self, initial_request, headers, content_id, offset=0, limit=10, rvb=None,
                restriction_age=None):
        api_version = self.get_url_part_by_user_type(initial_request, settings.VH_API_VERSION,
                                                     settings.VH_BETA_API_VERSION)
        endpoint = f'{api_version}/doc2doc'
        params = self.get_default_params(initial_request)

        params['content_id'] = content_id
        params['limit'] = limit

        self.fill_optional_param(params, 'params', offset)
        if rvb:
            self.fill_optional_param(params, 'rvb', rvb)
        if restriction_age is not None:
            self.fill_optional_param(params, 'relev', f'age_limit={restriction_age}')

        response = self._request('GET', endpoint=endpoint, headers=headers, params=params)
        VhIdLogger.log_content_ids(response, VhIdLogger.KEY_DOC2DOC)
        return response

    def carousel_content_filters(self, initial_request, headers):
        api_version = self.get_url_part_by_user_type(initial_request, settings.VH_API_VERSION,
                                                     settings.VH_BETA_API_VERSION)
        endpoint = f'{api_version}/filter_identities'
        params = self.get_default_params(initial_request)

        return self._request('GET', endpoint=endpoint, headers=headers, params=params)

    def is_connection_available(self, headers) -> bool:
        """
        Make a lightweight HEAD request to the most lightweight VH hook 'player'
        If no `ConnectionError` raised - connection is available

        Required from distinguishing a real connection error, from VH's broken connection when made a request with
        unknown param(to any hook).
        It's better for VH to respond 404 in this case, but it just breaks connection in this case

        So this methods, helps to check is connection really isn't available, or it's VH's 404 alternative response
        """
        try:
            self.content_detail(
                initial_request=None,
                headers=headers,
                content_id=GOOD_CONTENT_ID,
                method='HEAD'
            )
            return True
        except ConnectionError:
            return False


class ChannelIdFaker(FrontendVHApi):
    """
    Обертка вокруг апи VH для подмены channel_id тв-каналов
    see: SMARTTVBACKEND-1001
    """
    def __init__(self, offset, **kwargs):
        self.offset = offset
        super().__init__(**kwargs)

    def channels_regions(self, initial_request, headers, region_id=None, project_alias=None,
                         with_filtration=1):
        result = super().channels_regions(
            initial_request,
            headers,
            region_id=region_id,
            project_alias=project_alias,
            with_filtration=with_filtration
        )

        replaced = 0
        for item in result['set']:
            if item.get('channel_id'):
                item['channel_id'] = self._map_channel_id(item['channel_id'])
                replaced += 1

        logger.info('Replaced %d channel ids', replaced)
        return result

    def episodes(self, initial_request, headers, from_time=None, to_time=None, **kwargs):
        result = super().episodes(initial_request, headers, from_time=from_time, to_time=to_time, **kwargs)

        replaced = 0
        for episode in result['set']:
            if episode.get('channel_id'):
                episode['channel_id'] = self._map_channel_id(episode['channel_id'])
                replaced += 1

        logger.info('Replaced %d channel ids in episodes', replaced)

        return result

    def _map_channel_id(self, channel_id):
        if channel_id and isinstance(channel_id, int):
            return channel_id + self.offset

        return channel_id


retry_methods = frozenset(('GET',))
vh_retry_http_codes = frozenset((408, 500, 502, 504))
vh_retry = Retry(total=settings.FRONTENDVH_DEFAULT_RETRIES,
                 read=settings.FRONTENDVH_DEFAULT_RETRIES,
                 connect=settings.FRONTENDVH_DEFAULT_RETRIES,
                 status=settings.FRONTENDVH_DEFAULT_RETRIES,
                 status_forcelist=vh_retry_http_codes,
                 method_whitelist=retry_methods,
                 raise_on_status=False,
                 backoff_factor=0)
client = FrontendVHApi(
    url=settings.FRONTENDVH_API_URL,
    timeout=settings.FRONTENDVH_DEFAULT_TIMEOUT,
    retries=vh_retry
)

if settings.VH_CHANNEL_ID_OFFSET:
    logger.info('VH channel number offset is %d', settings.VH_CHANNEL_ID_OFFSET)
    channel_client = ChannelIdFaker(
        offset=settings.VH_CHANNEL_ID_OFFSET,
        url=settings.FRONTENDVH_API_URL,
        timeout=settings.FRONTENDVH_DEFAULT_TIMEOUT,
        retries=vh_retry
    )
else:
    channel_client = client
