import logging

from django.conf import settings

from smarttv.droideka.proxy.api import base
from smarttv.droideka.utils.default_values import DEFAULT_DICT
from smarttv.droideka.proxy.vh.constants import FIELD_LEGAL, FIELD_VH_LICENSES, FIELD_UUID

logger = logging.getLogger(__name__)


class VideoSearchExtendedLogger:
    KEY_CLIPS = 'clips'
    KEY_URL = 'url'
    KEY_ENTITY_DATA = 'entity_data'
    KEY_RELATED_OBJECT = 'related_object'
    KEY_RELATED_OBJECT_TYPE = 'type'
    KEY_SEARCH_REQUEST = 'search_request'
    KEY_ENTREF = 'entref'
    KEY_OBJECT = 'object'
    KEY_NAME = 'name'
    KEY_ID = 'id'
    KEY_LEGAL_ASSOC = 'legal_assoc'
    KEY_BASE_INFO = 'base_info'
    KEY_TITLE = 'title'

    RELATED_OBJ_TYPE_COLLECTIONS = 'collections'
    RELATED_OBJ_TYPE_ASSOC = 'assoc'

    @classmethod
    def _get_clips_content(cls, response):
        clips = response.get(cls.KEY_CLIPS)
        if not clips:
            return None
        return [clip[cls.KEY_URL] for clip in clips if clip.get(cls.KEY_URL)]

    @classmethod
    def _get_collections_content(cls, collections):
        if not collections:
            return None
        result = {}
        search_request = collections.get(cls.KEY_SEARCH_REQUEST)
        if search_request is not None:
            result[cls.KEY_SEARCH_REQUEST] = search_request
        ent_ref = collections.get(cls.KEY_ENTREF)
        if ent_ref is not None:
            result[cls.KEY_ENTREF] = ent_ref
        return result

    @classmethod
    def _get_assoc_content(cls, assoc):
        if not assoc:
            return None
        result = {}
        search_request = assoc.get(cls.KEY_SEARCH_REQUEST)
        if search_request:
            result[cls.KEY_SEARCH_REQUEST] = search_request
        if cls.KEY_OBJECT not in assoc:
            return result
        assoc_objects = assoc[cls.KEY_OBJECT]
        if not assoc_objects:
            return result
        objects = []
        for item in assoc_objects:
            name = item.get(cls.KEY_NAME)
            id = item.get(cls.KEY_ID)
            if name or id:
                log_item = {}
                if name:
                    log_item[cls.KEY_NAME] = name
                if id:
                    log_item[cls.KEY_ID] = id
                if log_item:
                    objects.append(log_item)
        result[cls.KEY_OBJECT] = objects
        return result

    @classmethod
    def _get_legal_assoc_content(cls, entity_data):
        legal_assoc_list = entity_data.get(cls.KEY_LEGAL_ASSOC)
        if not legal_assoc_list:
            return None
        if len(legal_assoc_list) > 1:
            logger.warning('Legal assoc has more than 1 item')
        return cls._get_assoc_content(legal_assoc_list[0])

    @classmethod
    def _get_collections_and_assoc(cls, related_object):
        assoc = None
        collections = None
        for obj in related_object:
            if obj[cls.KEY_RELATED_OBJECT_TYPE] == cls.RELATED_OBJ_TYPE_COLLECTIONS:
                collections = obj
            elif obj[cls.KEY_RELATED_OBJECT_TYPE] == cls.RELATED_OBJ_TYPE_ASSOC:
                assoc = obj
            if assoc and collections:
                break
        return cls._get_collections_content(collections), cls._get_assoc_content(assoc)

    @classmethod
    def _get_base_info_content(cls, entity_data):
        if not entity_data:
            return None
        base_info = entity_data.get(cls.KEY_BASE_INFO)
        if not base_info:
            return None
        uuid = entity_data.get(
            cls.KEY_BASE_INFO, DEFAULT_DICT
        ).get(
            FIELD_LEGAL, DEFAULT_DICT
        ).get(
            FIELD_VH_LICENSES, DEFAULT_DICT
        ).get(
            FIELD_UUID
        )
        title = entity_data.get(
            cls.KEY_BASE_INFO, DEFAULT_DICT
        ).get(
            cls.KEY_TITLE
        )
        onto_id = entity_data.get(
            cls.KEY_BASE_INFO, DEFAULT_DICT
        ).get(
            cls.KEY_ID
        )
        result = {}
        if uuid:
            result[FIELD_UUID] = uuid
        if title:
            result[cls.KEY_TITLE] = title
        if onto_id:
            result[cls.KEY_ID] = onto_id
        return result

    @classmethod
    def get_log_content(cls, response):
        clips_content = cls._get_clips_content(response)
        entity_data = response.get(cls.KEY_ENTITY_DATA, DEFAULT_DICT)
        if entity_data:
            related_object = entity_data.get(cls.KEY_RELATED_OBJECT)
            if related_object:
                collections, assoc = cls._get_collections_and_assoc(related_object)
            else:
                collections, assoc = None, None
            legal_assoc_content = cls._get_legal_assoc_content(entity_data)
            base_info = cls._get_base_info_content(entity_data)
        else:
            base_info = None
            legal_assoc_content = None
            collections = None
            assoc = None
        result = {}
        if base_info:
            result[cls.KEY_BASE_INFO] = base_info
        if legal_assoc_content:
            result[cls.KEY_LEGAL_ASSOC] = legal_assoc_content
        if clips_content:
            result[cls.KEY_CLIPS] = clips_content
        if collections:
            result['collections'] = collections
        if assoc:
            result['assoc'] = assoc
        return result

    @classmethod
    def log_content(cls, response):
        if not settings.LOG_CONTENT_ID or not response:
            return None
        log_content = cls.get_log_content(response)
        logger.info(log_content)


class VideoSearchApi(base.BaseJsonApi):
    unistat_suffix = 'search'

    def update_params(self, params):
        params.update({
            'rearr': [
                'scheme_Local/VideoExtraItems/EntityDataForDevices=1',
                'forced_player_device=tvandroid',
                'scheme_Local/VideoPlayers/AddDevice=1',
                'scheme_Local/FilterBannedVideo/SvodForAll=0',
                'scheme_Local/FilterBannedVideo/SvodForYaPlusUsers=1',
            ],
            'gta': [
                'onto_id',
                'vh_uuid',
            ],
        })
        need_people = params.pop('need_people')
        if not need_people:
            params['rearr'].append('scheme_Local/VideoExtraItems/FilterUnusedObjectsForDevices=1')
        return params

    def search(self, headers=None, params=None):
        response = self._request(headers=headers, params=params)
        VideoSearchExtendedLogger.log_content(response)
        return response


class SuggestHistoryApi(base.BaseJsonApi):
    unistat_suffix = 'suggest'

    def update_params(self, params):
        params.update({
            'srv': 'smart_tv',
        })
        return params

    def suggest_history(self, headers=None, params=None):
        return self._request(headers=headers, params=params)


client = VideoSearchApi(
    url=settings.VIDEO_SEARCH_API_URL,
    timeout=settings.VIDEO_SEARCH_DEFAULT_TIMEOUT,
    retries=settings.VIDEO_SEARCH_DEFAULT_RETRIES,
)

history = SuggestHistoryApi(
    url=settings.SUGGEST_HISTORY_API_URL,
    timeout=settings.SUGGEST_HISTORY_DEFAULT_TIMEOUT,
    retries=settings.SUGGEST_HISTORY_DEFAULT_RETRIES,
)
