from typing import Optional, Collection, Pattern, List
import json
import re
import logging

from django.conf import settings

from smarttv.droideka.utils import PlatformInfo
from yaphone.utils.parsed_version import ParsedVersion
from smarttv.droideka.proxy.common import fix_schema
from smarttv.droideka.proxy.constants.home_app_versions import VER_1_4
from alice.protos.data.video.video_pb2 import TAvatarMdsImage
from smarttv.droideka.proxy.constants.orientation import Orientation


GENERATED_IMAGES_MINIMUM_VERSION = VER_1_4


logger = logging.getLogger(__name__)


config_aliases = {
    'vh': 'get-vh',
    'es': 'get-entity_search',
    'ott': 'get-ott',
    'music': 'music',
    'hotels': 'hotels',
    'dialogs': 'get-dialogs',
    'cover': 'cover',
}

suffix_config_aliases = {
    Orientation.VERTICAL.value: Orientation.VERTICAL.value,
    Orientation.HORIZONTAL.value: Orientation.HORIZONTAL.value,
}

known_buckets = frozenset(config_aliases.values())

AVATARS_MDS_URL_PATTERN = r'(?P<base_url>https?://avatars.mds.yandex.net/(?P<bucket>({}))/.*/.*/)'.format(
    '|'.join(known_buckets))
AVATARS_MDS_REGEX = re.compile(AVATARS_MDS_URL_PATTERN)

GENERAL_AVATARS_MDS_REGEX: Pattern[str] = \
    re.compile(r'(?P<base_url>(https?:)?//avatars.mds.yandex.net(:(80|443))?/[a-zA-Z_\-]+/.+/.+/)')


class Resolution:
    def __init__(self, is_original=False, width=None, height=None):
        self.is_original = is_original
        if not is_original:
            self.width = width
            self.height = height

    def as_string(self):
        if self.is_original:
            return 'orig'
        else:
            return f'{self.width}x{self.height}'


class Config:
    def __init__(self, name: str, resolutions: Optional[Collection[Resolution]]):
        self.name = name
        self.resolutions = resolutions


class ImageGenerator:
    def __init__(self, config):
        self.config = config

    def generate_url(self, original_url: str) -> dict:
        result = {}
        for resolution in self.config.resolutions:
            res_str = resolution.as_string()
            result[res_str] = original_url + res_str
        return result

    def generate_url_from_pattern(self, pattern_url: str) -> dict:
        result = {}
        for resolution in self.config.resolutions:
            res_str = resolution.as_string()
            result[res_str] = fix_schema(pattern_url.replace('%%', res_str))
        return result

    def generate_url_sizes(self) -> List[str]:
        result = []
        for resolution in self.config.resolutions:
            res_str = resolution.as_string()
            result.append(res_str)
        return result


def parse_configs(raw_configs: str) -> list:
    if not raw_configs:
        return []
    configs = json.loads(raw_configs)
    result = []
    for config_name, raw_resolutions in configs.items():
        result.append(Config(
            config_name,
            [Resolution(raw.get('orig', False), raw.get('w'), raw.get('h')) for raw in raw_resolutions]
        ))
    return result


def get_generators(setting: str, aliases: dict) -> dict:
    result = {}
    for config in parse_configs(setting):
        if config.name in aliases:
            result[aliases[config.name]] = ImageGenerator(config)
    return result


def default_image_generators():
    return get_generators(settings.AVATARS_MDS_IMAGE_GENERATION_CONFIGS, config_aliases)


def get_sizes_generators() -> dict:
    return get_generators(settings.AVATARS_MDS_SUFFIX_IMAGE_GENERATION_CONFIGS, suffix_config_aliases)


def remove_mds_avatar_size(mds_avatar_url: Optional[str]) -> Optional[str]:
    if not mds_avatar_url:
        return None
    match = GENERAL_AVATARS_MDS_REGEX.match(mds_avatar_url)
    if not match:
        return None
    container = match.groupdict()
    base_url = container['base_url']
    if base_url.endswith('/'):
        base_url = base_url.rstrip('/')
    return base_url


generators = default_image_generators()
sizes_generator = get_sizes_generators()


def get_url_info(orig_url: Optional[str], platform_info: PlatformInfo, matcher: Pattern[str]) -> Optional[dict]:
    if not platform_info or ParsedVersion(platform_info.app_version) < GENERATED_IMAGES_MINIMUM_VERSION:
        return None
    if not generators or not orig_url:
        return None
    match = matcher.match(orig_url)
    if not match:
        # TODO(alex-garmash): sometimes urls not from avatars.mds are appearing(wikipedia / kinopoisk.ru / prdisk.ru / etc.), what to do with them?
        logger.warning('Can not process url: %s', orig_url)
        return None
    return match.groupdict()


def fill_images_from_url(orig_url: Optional[str], target_name: str, destination: dict, platform_info: PlatformInfo, sizes_key: str = None):
    if not sizes_key:
        url_info = get_url_info(orig_url, platform_info, AVATARS_MDS_REGEX)
        if not url_info:
            return None
        generator = generators[url_info['bucket']]
        destination[target_name] = generator.generate_url(url_info['base_url'])
    else:
        url_info = get_url_info(orig_url, platform_info, GENERAL_AVATARS_MDS_REGEX)
        if not url_info:
            return None
        generator = generators[sizes_key]
        destination[target_name] = generator.generate_url(fix_schema(url_info['base_url']))


def fill_images_from_id(avatar_id: Optional[str], bucket: str, host: str, target_name: str, destination: dict,
                        platform_info: PlatformInfo):
    if not platform_info or ParsedVersion(platform_info.app_version) < GENERATED_IMAGES_MINIMUM_VERSION:
        return
    if not generators or not avatar_id or bucket not in generators:
        return
    generator = generators[bucket]
    destination[target_name] = generator.generate_url(f'https://{host}/{bucket}/{avatar_id}/')


def fill_images_from_pattern(pattern_url: str, target_name: str, destination: dict, config_key: str):
    """
    Build thumbs list from pattern like http://host/bucket/%%, replacing %% by image size
    """
    if not generators or not pattern_url:
        return

    generator = generators[config_key]
    destination[target_name] = generator.generate_url_from_pattern(pattern_url)


def get_proto_images_from_url(
        orig_url: str, platform_info: PlatformInfo, orientation: Orientation, result: TAvatarMdsImage):
    url_info = get_url_info(orig_url, platform_info, GENERAL_AVATARS_MDS_REGEX)
    if not url_info:
        return None
    result.BaseUrl = url_info['base_url']
    result.Sizes.extend(sizes_generator[orientation.value].generate_url_sizes())
