import logging
from abc import ABC, abstractmethod
from typing import Iterable
from enum import Enum

from django.conf import settings

from blackbox import JsonBlackbox
from smarttv.utils import headers
from yaphone.utils.django import get_http_header, get_user_ip
from yaphone.utils.parsers import parsers
from yaphone.utils.parsed_version import ParsedVersion

from smarttv.droideka.utils.default_values import DEFAULT_DICT
from smarttv.droideka.proxy import tvm

logger = logging.getLogger(__name__)

blackbox = JsonBlackbox(blackbox_client=settings.BLACKBOX_CLIENT)


class AuthentificationParseError(Exception):
    pass


class DbFields:
    have_plus = 1015
    kinopoisk_ott_subscription_name = 1016

    value_to_name = {
        have_plus: 'account.have_plus',
        kinopoisk_ott_subscription_name: 'account.kinopoisk_ott_subscription_name'
    }

    @classmethod
    def get_name(cls, attr: int):
        return cls.value_to_name.get(attr)


plus_fields = (DbFields.have_plus, DbFields.kinopoisk_ott_subscription_name)


class SubscriptionType(Enum):
    """
    See https://a.yandex-team.ru/arc/trunk/arcadia/media-billing/mediabilling-core/src/main/java/ru/yandex/music/support/billing/plus/SubscriptionOttType.java#L6
    """
    YA_PLUS = 'YA_PLUS'
    YA_PLUS_3M = 'YA_PLUS_3M'
    YA_PLUS_KP = 'YA_PLUS_KP'
    KP_BASIC = 'KP_BASIC'
    YA_PLUS_SUPER = 'YA_PLUS_SUPER'
    YA_PREMIUM = 'YA_PREMIUM'

    UNKNOWN = None


class UserInfo:
    def __init__(self, passport_uid: str = None, attributes: dict = None, user_ticket: str = None):
        self.passport_uid = passport_uid
        self.attributes = attributes
        self.user_ticket = user_ticket

    @property
    def subscription(self) -> SubscriptionType:
        if self.attributes and self.attributes.get(DbFields.have_plus):
            raw_subscription_type = self.attributes.get(DbFields.kinopoisk_ott_subscription_name)
            try:
                return SubscriptionType(raw_subscription_type)
            except ValueError:
                logger.warning('Unknown subscription type: %s', raw_subscription_type)
        return SubscriptionType.UNKNOWN


class InvalidTokenError(Exception):
    pass


class BlackBoxClient(ABC):
    RESPONSE_ERROR_KEY = 'error'
    RESPONSE_STATUS_KEY = 'status'
    INVALID_OAUTH_STATUS = 5

    def __init__(self, request):
        self.user_ip = get_user_ip(request)

    @abstractmethod
    def bb_call(self, attributes: str, headers: dict) -> dict:
        pass

    @staticmethod
    def parse_user_agent(request):
        user_agent = get_http_header(request, headers.USER_AGENT_HEADER)
        if user_agent:
            return parsers.UaParser.parse(user_agent)
        return DEFAULT_DICT

    def ignore_expired_token(self):
        return True

    # noinspection PyMethodMayBeStatic
    def as_multivalues_param(self, attrs: Iterable):
        return ','.join(str(item) for item in attrs)

    def get_bb_response(self, requested_attrs: Iterable[int]) -> dict:
        headers = tvm.add_service_ticket(settings.BLACKBOX_CLIENT_ID)
        return self.bb_call(self.as_multivalues_param(requested_attrs), headers)

    def get_user_info(self, requested_attrs: Iterable[int] = None, request=None) -> UserInfo:
        requested_attrs = requested_attrs or ()

        response = self.get_bb_response(requested_attrs)

        if response[self.RESPONSE_STATUS_KEY].get('id') == self.INVALID_OAUTH_STATUS:
            # Error bb answer example:
            # {
            #     'error': 'expired_token',
            #     'status': {
            #         'id': 5,
            #         'value': 'INVALID'
            #     }
            # }
            logger.info('OAuth token is invalid, reason: %s', response[self.RESPONSE_ERROR_KEY])
            raise InvalidTokenError()

        try:
            uid = response['uid']['value']
            user_ticket = response['user_ticket']
            attrs = {}
            if requested_attrs and 'attributes' in response:
                attributes = response['attributes']
                for item in requested_attrs:
                    if str(item) in attributes:
                        attrs[item] = attributes[str(item)]
            return UserInfo(uid, attrs, user_ticket)
        except KeyError as ex:
            logger.warning('Error "%s" on blackbox response: %s', ex, response)
            return UserInfo()


class AndroidBlackBoxClient(BlackBoxClient):
    DEFAULT_VERSION = ParsedVersion('1.1')
    VERSION_WITH_EXPIRED_TOKEN_SUPPORT = ParsedVersion('1.2')

    def __init__(self, request):
        self.token = self.get_oauth_token(request)
        self.user_agent = self.parse_user_agent(request)
        super().__init__(request)

    def get_client_version(self):
        app_version_string = self.user_agent.get('app_version_string')
        app_name = self.user_agent.get('app_name', '')
        if (
            (app_version_string is None) or
            ('.' not in app_version_string) or
            ('Dalvik' in app_name)
        ):
            return self.DEFAULT_VERSION
        else:
            return ParsedVersion(app_version_string)

    def ignore_expired_token(self):
        return self.get_client_version() < self.VERSION_WITH_EXPIRED_TOKEN_SUPPORT

    @staticmethod
    def parse_authentification(header_value):
        auth = header_value.split()
        if len(auth) != 2:
            raise AuthentificationParseError(f'Incorrect authentification'
                                             f' header value - {header_value}')

        # noinspection PyShadowingBuiltins
        type, credentials = auth
        if type.lower() != 'oauth':
            raise AuthentificationParseError(f'Authentification type != oauth not supported'
                                             f' on header {header_value}')

        return credentials

    def get_oauth_token(self, request):
        value = get_http_header(request, headers.AUTHORIZATION_HEADER)

        if value:
            try:
                return self.parse_authentification(value)
            except AuthentificationParseError as ex:
                logger.warning(str(ex))
        return None

    def bb_call(self, attributes: str, headers: dict) -> dict:
        return blackbox.oauth(
            userip=self.user_ip,
            oauth_token=self.token,
            headers=headers,
            attributes=attributes,
            get_user_ticket='yes',
        )


class TvmBlackboxClient(BlackBoxClient):

    def __init__(self, request):
        self.validate_service_ticket(request)
        self.user_agent = self.parse_user_agent(request)
        self.uid = self.get_uid(request)
        super().__init__(request)

    def bb_call(self, attributes: str, headers: dict) -> dict:
        return blackbox.userinfo(
            userip=self.user_ip,
            uid=self.uid,
            headers=headers,
            attributes=attributes
        )

    def get_user_info(self, requested_attrs: Iterable[int] = None, request=None) -> UserInfo:
        user_ticket = get_http_header(request, headers.TVM_USER_TICKET) if request else None
        return UserInfo(passport_uid=str(self.uid), attributes={}, user_ticket=user_ticket)

    @staticmethod
    def validate_service_ticket(request):
        ticket = get_http_header(request, headers.TVM_SERVICE_TICKET)
        tvm.check_service_ticket(ticket)

    @staticmethod
    def get_uid(request) -> int:
        ticket = get_http_header(request, headers.TVM_USER_TICKET)
        return tvm.get_uid_from_user_ticket(ticket)
