import logging

from abc import abstractmethod, ABC
from typing import Optional, Iterable

from cache_memoize import cache_memoize
from django.conf import settings

from smarttv.droideka.proxy import cache
from smarttv.droideka.utils import PlatformInfo, RequestInfo
from smarttv.droideka.proxy.api.vh import experiment_errors_counter
from smarttv.droideka.proxy.models import PlatformType, Category2, Category2Editable
from smarttv.droideka.unistat.metrics import CacheType, LocalCachePlace

logger = logging.getLogger(__name__)


KEY_CATEGORY_ID = 'category_id'
KEY_TITLE = 'title'
KEY_ICON_S3_KEY = 'icon_s3_key'
KEY_RANK = 'rank'
KEY_POSITION = 'position'
KEY_PARENT_CATEGORY_ID = 'parent_category_id'
KEY_PARENT_CATEGORY = 'parent_category'
KEY_THUMBNAIL_S3_KEY = 'thumbnail_s3_key'
KEY_LOGO_S3_KEY = 'logo_s3_key'
KEY_BANNER_S3_KEY = 'banner_S3_key'
KEY_DESCRIPTION = 'description'
KEY_CONTENT_TYPE = 'content_type'
KEY_CAROUSEL_TYPE = 'carousel_type'
KEY_EXCLUDE_PLATFORMS = 'exclude_platforms'
KEY_INCLUDE_PLATFORMS = 'include_platforms'
KEY_ABOVE_PLATFORMS = 'above_platforms'
KEY_BELOW_PLATFORMS = 'below_platforms'
KEY_AUTHORIZATION_REQUIRED = 'authorization_required'
KEY_SHOW_IN_TANDEM = 'show_in_tandem'
KEY_CATEGORY_EXPERIMENTS = 'category_experiments'
KEY_PERSISTENT_CLIENT_CATEGORY_ID = 'persistent_client_category_id'
GENERAL_CATEGORY_INFO = (KEY_CATEGORY_ID,
                         KEY_TITLE,
                         KEY_ICON_S3_KEY,
                         KEY_RANK,
                         KEY_POSITION,
                         KEY_THUMBNAIL_S3_KEY,
                         KEY_LOGO_S3_KEY,
                         KEY_BANNER_S3_KEY,
                         KEY_DESCRIPTION,
                         KEY_CONTENT_TYPE,
                         KEY_CAROUSEL_TYPE,
                         KEY_PERSISTENT_CLIENT_CATEGORY_ID)


class StorageNotAvailable(Exception):
    pass


def is_platforms_specified(category: dict, platforms_key: str):
    specified_platforms = category[platforms_key]
    return not (len(specified_platforms) == 1 and specified_platforms[0].platform_type == PlatformType.NONE)


class BasePlatformFilter(ABC):
    @property
    @abstractmethod
    def platforms_key(self) -> str:
        pass

    @abstractmethod
    def is_item_matched(self, platform_info: PlatformInfo, candidate_platform: PlatformInfo) -> bool:
        pass

    def is_matched(self, category: dict, platform_info: PlatformInfo) -> bool:
        for platform in category[self.platforms_key]:
            if self.is_item_matched(platform_info=platform_info,
                                    candidate_platform=platform):
                return True
        return False


class IncludeFilter(BasePlatformFilter):
    @property
    def platforms_key(self):
        return KEY_INCLUDE_PLATFORMS

    def is_item_matched(self, platform_info: PlatformInfo, candidate_platform: PlatformInfo) -> bool:
        return platform_info in candidate_platform


class ExcludeFilter(BasePlatformFilter):
    @property
    def platforms_key(self):
        return KEY_EXCLUDE_PLATFORMS

    def is_item_matched(self, platform_info: PlatformInfo, candidate_platform: PlatformInfo) -> bool:
        return platform_info in candidate_platform


class AboveFilter(BasePlatformFilter):

    @property
    def platforms_key(self) -> str:
        return KEY_ABOVE_PLATFORMS

    def is_item_matched(self, platform_info: PlatformInfo, candidate_platform: PlatformInfo) -> bool:
        try:
            return platform_info >= candidate_platform
        except PlatformInfo.NotComparable:
            return False


class BelowFilter(BasePlatformFilter):

    @property
    def platforms_key(self):
        return KEY_BELOW_PLATFORMS

    def is_item_matched(self, platform_info: PlatformInfo, candidate_platform: PlatformInfo) -> bool:
        try:
            return platform_info <= candidate_platform
        except PlatformInfo.NotComparable:
            return False


class CategoriesCache:

    @cache_memoize(timeout=settings.DEFAULT_RESPONSE_CACHE_TIME,
                   cache_alias='local',
                   **cache.get_cache_memoize_callables(LocalCachePlace.RAW_CATEGORIES.value, CacheType.LOCAL))
    def _get_categories(self):
        return [category.to_json() for category in Category2.objects.using(settings.DB_REPLICA).all()]

    @cache_memoize(
        timeout=settings.DEFAULT_RESPONSE_CACHE_TIME,
        cache_alias='local',
        **cache.get_cache_memoize_callables(
            LocalCachePlace.CATEGORIES_NON_PUBLISHED.value, CacheType.LOCAL
        )
    )
    def _get_non_published_categories(self):
        result = {}
        for category in Category2Editable.objects.using(settings.DB_REPLICA).filter(
            visible=False
        ):
            # возвращает легкий набор данных, используемых только в одном кейсе
            result[category.category_id] = {
                'title': category.title,
                'content_type': category.content_type
            }

        return result

    def patch_categories_platform_info(self, categories):
        for category in categories:
            for platform_key in (
                    KEY_ABOVE_PLATFORMS, KEY_BELOW_PLATFORMS, KEY_INCLUDE_PLATFORMS, KEY_EXCLUDE_PLATFORMS):
                serialized_platforms = []
                for platform in category[platform_key]:
                    serialized_platforms.append(PlatformInfo(
                        platform.get(PlatformType.KEY),
                        platform.get(PlatformInfo.KEY_PLATFORM_VERSION),
                        platform.get(PlatformInfo.KEY_APP_VERSION),
                        platform.get(PlatformInfo.KEY_DEVICE_MANUFACTURER),
                        platform.get(PlatformInfo.KEY_DEVICE_MODEL),
                        platform.get(PlatformInfo.KEY_QUASAR_PLATFORM),
                    ))
                category[platform_key] = serialized_platforms

    def get_categories(self):
        try:
            categories = self._get_categories()
        except Exception:
            categories = None
            logger.exception('Can not obtain categories')
        if categories is None:
            raise StorageNotAvailable()
        self.patch_categories_platform_info(categories)
        return categories

    def get_non_published_categories(self) -> dict:
        return self._get_non_published_categories()


class CategoriesProvider:
    include_filter = IncludeFilter()
    exclude_filter = ExcludeFilter()
    above_filter = AboveFilter()
    below_filter = BelowFilter()

    def __init__(self, categories_cache: CategoriesCache):
        self.categories_cache = categories_cache

    def _get_categories_with_specified_parent_id(self, categories: list, parent_category_id: str) -> list:
        return [category for category in categories if category[KEY_PARENT_CATEGORY_ID] == parent_category_id]

    def _is_category_in_range(self, category: dict, platform_info: PlatformInfo):
        is_above_platforms_specified = is_platforms_specified(category, self.above_filter.platforms_key)
        is_below_platforms_specified = is_platforms_specified(category, self.below_filter.platforms_key)
        below_matched = is_below_platforms_specified and self.below_filter.is_matched(category, platform_info)
        above_matched = is_above_platforms_specified and self.above_filter.is_matched(category, platform_info)
        if is_above_platforms_specified and is_below_platforms_specified:
            return below_matched and above_matched
        return below_matched or above_matched

    def _is_matched_by_platform_info(self, category: dict, platform_info: PlatformInfo) -> bool:
        # Check if category is excluded. I.e. it has platform filter(exclude_platforms) which matches
        # client's `platform_info`
        is_not_excluded = not self.exclude_filter.is_matched(category, platform_info)
        # Check if category is included in any of platform, specified in `include_platforms`.
        # For example, Platform(type=android) includes Platform(type=android, app_version=1.2)
        # For more examples, loot at the 'TestIncludedPlatformsQueryBuilder' in 'test_categories_provider'
        is_included = self.include_filter.is_matched(category, platform_info)
        # Check is category is in range. I.e. if specified only above filter, it looks like:
        #         ///////////
        # _______.___________
        #        x
        # in this case, category in range, if client's platform info, is at right of x, of any above filter in specified
        # category
        # Same for the case, when only below filter is specified, but client's platform info should be at left of x
        # If both filters specified, for example below x and above y - then client's platform info must be between x
        # and y:
        #         /////////
        # _______.________.______
        #        x        y
        is_in_range = self._is_category_in_range(category, platform_info)

        return is_not_excluded and (is_included or is_in_range)

    def _has_intersection(self, experiments1: Optional[set], experiments2: Optional[set]):
        return experiments1 and experiments2 and experiments1.intersection(experiments2)

    def _is_matched_by_experiments(
            self, category_experiments: Optional[set], category_disable_experiments: Optional[set],
            experiments: Optional[set]) -> bool:
        if not category_experiments and not category_disable_experiments:
            return True
        is_disabled_by_experiment = bool(self._has_intersection(experiments, category_disable_experiments))
        is_enabled_by_experiment = bool(self._has_intersection(experiments, category_experiments))
        return not is_disabled_by_experiment and (not category_experiments or is_enabled_by_experiment)

    def _get_request_experiments(self, raw_exp) -> Optional[list]:
        """
        raw_exp may be a single value or iterable, since AB backend can merge values from different experiments
        if only 1 experiment contains 'category_experiments' flag, then it will be a single value
        if 2 or more experiments contains this flag, then values will be merged to list

        This functions handles this situation and any value to array
        """
        if not raw_exp:
            return None
        result = []
        if isinstance(raw_exp, Iterable):
            result.extend(raw_exp)
        else:
            logger.error('Unknown type of experiments')
        return result

    def _is_matched_by_request_info(self, category: dict, request_info: RequestInfo):
        authorization_valid = not category[KEY_AUTHORIZATION_REQUIRED] or request_info.authorized
        try:
            category_experiments = request_info.experiments.get_value(KEY_CATEGORY_EXPERIMENTS)
        except:
            experiment_errors_counter.increment()
            category_experiments = []

        request_exp = self._get_request_experiments(category_experiments)
        experiments_matched = self._is_matched_by_experiments(
            set(category[Category2.KEY_CATEGORY_EXPERIMENTS] or []),
            set(category[Category2.KEY_CATEGORY_DISABLE_EXPERIMENTS] or []),
            set(request_exp or []))

        # you can disable category for module in tandem mode
        # (ex: exclude music carousel on the home screen)
        show_in_tandem = category[KEY_SHOW_IN_TANDEM]
        if not show_in_tandem and request_info.is_tandem:
            logger.debug('Skipping category %s because we are in tandem mode and show_in_tandem is disabled',
                         category['category_id'])
            return False

        return authorization_valid and experiments_matched

    def _is_matched(self, category: dict, platform_info: PlatformInfo, request_info: RequestInfo):
        return all((self._is_matched_by_platform_info(category, platform_info),
                    self._is_matched_by_request_info(category, request_info)))

    def _get_matching_categories(self, categories: list, platform_info: PlatformInfo, request_info: RequestInfo):
        return [categ for categ in categories if self._is_matched(categ, platform_info, request_info)]

    def _convert_to_extended_category(self, category_serializable: dict) -> Category2:
        general_info = {attr: value for attr, value in category_serializable.items() if attr in GENERAL_CATEGORY_INFO}
        return Category2(**general_info)

    def get_categories(self,
                       platform_info: PlatformInfo,
                       parent_category_id: str = None,
                       request_info: RequestInfo = None):
        """
        Returns categories for specified platform_info
        """
        categories = self.categories_cache.get_categories()
        categories = self._get_categories_with_specified_parent_id(categories, parent_category_id)
        categories = self._get_matching_categories(categories, platform_info, request_info)
        return [self._convert_to_extended_category(category) for category in categories]

    def get_category(self, category_id: str) -> Optional[dict]:
        categories = self.categories_cache.get_categories()
        for category in categories:
            if category['category_id'] == category_id:
                return category
        return None

    def get_non_published_category(self, category_id: str) -> Optional[dict]:
        return self.categories_cache.get_non_published_categories().get(category_id)


categories_cache = CategoriesCache()
categories_provider = CategoriesProvider(categories_cache)
