import re
from geobase6 import Lookup
from typing import Optional, Iterable

from furl import has_netloc
from rest_framework.exceptions import NotFound

ONTO_ID_PATTERN = re.compile(r'^[a-z]{3,}[0-9]+$')

geobase: Optional[Lookup] = None


def fix_schema(url: Optional[str]) -> Optional[str]:
    if url and url.startswith('//'):
        return 'https:' + url
    elif url and not has_netloc(url):
        # something like 'avatars.ya.ru/xx/yy'
        return 'https://' + url

    return url


def fix_avatars_mds_suffix(url: Optional[str]) -> Optional[str]:
    if url is not None and url.endswith('/'):
        url = url + 'orig'
    return url


def join_string_items(genres: Optional[Iterable[str]]) -> Optional[str]:
    if genres:
        return ', '.join(genres)
    return None


def is_onto_id(candidate: str) -> bool:
    """
    Chcks candidate for matching onto_id pattern

    Examples of onto_id's:
    - ruw123456
    - yam12345678
    - kin98874562
    :param candidate: string id which should be checked for matching onto_id pattern
    :return: True, if candidate has onto_id pattern. Otherwise - False
    """
    return bool(ONTO_ID_PATTERN.match(candidate))


def ensure_ott_content_id(candidate_content_id: str):
    """
    Raises 404 if candidate_content_id matches the onto_id pattern

    Needed for cases, when onto_id has passed as OTT's `content_id`. Because, anyway it's impossible to get data
    by `onto_id` from sources which expect `content_id` - it's better to raise 404 before sending this id to data
    sources.

    If it won't be filtered, anyway 404 will be received, but from external service

    See SMARTTVBACKEND-989,
    :param candidate_content_id: `content_id` to be checked for `onto_id` pattern
    :raises:
        NotFound: if `candidate_content_id` matches the pattern of `onto_id`
    :return: Does nothing, if not matches the `onto_id` pattern
    """
    if is_onto_id(candidate_content_id):
        raise NotFound(f"No document with id '{candidate_content_id}' found")
