import logging
import random
from abc import ABC, abstractmethod
from enum import Enum
from functools import partial
from typing import Optional, Any, Callable, List
from copy import deepcopy
from collections import deque
import datetime

import gevent
from cache_memoize import cache_memoize
from django.conf import settings
from django.http import Http404
from rest_framework.exceptions import NotFound, AuthenticationFailed
from requests.exceptions import ConnectionError, HTTPError

from smarttv.droideka.proxy import api, cache
from smarttv.droideka.proxy.categories_provider import categories_provider
from smarttv.droideka.proxy.constants.carousels import CarouselsExternal, MusicCarousel, VhFeed, VhCarousel, KpCarousel, \
    KpMultiSelection, DroidekaCarousel, DroidekaCategory, FILTERABLE_CAROUSEL_ID, \
    TAG_BY_CAROUSEL_MAPPING, EMBEDDED_SECTION_TYPE, DROIDEKA_INTERNAL_OTT_WINDOW_ID, PURCHASES_CAROUSEL_ID, \
    PURCHASES_FAKE_SELECTION_ID, CarouselType
from smarttv.droideka.proxy.constants.home_app_versions import VER_1_5, VER_2_100_12
from smarttv.droideka.proxy.response.carousels import KpMultiSelectionsResponseFields
from smarttv.droideka.proxy.models import Category2 as Category, Promo
from smarttv.droideka.proxy.request.carousels import MixedCarouselsInfo, KpCarouselInfo, KpCategoryInfo, \
    VhCarouselInfo, PurchasesCarouselInfo
from smarttv.droideka.proxy.request.promo import HotelRequestInfo, PromoRequestInfo
from smarttv.droideka.proxy.vh.constants import MAIN_CATEGORY_ID, ANDROID_TV_MAIN_CATEGORY_TAG
from smarttv.droideka.proxy import exceptions
from smarttv.droideka.utils import ParsedVersion
from smarttv.droideka.utils.chaining import DataSource, FilterableDataSource
from smarttv.droideka.utils.chaining import YlogGreenlet, get_results_from_greenlets
from smarttv.droideka.utils.url import build_view_url
from smarttv.droideka.utils.enumerate import enumerate as enumerate_with_step
from smarttv.utils.headers import AUTHORIZATION_HEADER, REQUEST_ID_HEADER, FORWARDED_HEADER, \
    USER_IP_HEADER, QUASAR_DEVICE_ID, TVM_USER_TICKET
from smarttv.utils.django import get_http_header
from smarttv.droideka.proxy.serializers.response import PromoSerializer
from smarttv.droideka.proxy import tvm
from smarttv.droideka.proxy.blackbox import UserInfo
from smarttv.utils import headers as headers_constants
from alice.memento.proto.api_pb2 import TReqChangeUserObjects, TConfigKeyAnyPair, EConfigKey
from alice.memento.proto.user_configs_pb2 import TSmartTvMusicPromoConfig
from google.protobuf import any_pb2

from smarttv.droideka.unistat.metrics import CacheType, LocalCachePlace

logger = logging.getLogger(__name__)

FIELD_INCLUDES = 'includes'
FIELD_SET = 'set'

FIELD_BANNED = 'banned'

FIELD_CATEGORY_MAPPING = 'category_mapping'

EXP_MUSIC_CAROUSEL_POSITION = 'music_carousel_position'


@cache_memoize(timeout=settings.DEFAULT_RESPONSE_CACHE_TIME, cache_alias='local',
               **cache.get_cache_memoize_callables(LocalCachePlace.PROMO.value, CacheType.LOCAL))
def get_promos():
    return Promo.objects.using(settings.DB_REPLICA).all()


def get_content_type(data):
    return data.get('content_type_name') if data else None


def negative_filter_by_attr_name(items, attr_name, other):
    return [item for item in items if not item.get(attr_name) or item.get(attr_name) not in other]


def generate_category_id_dict(category: Category) -> dict:
    return {'category_id': category.category_id}


def generate_carousel_id_dict(category: Category) -> dict:
    return {'carousel_id': category.category_id}


class ContentDetailDataSource(DataSource):
    """
    Loads the detail of content by content_id
    """

    def __init__(self, content_id, headers, request):
        self.request = request
        self.content_id = content_id
        self.headers = headers

    def get_result(self) -> dict:
        response = api.vh.client.content_detail(
            initial_request=self.request,
            headers=self.headers,
            content_id=self.content_id
        )

        # Sometimes, VH doesn't return 404 if there is no such 'content_id'. VH can return nothing
        # This behaviour reported to VH and they confirmed "it's OK"
        if not response:
            raise Http404(f"No document found for content_id: '{self.content_id}'")

        return response.get('content', {})


class HotelDataSource(DataSource):
    required_fields = ('title', 'subtitle', 'imageUrl', 'infoUrl')

    def __init__(self, hotel_request_info: Optional[HotelRequestInfo]):
        self.request_info = hotel_request_info

    def get_result(self) -> dict:
        if not self.request_info:
            logger.debug('Hotel promo not requested')
            return {}
        try:
            result = api.alice.client.device_info(self.request_info.device_id)
        except api.alice.client.NotFoundError as err:
            logger.info(f'Device with id {self.request_info.device_id} not found in alice b2b backend: {err}')
            return {}
        except api.alice.client.ResponseError:
            logger.exception('Can not get device information from Alice b2b api')
            return {}

        if self.has_null_fields(result):
            # alice b2b returns null in unfilled properties
            # and because we can not serialize nulls, we skip whole answer
            return {}

        if not result:
            return {}

        # auto_fields required in serializator
        response = {'auto_fields': {'promo_type': 'hotel'}}
        response['result'] = result

        return response

    def has_null_fields(self, result):
        if any([result.get(field) is None for field in self.required_fields]):
            logger.debug('Device info has empty required field. Skipping result: %s', result)
            return True

        return False


class ContentDetailIdentityInfoDataSource(DataSource):
    """
    Provides structure with ids for loading content detail from different sources concurrently
    """

    def __init__(self, content_id, onto_id, content_type, headers, request, passport_uid=None):
        super().__init__()
        self.content_id = content_id
        self.onto_id = onto_id
        self.content_type = content_type
        self.headers = headers
        self.request = request
        self.passport_uid = passport_uid

    def get_result(self) -> dict:
        return {
            'content_id': self.content_id,
            'onto_id': self.onto_id,
            'content_type_name': self.content_type,
            'headers': self.headers,
            'request': self.request,
            'passport_uid': self.passport_uid
        }


class SeriesSeasonsV6DataSource(DataSource):

    def __init__(self, request, series_id, offset, limit, headers):
        self.request = request
        self.headers = headers
        self.series_id = series_id
        self.offset = offset
        self.limit = limit

    def get_result(self) -> dict:
        return api.vh.client.seasons(
            initial_request=self.request,
            series_id=self.series_id,
            headers=self.headers,
            limit=self.limit,
            offset=self.offset,
        )


class SeriesEpisodesV6DataSource(DataSource):

    def __init__(self, request, season_id, offset, limit, headers):
        self.request = request
        self.season_id = season_id
        self.offset = offset
        self.limit = limit
        self.headers = headers

    def get_result(self) -> dict:
        return api.vh.client.series_episodes(
            initial_request=self.request,
            headers=self.headers,
            season_id=self.season_id,
            offset=self.offset,
            limit=self.limit,
        )


class FilterableSeriesEpisodesDataSource(FilterableDataSource):

    def __init__(self, request, season_id, offset, limit, headers):
        super().__init__(offset, limit, limit, auto_load_enabled=False)
        self.request = request
        self.season_id = season_id
        self.offset = offset
        self.limit = limit
        self.headers = headers

    def filter_bad_items(self, item) -> bool:
        return item and not item.get(FIELD_BANNED)

    def get_filters(self) -> list:
        return [lambda top_list: [item for item in top_list if self.filter_bad_items(item)]]

    def get_root_list(self, container) -> Optional[list]:
        return container.get(FIELD_SET)

    def get_nested_list(self, nested_item) -> list:
        return []

    def create_nested_filterable_source(self, nested_item):
        return None

    def get_next_page(self, offset, limit, additional_params=None) -> dict:
        return SeriesEpisodesV6DataSource(
            request=self.request,
            season_id=self.season_id,
            limit=self.limit,
            offset=self.offset,
            headers=self.headers
        ).get_result()

    def wrap_root_list(self, container, target_list, pagination_state_params) -> dict:
        container[FIELD_SET] = target_list
        return container

    def get_initial_additional_params(self):
        pass

    def extract_additional_params(self, page):
        pass


class EntitySearchSingleItemDataSource(DataSource):
    """
    Loads additional info from entity search
    """

    def __init__(self, onto_id_list, headers, request, passport_uid=None):
        self.onto_id_list = onto_id_list
        self.headers = headers
        self.passport_uid = passport_uid
        self.request = request

    def get_result(self) -> dict:
        logger.debug('onto_id=%s', self.onto_id_list)
        if self.onto_id_list:
            result = api.es.client.get_object_response(
                request=self.request, obj_id_list=self.onto_id_list, headers=self.headers,
                passport_uid=self.passport_uid)
            if result:
                cards = result.get('cards')
                if cards:
                    return cards[0]
        else:
            logger.debug('Skip loading object response')

        return {}


class OttMetadataDataSource(DataSource):
    """
    Loads covers and logos from OTT
    May return nothing, because not all content is marked up
    """

    def __init__(self, content_id, headers, content_detail, request):
        self.content_id = content_id
        self.headers = headers
        self.content_detail = content_detail
        self.request = request

    def get_result(self) -> dict:
        content_type = get_content_type(self.content_detail)
        if api.vh.is_ott_content(content_type):
            return api.ott.client.metadata(content_id=self.content_id, initial_request=self.request,
                                           headers=self.headers)

        logger.debug('{} is not OTT content, skip getting ott metadata'.format(self.content_id))
        return {}


class SimpleCarouselDataSource(DataSource):

    def __init__(self, carousel_id,
                 offset=0, limit=5, docs_cache_hash=None, restriction_age=None, headers=None,
                 tag=None, filters=None, request=None):
        super().__init__()
        self.carousel_id = carousel_id
        self.docs_cache_hash = docs_cache_hash
        self.headers = headers
        self.restriction_age = restriction_age
        self.offset = offset
        self.limit = limit
        self.tag = tag
        self.filters = filters
        self.request = request

    def get_result(self) -> dict:
        if self.carousel_id == FILTERABLE_CAROUSEL_ID:
            return api.vh.client.carousel(
                initial_request=self.request,
                headers=self.headers,
                carousel_id=self.carousel_id,
                tag=self.tag,
                filters=self.filters,
                offset=self.offset,
                limit=self.limit,
                cache_hash=self.docs_cache_hash,
            )
        return api.vh.client.carousel_videohub(
            initial_request=self.request,
            headers=self.headers,
            carousel_id=self.carousel_id,
            tag=self.tag,
            limit=self.limit,
            offset=self.offset,
            docs_cache_hash=self.docs_cache_hash,
            restriction_age=self.restriction_age
        )


class MoreUrlMixin(ABC):
    def get_more_url(self, view_name, request, pagination_params, partial_result, cache_key):
        # todo: если нужных ключей нет, то furl сгенерит '?limit&offset' - так себе решение (придумать что-то поизящнее)
        args = {
            'offset': pagination_params.get('offset'),
            'limit': pagination_params.get('requested_amount'),
        }

        docs_cache_hash = pagination_params.get(cache_key) or (partial_result and partial_result.get(cache_key))
        if docs_cache_hash:
            args[cache_key] = docs_cache_hash

        args.update(self.get_more_url_args(pagination_params))

        return build_view_url(args, view_name, request)

    @abstractmethod
    def get_more_url_args(self, pagination_params=None) -> dict:
        pass


class CarouselItemPredicate:
    """
    Checks is item fits the requirements for requested carousel
    """

    # noinspection PyMethodMayBeStatic
    def _is_banned(self, item):
        return item.get(FIELD_BANNED, False)

    def is_item_good(self, item):
        return not self._is_banned(item)


class FilterableCarouselDataSource(FilterableDataSource, MoreUrlMixin):

    def __init__(self, carousel_data, next_url_name, offset=0, limit=5, requested_amount=5, restriction_age=None,
                 headers=None, request=None, auto_load_enabled=True, filter_key=None, tag=None):
        super().__init__(offset, limit, requested_amount, auto_load_enabled=auto_load_enabled)

        self.carousel_data = carousel_data
        self.headers = headers
        self.request = request
        self.next_url_name = next_url_name
        self.restriction_age = restriction_age
        self.carousel_item_predicate = CarouselItemPredicate()
        self.filter_key = filter_key
        self.tag = tag

    @property
    def truncating_result_list_enabled(self):
        # disable it for carousel, since this hook uses 'cache_hash'
        return False

    def get_more_url_args(self, pagination_params=None) -> dict:
        result = {}
        carousel_id = self.carousel_data.get(VhCarousel.FIELD_CAROUSEL_ID)
        if carousel_id:
            result[VhCarousel.FIELD_CAROUSEL_ID] = carousel_id
        filters = pagination_params.get(VhCarousel.FIELD_FILTER)
        tag = pagination_params.get(VhCarousel.FIELD_TAG)
        if filters and tag:
            result[VhCarousel.FIELD_FILTER] = filters
            result[VhCarousel.FIELD_TAG] = tag
        return result

    def get_filters(self) -> list:
        return [lambda top_list: [item for item in top_list if self.carousel_item_predicate.is_item_good(item)]]

    def wrap_root_list(self, container, target_list, pagination_state_params) -> dict:
        container[FIELD_INCLUDES] = target_list
        if (not self.auto_load_enabled and len(target_list) > 0) or len(target_list) >= self.limit:
            more_url = self.get_more_url(
                self.next_url_name, self.request, pagination_state_params, container, VhCarousel.FIELD_DOCS_CACHE_HASH)
            container[VhFeed.FIELD_MORE] = more_url
            container[VhCarousel.FIELD_MORE_INFO] = {VhCarousel.FIELD_MORE_URL: more_url}
        return container

    def get_root_list(self, container) -> Optional[list]:
        return container.get(FIELD_INCLUDES)

    def get_nested_list(self, nested_item) -> list:
        return []

    def get_initial_additional_params(self) -> dict:
        result = {VhCarousel.FIELD_DOCS_CACHE_HASH: self.carousel_data.get(VhCarousel.FIELD_DOCS_CACHE_HASH)}
        if self.filter_key:
            result[VhCarousel.FIELD_FILTER] = self.filter_key
        if self.tag:
            result[VhCarousel.FIELD_TAG] = self.tag
        return result

    def extract_additional_params(self, page) -> dict:
        result = {VhCarousel.FIELD_DOCS_CACHE_HASH: page.get(VhFeed.FIELD_CACHE_HASH)}
        if self.filter_key:
            result[VhCarousel.FIELD_FILTER] = self.filter_key
        if self.tag:
            result[VhCarousel.FIELD_TAG] = self.tag
        return result

    def _get_docs_cache_hash(self, params):
        if not params:
            return None
        return params.get(VhCarousel.FIELD_DOCS_CACHE_HASH)

    def get_initial_page(self):
        return self.carousel_data

    def get_next_page(self, offset, limit, additional_params=None) -> dict:
        carousel_id = self.carousel_data[VhCarousel.FIELD_CAROUSEL_ID]
        cache_hash = self._get_docs_cache_hash(additional_params)
        data_source = SimpleCarouselDataSource(
            request=self.request,
            carousel_id=carousel_id,
            offset=offset,
            limit=limit,
            docs_cache_hash=cache_hash,
            restriction_age=self.restriction_age,
            headers=self.headers,
            tag=self.tag,
            filters=self.filter_key,
        )

        result = data_source.get_result()
        if FIELD_SET in result:
            result[FIELD_INCLUDES] = result.pop(FIELD_SET)

        return result

    def create_nested_filterable_source(self, nested_item):
        return None


class SimpleCarouselsDataSource(DataSource):
    """
    Retrieves portion of carousels from feed
    """

    def __init__(self, category_id, cache_hash, limit, offset, max_item_count, restriction_age, headers, request):
        super().__init__()

        self.category_id = category_id
        self.cache_hash = cache_hash
        self.max_item_count = max_item_count
        self.headers = headers
        self.restriction_age = restriction_age
        self._root_list_field_name = 'items'
        self.limit = limit
        self.offset = offset
        self.request = request

    @property
    def main_category_tag(self):
        return ANDROID_TV_MAIN_CATEGORY_TAG

    def get_vh_tag(self):
        return self.category_id if self.category_id != MAIN_CATEGORY_ID else self.main_category_tag

    def get_result(self) -> dict:
        result = api.vh.client.feed(
            initial_request=self.request,
            headers=self.headers,
            tag=self.get_vh_tag(),
            num_docs=self.max_item_count,
            cache_hash=self.cache_hash,
            limit=self.limit,
            offset=self.offset,
            restriction_age=self.restriction_age,
            block_recommendations=self.category_id == MAIN_CATEGORY_ID,
        )

        return result


class CarouselsDataSource(FilterableDataSource, MoreUrlMixin):
    """
    DataSource for retrieving carousels from carousels_videohub nd filtering banned
    """
    FIELD_ROOT_LIST = 'items'
    FIELD_CACHE_HASH = VhFeed.FIELD_CACHE_HASH
    BANNED_CAROUSELS = tuple()

    def __init__(self, category_id, next_url_name, nested_next_url_name, cache_hash=None, max_items_count=10, offset=0,
                 limit=10, requested_amount=10, restriction_age=None, headers=None, request=None):
        super().__init__(offset, limit, requested_amount)
        self.category_id = category_id
        self.headers = headers
        self.request = request
        self.max_items_count = max_items_count
        self.cache_hash = cache_hash
        self.next_url_name = next_url_name
        self.nested_next_url_name = nested_next_url_name
        self.restriction_age = restriction_age

    @property
    def truncating_result_list_enabled(self):
        # disable it for carousels, since this hook uses 'cache_hash'
        return False

    @staticmethod
    def filter_banned_carousels(top_list):
        return [carousel for carousel in top_list if not carousel.get(FIELD_BANNED, False)]

    def filter(self, top_list):
        return negative_filter_by_attr_name(top_list, 'carousel_id', self.BANNED_CAROUSELS)

    def get_filters(self) -> list:
        filters = [self.filter_banned_carousels]
        if self.category_id == MAIN_CATEGORY_ID:
            filters.append(self.filter)
        return filters

    def wrap_root_list(self, container, target_list, pagination_state_params) -> dict:
        container[self.FIELD_ROOT_LIST] = target_list
        if len(target_list) >= self.limit:
            container[VhFeed.FIELD_MORE] = self.get_more_url(
                self.next_url_name, self.request, pagination_state_params, container, self.FIELD_CACHE_HASH)
        return container

    def get_root_list(self, container) -> Optional[list]:
        return container.get(self.FIELD_ROOT_LIST)

    def get_nested_list(self, nested_item) -> list:
        return nested_item.get(FIELD_INCLUDES)

    def get_initial_additional_params(self) -> dict:
        return {VhFeed.FIELD_CACHE_HASH: self.cache_hash}

    def extract_additional_params(self, page) -> dict:
        self.cache_hash = page.get(VhFeed.FIELD_CACHE_HASH)
        return {self.FIELD_CACHE_HASH: page.get(self.FIELD_CACHE_HASH)}

    def get_next_page(self, offset, limit, additional_params=None) -> dict:
        # 'cache_hash' must be updated after every request
        cache_hash = (additional_params or {}).get(self.FIELD_CACHE_HASH)
        datasource = SimpleCarouselsDataSource(
            category_id=self.category_id,
            cache_hash=cache_hash,
            limit=limit,
            offset=offset,
            max_item_count=self.max_items_count,
            restriction_age=self.restriction_age,
            headers=self.headers,
            request=self.request,
        )
        return datasource.get_result()

    def create_nested_filterable_source(self, nested_item) -> FilterableCarouselDataSource:
        cache_hash = nested_item.get(self.FIELD_CACHE_HASH)
        if cache_hash:
            nested_item[VhCarousel.FIELD_DOCS_CACHE_HASH] = cache_hash
        return FilterableCarouselDataSource(
            carousel_data=nested_item,
            next_url_name=self.nested_next_url_name,
            requested_amount=self.max_items_count,
            limit=self.max_items_count,
            restriction_age=self.restriction_age,
            request=self.request,
            headers=self.headers,
        )

    @property
    def has_nested_filterable_data_source(self):
        return True

    def get_more_url_args(self, pagination_params=None) -> dict:
        return {VhFeed.FIELD_CATEGORY_ID: self.category_id}


class CarouselsV7DataSource(FilterableDataSource):
    """
    DataSource for retrieving carousels from carousels_videohub nd filtering banned
    """
    FIELD_ROOT_LIST = 'items'
    FIELD_CACHE_HASH = VhFeed.FIELD_CACHE_HASH

    def __init__(self, request_info: MixedCarouselsInfo):
        super().__init__(request_info.vh_offset, request_info.vh_limit, request_info.vh_limit)
        self.total_offset = request_info.offset
        self.category_id = request_info.category_id
        self.headers = request_info.headers
        self.request = request_info.request
        self.max_items_count = request_info.max_item_count
        self.cache_hash = request_info.cache_hash
        self.next_url_name = request_info.next_url_name
        self.nested_next_url_name = request_info.nested_next_url_name
        self.restriction_age = request_info.restriction_age
        self.external_carousel_offset = request_info.external_carousel_offset
        self.current_external_carousel_amount = request_info.current_external_carousel_amount

    @property
    def truncating_result_list_enabled(self):
        # disable it for carousels, since this hook uses 'cache_hash'
        return False

    @staticmethod
    def filter_banned_carousels(top_list):
        return [carousel for carousel in top_list if not carousel.get(FIELD_BANNED, False)]

    def get_filters(self) -> list:
        return [self.filter_banned_carousels]

    def wrap_root_list(self, container, target_list, pagination_state_params) -> dict:
        container[self.FIELD_ROOT_LIST] = target_list
        return container

    def get_root_list(self, container) -> Optional[list]:
        return container.get(self.FIELD_ROOT_LIST)

    def get_nested_list(self, nested_item) -> list:
        return nested_item.get(FIELD_INCLUDES)

    def get_initial_additional_params(self) -> dict:
        return {VhFeed.FIELD_CACHE_HASH: self.cache_hash}

    def extract_additional_params(self, page) -> dict:
        self.cache_hash = page.get(VhFeed.FIELD_CACHE_HASH)
        return {self.FIELD_CACHE_HASH: page.get(self.FIELD_CACHE_HASH)}

    def get_next_page(self, offset, limit, additional_params=None) -> dict:
        # 'cache_hash' must be updated after every request
        if additional_params and additional_params.get(self.FIELD_CACHE_HASH):
            cache_hash = additional_params[self.FIELD_CACHE_HASH]
        else:
            cache_hash = 0
        datasource = SimpleCarouselsDataSource(
            category_id=self.category_id,
            cache_hash=cache_hash,
            limit=limit,
            offset=offset,
            max_item_count=self.max_items_count,
            restriction_age=self.restriction_age,
            headers=self.headers,
            request=self.request,
        )
        return datasource.get_result()

    def create_nested_filterable_source(self, nested_item) -> FilterableCarouselDataSource:
        cache_hash = nested_item.get(self.FIELD_CACHE_HASH)
        if cache_hash:
            nested_item[VhCarousel.FIELD_DOCS_CACHE_HASH] = cache_hash
        return FilterableCarouselDataSource(
            carousel_data=nested_item,
            next_url_name=self.nested_next_url_name,
            requested_amount=self.max_items_count,
            limit=self.max_items_count,
            restriction_age=self.restriction_age,
            request=self.request,
            headers=self.headers,
        )

    @property
    def has_nested_filterable_data_source(self):
        return True


class OttSeriesStructureDataSource(DataSource):
    """
    Loads seasons + series from
    """

    def __init__(self, content_id, content_type, headers, request):
        self.content_id = content_id
        self.content_type = content_type
        self.headers = headers
        self.request = request

    def get_result(self) -> dict:
        return api.ott.client.children(self.content_id, self.request, self.headers)


class FeedDataSource(DataSource):
    KEY_CATEGORY_MAPPING = 'category_mapping'

    FIELD_NESTED_NEXT_URL_NAME = 'nested_next_url_name'
    FIELD_NEXT_URL_NAME = 'next_url_name'
    FIELD_REQUEST = 'request'

    def __init__(self, carousels_info: MixedCarouselsInfo):
        self.carousels_info = carousels_info

    @cache_memoize(timeout=settings.DEFAULT_RESPONSE_CACHE_TIME,
                   cache_alias='local',
                   **cache.get_cache_memoize_callables(LocalCachePlace.CHILD_CATEGORIES_BY_PARENT_ID.value, CacheType.LOCAL))
    def get_child_categories_by_parent_id(self, parent_id: str) -> list:
        return categories_provider.get_categories(
            self.carousels_info.request.platform_info,
            parent_id,
            self.carousels_info.request.request_info
        )

    def get_category_position(self, category: Category):
        if self.carousels_info.category_id == 'main' and category.content_type == MusicCarousel.TYPE:
            music_main_carousel_position = self.carousels_info.request.request_info.experiments.get_value_or_default(
                EXP_MUSIC_CAROUSEL_POSITION)
            if music_main_carousel_position is not None and isinstance(music_main_carousel_position, int) \
                    and music_main_carousel_position >= 0:
                return music_main_carousel_position - 1
        return category.position

    def filter_categories_by_paging_info(self, available_categories, offset, limit):
        result = []
        for category in available_categories:
            position = self.get_category_position(category)
            if position is not None and offset <= position < (offset + limit):
                result.append(category)
        return result

    def get_result(self) -> dict:
        available_categories = self.get_child_categories_by_parent_id(self.carousels_info.category_id)
        available_categories = self.filter_categories_by_paging_info(
            available_categories,
            self.carousels_info.offset,
            self.carousels_info.limit
        )
        self.carousels_info.extend_categories(available_categories)

        return {MixedCarouselsInfo.FIELD_KEY: self.carousels_info}


class KpSelectionMixin(MoreUrlMixin):

    def __init__(self, request, headers, next_url_name):
        self.request = request
        self.headers = headers
        self.next_url_name = next_url_name

    def get_more_url_args(self, pagination_params=None) -> dict:
        result = {CarouselsExternal.FIELD_CAROUSEL_TYPE: KpCarousel.TYPE}
        if pagination_params and DroidekaCarousel.FIELD_CAROUSEL_ID in pagination_params:
            result[DroidekaCarousel.FIELD_CAROUSEL_ID] = pagination_params[DroidekaCarousel.FIELD_CAROUSEL_ID]
        return result

    def get_pagination_state_params(self, paging_meta: dict, carousel_id: str,
                                    carousel_type: str = None, more_url_limit: int = None):
        _from = paging_meta.get('from')
        to = paging_meta.get('to')
        offset = to
        if more_url_limit:
            limit = more_url_limit
        else:
            limit = to - _from
        return {
            'offset': offset,
            'requested_amount': limit,
            VhCarousel.FIELD_DOCS_CACHE_HASH: paging_meta.get('sessionId'),
            'carousel_id': carousel_id,
            CarouselsExternal.FIELD_CAROUSEL_TYPE: carousel_type
        }


class KpSelectionProvider(KpSelectionMixin):

    def __init__(self, request, headers, next_url_name, more_url_limit=None):
        super().__init__(request, headers, next_url_name)
        self.more_url_limit = more_url_limit

    def get_and_enrich_kp_carousel(
        self,
        selection_window_id,
        selection_id,
        carousel_type,
        carousel_id,
        limit,
        offset=0,
        position=None,
        custom_title=None,
        session_id=None,
    ):
        kp_carousel = api.ott.client.selections(selection_window_id=selection_window_id,
                                                selection_id=selection_id,
                                                initial_request=self.request,
                                                headers=self.headers,
                                                limit=limit,
                                                session_id=session_id,
                                                offset=offset)
        if position:
            kp_carousel['position'] = position
        kp_carousel[CarouselsExternal.FIELD_CAROUSEL_TYPE] = carousel_type
        if custom_title:
            kp_carousel['title'] = custom_title
        kp_carousel[api.ott.KEY_SELECTION_ID] = selection_id
        kp_carousel[api.ott.KEY_SELECTION_WINDOW_ID] = selection_window_id
        paging_meta = kp_carousel.get(KpCarousel.FIELD_PAGING_META)
        if paging_meta and paging_meta.get(KpCarousel.FIELD_HAS_MORE):
            pagination_params = self.get_pagination_state_params(
                kp_carousel[KpCarousel.FIELD_PAGING_META], carousel_id, carousel_type, self.more_url_limit)
            more_url = self.get_more_url(
                self.next_url_name,
                self.request,
                pagination_params,
                None,
                VhCarousel.FIELD_DOCS_CACHE_HASH
            )
            kp_carousel[VhFeed.FIELD_MORE] = more_url
            kp_carousel[VhCarousel.FIELD_MORE_INFO] = {VhCarousel.FIELD_MORE_URL: more_url}
        return kp_carousel


class KpSelectionDataSource(DataSource):

    def __init__(self, carousel_info: KpCarouselInfo):
        self.carousel_info = carousel_info
        self.provider = KpSelectionProvider(
            carousel_info.request, carousel_info.headers, carousel_info.next_url_name, carousel_info.more_url_limit)

    def get_result(self) -> dict:
        return self.provider.get_and_enrich_kp_carousel(
            self.carousel_info.window_id,
            self.carousel_info.selection_id,
            KpCarousel.TYPE,
            self.carousel_info.carousel_id,
            self.carousel_info.limit,
            offset=self.carousel_info.offset,
            session_id=self.carousel_info.session_id
        )


class KpSelectionsDataSource(DataSource):

    def __init__(self, request_info: MixedCarouselsInfo):
        self.kp_carousel_info = request_info.external_categories_mapping.get(KpCarousel.TYPE)
        self.next_url_view_name = request_info.nested_next_url_name
        self.request = request_info.request
        self.headers = request_info.headers
        self.limit = request_info.max_item_count
        self.provider = KpSelectionProvider(self.request, self.headers, self.next_url_view_name)

    def get_result(self) -> list:
        if not self.kp_carousel_info:
            return []
        category: Category
        threads = []
        for category in self.kp_carousel_info:
            window_id, selection_id = category.category_id.split('/')
            if selection_id == PURCHASES_FAKE_SELECTION_ID:
                continue
            get_selections = partial(self.provider.get_and_enrich_kp_carousel,
                                     selection_window_id=window_id,
                                     selection_id=selection_id,
                                     position=category.internal_position,
                                     carousel_type=category.content_type,
                                     custom_title=category.title,
                                     carousel_id=category.category_id,
                                     limit=self.limit)
            threads.append(YlogGreenlet.spawn(get_selections))
        gevent.wait(threads, timeout=settings.DEFAULT_NETWORK_GREENLET_TIMEOUT)
        result = get_results_from_greenlets(
            threads,
            error_message='Error loading selection',
            timeout_message='Timeout exceed for loading selection',
            message_prefix=self.__class__.__name__,
        )
        return result


class KpMultiSelectionsDataSource(DataSource):

    def __init__(self, request_info: KpCategoryInfo, music_carousel_category: Optional[Category]):
        self.kp_category_info = request_info
        self.music_category = music_carousel_category

    def _build_multi_selection_url(self, selection_id):
        args = {
            DroidekaCarousel.FIELD_CAROUSEL_ID: f'{self.kp_category_info.window_id}/{selection_id}',
            DroidekaCarousel.FIELD_LIMIT: settings.EMBEDDED_CAROUSEL_LIMIT,
            DroidekaCarousel.FIELD_CAROUSEL_TYPE: KpCarousel.TYPE,
            DroidekaCarousel.FIELD_MORE_URL_LIMIT: self.kp_category_info.items_limit
        }
        return build_view_url(args, self.kp_category_info.nested_next_url_name, self.kp_category_info.request)

    def _fill_info_for_carousel_items(self, carousel):
        carousel_type = carousel[KpMultiSelectionsResponseFields.FIELD_TYPE]
        if carousel.get(KpMultiSelectionsResponseFields.FIELD_DATA):
            for item in carousel.get(KpMultiSelectionsResponseFields.FIELD_DATA):
                selection_id = item.get(KpMultiSelectionsResponseFields.FIELD_SELECTION_ID)
                if selection_id:
                    item[api.ott.KEY_SELECTION_WINDOW_ID] = self.kp_category_info.window_id
                    item[api.ott.KEY_SELECTION_ID] = selection_id

                if carousel_type == KpMultiSelectionsResponseFields.TYPE_MULTI_SELECTION:
                    item[KpMultiSelection.FIELD_MULTISELECTION_URL] = self._build_multi_selection_url(selection_id)

    def _fill_carousel_type_if_necessary(self, carousel):
        content_type = carousel[KpMultiSelectionsResponseFields.FIELD_TYPE]
        if content_type == KpMultiSelectionsResponseFields.TYPE_MULTI_SELECTION:
            if ParsedVersion(self.kp_category_info.request.platform_info.app_version) < VER_1_5:
                carousel[KpMultiSelectionsResponseFields.FIELD_CAROUSEL_TYPE] = CarouselType.TYPE_SQUARE
            else:
                carousel[KpMultiSelectionsResponseFields.FIELD_CAROUSEL_TYPE] = CarouselType.TYPE_SQUARE_BIG

    def _get_paging_meta(self, obj):
        paging_meta = obj.get(KpCarousel.FIELD_PAGING_META)
        if not paging_meta:
            raise exceptions.NoMoreError()
        has_more = paging_meta.get(KpCarousel.FIELD_HAS_MORE)
        if not has_more:
            raise exceptions.NoMoreError()
        return paging_meta

    def _fill_more_url_for_selections_container(self, obj):
        try:
            paging_meta = self._get_paging_meta(obj)
        except exceptions.NoMoreError:
            return
        session_id = paging_meta.get(KpCarousel.FIELD_SESSION_ID)
        args = {
            DroidekaCategory.FIELD_CATEGORY_ID: self.kp_category_info.window_id,
            DroidekaCategory.FIELD_CACHE_HASH: session_id,
            DroidekaCategory.FIELD_MAX_ITEMS_COUNT: self.kp_category_info.items_limit,
            DroidekaCategory.FIELD_OFFSET: paging_meta.get(
                KpMultiSelectionsResponseFields.FIELD_TO,
                self.kp_category_info.selections_offset + self.kp_category_info.selections_limit),
            DroidekaCategory.FIELD_LIMIT: self.kp_category_info.selections_limit
        }

        obj[DroidekaCategory.FIELD_MORE_URL] = build_view_url(
            args, self.kp_category_info.next_url_name, self.kp_category_info.request)

    def _fill_more_url_for_carousel(self, obj):
        try:
            paging_meta = self._get_paging_meta(obj)
        except exceptions.NoMoreError:
            return
        session_id = paging_meta.get(KpCarousel.FIELD_SESSION_ID)
        selection_id = obj[KpMultiSelectionsResponseFields.FIELD_SELECTION_ID]
        current_items_amount = len(obj[KpMultiSelectionsResponseFields.FIELD_DATA])
        args = {
            DroidekaCarousel.FIELD_CAROUSEL_ID: f'{self.kp_category_info.window_id}/{selection_id}',
            DroidekaCarousel.FIELD_CACHE_HASH: session_id,
            DroidekaCarousel.FIELD_CAROUSEL_TYPE: KpCarousel.TYPE,
            DroidekaCarousel.FIELD_LIMIT: self.kp_category_info.items_limit,
            DroidekaCarousel.FIELD_OFFSET: paging_meta.get(
                KpMultiSelectionsResponseFields.FIELD_TO, current_items_amount),
        }

        url = build_view_url(args, self.kp_category_info.nested_next_url_name, self.kp_category_info.request)

        obj[DroidekaCarousel.FIELD_MORE_INFO] = {
            DroidekaCarousel.FIELD_MORE_URL: url
        }

    def _fill_info_for_carousels(self, obj):
        if obj.get(KpMultiSelectionsResponseFields.FIELD_COLLECTIONS):
            self._fill_more_url_for_selections_container(obj)
            for carousel in obj[KpMultiSelectionsResponseFields.FIELD_COLLECTIONS]:
                carousel[api.ott.KEY_SELECTION_WINDOW_ID] = self.kp_category_info.window_id
                carousel[api.ott.KEY_SELECTION_ID] = carousel[KpMultiSelectionsResponseFields.FIELD_SELECTION_ID]
                self._fill_more_url_for_carousel(carousel)
                self._fill_info_for_carousel_items(carousel)
                self._fill_carousel_type_if_necessary(carousel)

    def add_music_request_info(self, result: dict):
        if not self.music_category:
            return
        music_request_info = MixedCarouselsInfo(
            {DroidekaCategory.FIELD_CATEGORY_ID: self.kp_category_info.window_id,
             DroidekaCategory.FIELD_OFFSET: self.kp_category_info.selections_offset,
             DroidekaCategory.FIELD_LIMIT: self.kp_category_info.selections_limit,
             DroidekaCategory.FIELD_CACHE_HASH: self.kp_category_info.session_id,
             DroidekaCategory.FIELD_MAX_ITEMS_COUNT: self.kp_category_info.items_limit,
             CarouselsExternal.FIELD_EXTERNAL_CAROUSEL_OFFSET: 0},
            self.kp_category_info.request, self.kp_category_info.headers, '', '')
        music_request_info.extend_categories([self.music_category])
        result[MixedCarouselsInfo.FIELD_KEY] = music_request_info

    def get_result(self) -> Any:
        if not self.kp_category_info:
            return {}
        result = api.ott.client.multi_selections(
            self.kp_category_info.window_id,
            self.kp_category_info.headers,
            self.kp_category_info.request,
            self.kp_category_info.items_limit,
            self.kp_category_info.selections_limit,
            self.kp_category_info.selections_offset,
            self.kp_category_info.session_id
        )
        self._fill_info_for_carousels(result)
        self.add_music_request_info(result)

        return result


class EmbeddedSectionUrlConfig:

    def __init__(self, id_generator_func: Callable[[Category], dict], args: dict, view_name: str):
        self.id_generator_func = id_generator_func
        self.args = args
        self.view_name = view_name


class EmbeddedSectionDataSource(DataSource):

    def __init__(self, carousel_info: MixedCarouselsInfo):
        self.carousel_info = carousel_info
        self.config_mapping = {
            '': EmbeddedSectionUrlConfig(
                generate_category_id_dict,
                {
                    'offset': 0,
                    'limit': self.carousel_info.limit,
                    'max_items_count': self.carousel_info.max_item_count
                },
                carousel_info.carousels_next_url_name
            ),
            'vh_carousel': EmbeddedSectionUrlConfig(
                generate_carousel_id_dict,
                {
                    'offset': 0,
                    'limit': self.carousel_info.max_item_count
                },
                carousel_info.carousel_next_url_name
            )
        }

    def fill_url(self, config: EmbeddedSectionUrlConfig, category: Category):
        args = deepcopy(config.args)
        args.update(config.id_generator_func(category))
        url = build_view_url(args, config.view_name, self.carousel_info.request)
        if url:
            category.url = url

    def get_result(self):
        embedded_sections = self.carousel_info.external_categories_mapping.get(EMBEDDED_SECTION_TYPE)
        if not embedded_sections:
            category_id = self.carousel_info.category_id
            offset = self.carousel_info.offset
            limit = self.carousel_info.limit
            logger.info('No embedded sections for category_id: %s, offset: %s, limit: %s', category_id, offset, limit)
            return None
        result = []
        for section in embedded_sections:
            categories = categories_provider.get_categories(
                self.carousel_info.request.platform_info,
                section.category_id,
                self.carousel_info.request.request_info
            )
            if categories:
                for category in categories:
                    try:
                        self.fill_url(self.config_mapping[category.content_type], category)
                    except KeyError:
                        raise ValueError(f"Unhandled content type '{category.content_type}'")
                section.includes = categories
                result.append(section)
        return result


class CarouselContentFiltersDataSource(DataSource):

    def build_genres_filter(self, vh_genres_filter: dict) -> Optional[dict]:
        if vh_genres_filter['type'] != 'select':
            return None
        keys = []
        raw_genres = vh_genres_filter['values']
        raw_genres.sort(key=lambda x: x['title'])
        for index, item in enumerate_with_step(raw_genres, start=10, step=10):
            keys.append({'title': item['title'], 'key': f"genre={item['id']}", 'rank': index})

        return {'title': vh_genres_filter['title'], 'keys': keys, 'rank': 20}

    def build_years_filter(self, vh_years_filter: dict) -> Optional[dict]:
        if vh_years_filter['type'] != 'range':
            return None
        keys = [
            {'title': 'За всё время', 'key': None, 'rank': 10},
            {'title': 'Новинки', 'key': 'year=2021', 'rank': 20},
            {'title': '2010 - 2020', 'key': 'year=ge:2010,le:2020', 'rank': 30},
            {'title': '2000 - 2010', 'key': 'year=ge:2000,le:2010', 'rank': 40},
            {'title': '1980 - 2000', 'key': 'year=ge:1980,le:2000', 'rank': 50},
            {'title': 'до 1980', 'key': 'year=le:1980', 'rank': 60},
        ]
        return {'title': vh_years_filter['title'], 'keys': keys, 'rank': 10}

    def build_kp_rating_filter(self, vh_kp_rating_filter: dict) -> Optional[dict]:
        if vh_kp_rating_filter['type'] != 'range':
            return None
        keys = [
            {'title': '★ 9', 'key': 'kp_rating=ge:9', 'rank': 10},
            {'title': '★ 8', 'key': 'kp_rating=ge:8', 'rank': 20},
            {'title': '★ 7', 'key': 'kp_rating=ge:7', 'rank': 30},
            {'title': 'Спорное', 'key': 'kp_rating=le:7', 'rank': 40},
        ]
        return {'title': vh_kp_rating_filter['title'], 'keys': keys, 'rank': 30}

    filter_mapping = {
        'genre': build_genres_filter,
        'year': build_years_filter,
        'kp_rating': build_kp_rating_filter,
    }

    def __init__(self, carousel_info: VhCarouselInfo):
        self.carousel_info = carousel_info
        self.request = carousel_info.request
        self.headers = carousel_info.headers

    def build_base_url(self):
        args = {
            DroidekaCarousel.FIELD_CAROUSEL_ID: FILTERABLE_CAROUSEL_ID,
            DroidekaCarousel.FIELD_OFFSET: 0,
            DroidekaCarousel.FIELD_LIMIT: self.carousel_info.limit,
            DroidekaCarousel.FIELD_TAG: TAG_BY_CAROUSEL_MAPPING[self.carousel_info.carousel_id]
        }

        return build_view_url(args, self.carousel_info.next_url_name, self.request)

    def get_result(self):
        filters = api.vh.client.carousel_content_filters(self.request, self.headers)
        result_filters = []
        for vh_filter in filters:
            filter_builder = self.filter_mapping.get(vh_filter['id'])
            if filter_builder:
                built_filter = filter_builder(self, vh_filter)
                if built_filter:
                    result_filters.append(built_filter)

        return {'filters': result_filters, 'base_url': self.build_base_url()}


class CarouselRequestInfoDataSource(DataSource):

    def __init__(self, carousel_info):
        self.carousel_info = carousel_info

    def get_result(self):
        return {'carousel_info': self.carousel_info}


class PurchasesDataSource(DataSource):

    def __init__(self, carousel_info: PurchasesCarouselInfo):
        self.carousel_info = carousel_info

    def fill_optional_field(self, params, key, value):
        if value:
            params[key] = value

    def propagate_fields(self, result: dict, carousel_id: str, kp_category: Optional[Category] = None):
        result[DroidekaCarousel.FIELD_CAROUSEL_ID] = carousel_id
        result[api.ott.KEY_SELECTION_ID] = PURCHASES_FAKE_SELECTION_ID
        result[api.ott.KEY_SELECTION_WINDOW_ID] = DROIDEKA_INTERNAL_OTT_WINDOW_ID
        if kp_category:
            self.fill_optional_field(result, 'position', kp_category.internal_position)
            self.fill_optional_field(result, 'title', kp_category.title)

    def find_purchases_category(self):
        kp_categories = self.carousel_info.external_categories_mapping.get(KpCarousel.TYPE)
        if not kp_categories:
            return None
        for category in kp_categories:
            if category.category_id == PURCHASES_CAROUSEL_ID:
                return category
        return None

    def _fill_more_url_for(self, result, offset, limit, next_url_name, available_only, request):
        if not result or not result.get('data') or len(result['data']) < limit:
            return
        args = {
            DroidekaCarousel.FIELD_CAROUSEL_ID: PURCHASES_CAROUSEL_ID,
            DroidekaCarousel.FIELD_CAROUSEL_TYPE: KpCarousel.TYPE,
            DroidekaCarousel.FIELD_LIMIT: limit,
            DroidekaCarousel.FIELD_OFFSET: offset + limit,
        }
        if available_only is not None:
            args[DroidekaCarousel.FIELD_AVAILABLE_ONLY] = available_only

        url = build_view_url(args, next_url_name, request)

        result[DroidekaCarousel.FIELD_MORE_INFO] = {
            DroidekaCarousel.FIELD_MORE_URL: url
        }

    def get_result(self):
        # Purchases needed in 2 cases:
        # 1. PURCHASES_CAROUSEL_ID carousel id requested (from 'carousel' hook)
        # 2. Purchases category presented in external categories mapping (from 'carousels' hook)
        carousel_id = self.carousel_info.carousel_id
        category = None
        if not carousel_id:
            category = self.find_purchases_category()
            carousel_id = category.category_id if category else None
        if not carousel_id:
            # Nor carousel_id or category presented, is presented, then no need to load purchases
            return None

        result = api.ott.client.purchases(
            self.carousel_info.headers,
            self.carousel_info.request,
            self.carousel_info.offset,
            self.carousel_info.limit,
            self.carousel_info.available_only
        )
        self.propagate_fields(result, carousel_id, category)
        self._fill_more_url_for(
            result, self.carousel_info.offset, self.carousel_info.limit, self.carousel_info.next_url_name,
            self.carousel_info.available_only, self.carousel_info.request)
        return result


class UnboundSeasonsQueryConfig:
    """
    Helper for handling 'backward_count' / 'forward_count' params
    'backward_count' / 'forward_count' are params for querying episodes, but we don't know how much episodes every
    season contains
    To know this we need to query seasons first
    To avoid case, when we queried seasons, and understood, and we have no enough seasons and need to query more - we
    consider, that every season in worst case has only 1 episodes. Based on this, we need to query almost same
    amount of seasons, as episodes
    """

    def __init__(self, season_number, backward_episodes_count, forward_episodes_count, stop_at_season_boundaries):
        self.season_number = season_number
        self.backward_count = backward_episodes_count
        self.forward_count = forward_episodes_count
        self.stop_at_season_boundaries = stop_at_season_boundaries

    @property
    def seasons_offset(self):
        if self.stop_at_season_boundaries:
            # if episodes from single season requested, use only index of season
            return max(0, self.season_number - 1)
        return max(0, self.season_number - self.backward_count)

    @property
    def seasons_limit(self):
        if self.stop_at_season_boundaries:
            # if episodes from single season requested, query only 1 season
            return 1
        return self.season_number + self.forward_count


class FillDirection(Enum):
    BACKWARD = 1
    FORWARD = 2


class UnboundSeasonsDataSource(DataSource):

    def __init__(self, data: dict, headers: dict = None, request=None):
        self.seasons_query_config = UnboundSeasonsQueryConfig(
            data['season_number'],
            data['backward_count'],
            data['forward_count'],
            data['stop_at_season_boundaries']
        )
        self.data = data
        self.series_id = self.data['series_id']
        self.pivot_episode_number = self.data['pivot_episode_number']
        self.request = request
        self.headers = headers

    def _get_raw_seasons(self, seasons_response: dict) -> list:
        if not seasons_response or not seasons_response.get('set'):
            raise NotFound(f'No seasons found for series "{self.series_id}"')
        return seasons_response['set']

    def _get_season(self, seasons: Optional[list], season_number):
        if seasons and season_number is not None:
            index = 0
            for season in seasons:
                if season['season_number'] == season_number:
                    return index, season
                index += 1
        raise NotFound(f'Season with number "{season_number}" not presented in "{self.series_id}"')

    def _get_rest_episodes_count(self, fill_direction: FillDirection) -> int:
        if fill_direction == FillDirection.BACKWARD:
            return self.data['backward_count']
        else:
            return self.data['forward_count'] + 1

    def _get_seasons(self, seasons: Optional[list]):
        result = deque()
        pivot_season_index, pivot_season = self._get_season(seasons, self.data['season_number'])
        pivot_episode_index = self.data['pivot_episode_number'] - 1

        if pivot_episode_index > pivot_season['episodes_count'] - 1:
            season_number = self.data['season_number']
            raise NotFound(f'Episode with number "{self.pivot_episode_number + 1}" not presented in season'
                           f' "{season_number}"')

        for fill_direction in FillDirection:
            index = pivot_season_index
            if fill_direction == FillDirection.BACKWARD:
                current_episode_index = pivot_episode_index - 1
            else:
                current_episode_index = pivot_episode_index

            required_episodes_count = self._get_rest_episodes_count(fill_direction)

            season = seasons[index]
            season_episode_count = season['episodes_count']
            while required_episodes_count > 0:
                start_episode = current_episode_index

                if fill_direction == FillDirection.BACKWARD:
                    collected_episodes = min(start_episode + 1, required_episodes_count)
                    if collected_episodes > 0:
                        offset = max(0, start_episode - collected_episodes + 1)
                        result.appendleft({'season_id': season['content_id'], 'offset': offset,
                                          'limit': collected_episodes})
                    index -= 1
                else:
                    collected_episodes = min(required_episodes_count, season_episode_count - start_episode)
                    if collected_episodes > 0:
                        result.append({'season_id': season['content_id'], 'offset': start_episode,
                                       'limit': collected_episodes})
                    index += 1
                if index < 0 or index >= len(seasons) or self.data['stop_at_season_boundaries']:
                    required_episodes_count = 0
                else:
                    required_episodes_count -= collected_episodes
                    season = seasons[index]
                    season_episode_count = season['episodes_count']
                    if fill_direction == FillDirection.BACKWARD:
                        current_episode_index = season_episode_count - 1
                    else:
                        current_episode_index = 0
        return result

    @classmethod
    def merge_same_seasons(cls, seasons: list):
        result = []
        prev_season = None
        for cur_season in seasons:
            if prev_season is not None and prev_season['season_id'] == cur_season['season_id']:
                prev_season.update(season_id=cur_season['season_id'], offset=prev_season['offset'],
                                   limit=prev_season['limit'] + cur_season['limit'])
            else:
                result.append(cur_season)
            prev_season = cur_season
        return result

    def get_result(self):
        seasons_response = api.vh.client.seasons(
            self.request,
            self.headers,
            self.series_id,
            self.seasons_query_config.seasons_limit,
            self.seasons_query_config.seasons_offset)

        seasons = self._get_seasons(self._get_raw_seasons(seasons_response))
        seasons = self.merge_same_seasons(seasons)

        return {'seasons': seasons, 'series_id': self.series_id}


class UnboundEpisodesDataSource(DataSource):

    def __init__(self, seasons: dict, headers: dict, request):
        self.seasons = seasons['seasons']
        self.series_id = seasons['series_id']
        self.headers = headers
        self.request = request

    def episode_sort_key_builder_func(self, e: dict):
        return 100000 * e['season']['season_number'] + e['episode_number']

    def get_filtered_episodes_from_multiple_responses(self, responses):
        result_episodes = []
        for episodes_response in responses:
            episodes = episodes_response.get('set')
            if episodes:
                for episode in episodes:
                    if not episode.get('banned'):
                        result_episodes.append(episode)
        return result_episodes

    def get_result(self):
        if not self.seasons:
            raise NotFound(f'No seasons found for series "{self.series_id}"')
        threads = []
        for season in self.seasons:
            get_series_episodes = partial(
                api.vh.client.series_episodes,
                initial_request=self.request,
                headers=self.headers,
                season_id=season['season_id'],
                limit=season['limit'],
                offset=season['offset'])
            threads.append(YlogGreenlet.spawn(get_series_episodes))
        gevent.wait(threads, timeout=settings.DEFAULT_NETWORK_GREENLET_TIMEOUT)
        responses = get_results_from_greenlets(
            threads,
            error_message='Error loading series episodes',
            timeout_message='Timeout exceed for loading series episodes',
            message_prefix=self.__class__.__name__,
        )

        result_episodes = self.get_filtered_episodes_from_multiple_responses(responses)
        result_episodes.sort(key=self.episode_sort_key_builder_func)

        return result_episodes


class MusicHeadersMixin:
    headers_to_copy = [AUTHORIZATION_HEADER, REQUEST_ID_HEADER, FORWARDED_HEADER,
                       USER_IP_HEADER]

    def get_headers(self):
        return {key: self.headers[key] for key in self.headers_to_copy if key in self.headers}


class MainMusicCarouselDataSource(DataSource, MusicHeadersMixin):
    """
    Music carousels for Home screen
    """
    FIELD_CATEGORY_ID = 'category_id'
    FIELD_ADMIN_TITLE = 'admin_title'
    FIELD_POSITION = 'position'
    FIELD_MUSIC_RESPONSE = 'music_response'

    def __init__(self, request_info: MixedCarouselsInfo):
        self.request_info = request_info
        self.headers = request_info.headers
        self.music_carousels = request_info.external_categories_mapping.get(MusicCarousel.TYPE)

    def get_result(self):
        if not self.music_carousels:
            return {}

        # ask music feed
        headers = self.get_headers()
        response = api.music.client.infinite_feed(headers=headers)

        # and put it as content for every requested injactable carousel
        result = []
        for carousel in self.music_carousels:
            result.append({
                self.FIELD_CATEGORY_ID: carousel.category_id,
                self.FIELD_ADMIN_TITLE: carousel.title,
                self.FIELD_POSITION: carousel.internal_position,
                self.FIELD_MUSIC_RESPONSE: response['result']['rows'][0],
            })

        return result


class MusicSectionDataSource(DataSource, MusicHeadersMixin):
    """
    Music /infinite-feed output for Music section
    """
    def __init__(self, request_info: MixedCarouselsInfo):
        self.request_info = request_info
        self.headers = request_info.headers

    def get_result(self) -> Any:
        headers = self.get_headers()

        response = api.music.client.infinite_feed(headers=headers)

        if 'error' in response:
            error = response.get('error', {})
            if error.get('name') == 'validate' and 'eitherUserId' in error.get('message', ''):
                # show HTTP 403 to client
                raise AuthenticationFailed('Your OAuth token is likely does not have required scopes')

            if error.get('name') == 'session-expired':
                raise AuthenticationFailed('Your OAuth token is likely expired')

        if 'result' not in response or 'error' in response:
            logging.warning('Incorrect unhandled music response: %s', response)

        return response


class TopCarouselPromoDataSource(DataSource):
    """
    Этот дата-сорс не запрашивает данные карточек, он только определяет,
    какие промо-предложения мы дальше будем
    """
    HOTEL_PROMO_REQUEST_INFO = 'hotel_promo_request_info'
    ZALOGIN_PROMO_REQUEST_INFO = 'zalogin_promo_request_info'
    GIFT_PROMO_REQUEST_INFO = 'gift_promo_request_info'
    MUSIC_PROMO_REQUEST_INFO = 'music_promo_request_info'
    OLYMPICS_PROMO_REQUEST_INFO = 'olympics_request_info'

    # экспериментальные флаги для включения промок
    EXP_ZALOGIN_PROMO_TYPES = 'zalogin_promo_types'
    EXP_GIFT_PROMO_TYPES = 'gift_promo_types'
    EXP_MUSIC_PROMO_TYPES = 'music_promo_types'
    EXP_OLYMPICS_PROMO_DISABLED = 'olympics_promo_disabled'

    MUSIC_PROMO_EXPIRATION_TIME_DELTA = datetime.timedelta(days=2)

    def __init__(self, carousel_info: VhCarouselInfo):
        self.carousel_info = carousel_info

    def get_music_promo_expiration_date_timestamp(self, first_show_time_ts: float):
        expire_date_ts = datetime.datetime.fromtimestamp(first_show_time_ts) + datetime.timedelta(days=2)
        return expire_date_ts.timestamp()

    def is_music_promo_expired(self, first_show_time_utc_ts: float, current_time_utc_ts: float) -> bool:
        return bool(first_show_time_utc_ts) and \
            current_time_utc_ts >= self.get_music_promo_expiration_date_timestamp(first_show_time_utc_ts)

    def mark_music_promo_banner_shown(self, current_utc_ts: float):
        user_info: UserInfo = self.carousel_info.request.user_info
        if not user_info.user_ticket:
            return
        request_headers = tvm.add_service_ticket(settings.MEMENTO_CLIENT_ID)
        request_headers[TVM_USER_TICKET] = user_info.user_ticket

        smarttv_music_config = any_pb2.Any()
        smarttv_music_config.Pack(TSmartTvMusicPromoConfig(FirstShowTime=current_utc_ts))

        current_surface_id = get_http_header(self.carousel_info.request, headers_constants.QUASAR_DEVICE_ID) or \
            get_http_header(self.carousel_info.request, headers_constants.DEVICE_ID_HEADER)

        request_data = TReqChangeUserObjects(
            UserConfigs=[
                TConfigKeyAnyPair(
                    Key=EConfigKey.CK_SMART_TV_MUSIC_PROMO,
                    Value=smarttv_music_config)],
            CurrentSurfaceId=current_surface_id
        )
        try:
            api.memento.client.update_objects(request_data, request_headers)
        except (ConnectionError, HTTPError):
            logger.error('Error saving first show time for music banner')

    def fill_hotel_request_info(self, result: dict) -> bool:
        device_ids = cache.budapest_device_ids
        device_id = get_http_header(self.carousel_info.request, QUASAR_DEVICE_ID)
        if not device_id or not device_ids or device_id not in device_ids:
            logger.debug('No device id provided. Skip hotel promo')
            return False
        result[self.HOTEL_PROMO_REQUEST_INFO] = HotelRequestInfo(device_id)
        return True

    def get_promo_variants(self, raw_value: Optional[str]) -> Optional[List[str]]:
        if not raw_value:
            return None
        return [variant.strip() for variant in raw_value.split(',')]

    def fill_promo_params_from_experiment(self, result: dict, exp_key: str) -> bool:
        promo_variants = self.get_promo_variants(self.carousel_info.request.request_info.experiments.get_value(exp_key))

        if not promo_variants:
            logger.debug('No "%s" key in experiment data', exp_key)
            return False

        # todo: hellodima - перенести эту логику на уровень выше
        if exp_key == self.EXP_ZALOGIN_PROMO_TYPES:
            result[self.ZALOGIN_PROMO_REQUEST_INFO] = PromoRequestInfo(Promo.ZALOGIN, promo_variants)
            return True
        elif exp_key == self.EXP_GIFT_PROMO_TYPES:
            result[self.GIFT_PROMO_REQUEST_INFO] = PromoRequestInfo(Promo.GIFT, promo_variants)
            return True
        elif exp_key == self.EXP_MUSIC_PROMO_TYPES:
            result[self.MUSIC_PROMO_REQUEST_INFO] = PromoRequestInfo(Promo.MUSIC, promo_variants)
            return True

        return False

    def fill_zalogin_request_info(self, result: dict) -> bool:
        if self.carousel_info.request.request_info.authorized:
            return False
        return self.fill_promo_params_from_experiment(result, self.EXP_ZALOGIN_PROMO_TYPES)

    def fill_gift_request_info(self, result: dict) -> bool:
        return self.fill_promo_params_from_experiment(result, self.EXP_GIFT_PROMO_TYPES)

    def fill_music_request_info(self, result: dict) -> bool:
        if not self.carousel_info.request.request_info.authorized or \
           ParsedVersion(self.carousel_info.request.platform_info.app_version) < VER_1_5:
            return False
        music_promo_config = self.carousel_info.request.request_info.memento_configs.music_promo_config
        if not music_promo_config:
            logger.info('Memento config wasn\'t loaded')
            return False
        first_show_timestamp = music_promo_config.FirstShowTime
        uts_ts = int(datetime.datetime.utcnow().timestamp())
        if self.is_music_promo_expired(first_show_timestamp, uts_ts):
            return False
        if not first_show_timestamp:
            self.mark_music_promo_banner_shown(uts_ts)
        return self.fill_promo_params_from_experiment(result, self.EXP_MUSIC_PROMO_TYPES)

    def fill_olympics_request_info(self, result: dict) -> bool:
        if ParsedVersion(self.carousel_info.request.platform_info.app_version) < VER_2_100_12:
            logger.info('Olympics banner disabled because of old client version')
            return False

        if self.carousel_info.request.request_info.experiments.get_value(self.EXP_OLYMPICS_PROMO_DISABLED):
            logger.info('Olympics banner disabled by exp key "%s"', self.EXP_OLYMPICS_PROMO_DISABLED)
            return False

        logger.info('Olympics banner enabled for this client (this is default)')
        # пусть покажется любой баннер из категории olympics
        result[self.OLYMPICS_PROMO_REQUEST_INFO] = PromoRequestInfo(Promo.OLYMPICS, [])
        return True

    def get_result(self) -> dict:
        result = {}
        if self.fill_hotel_request_info(result):
            return result
        if self.fill_zalogin_request_info(result):
            return result
        if self.fill_olympics_request_info(result):
            return result
        if self.fill_music_request_info(result):
            return result
        self.fill_gift_request_info(result)
        return result


class CustomPromoDataSource(DataSource):
    """
    Этот датасорс загружает из базы промку, которая задана в request_info
    типом и возможными айдишникам
    """

    def __init__(self, request_info: Optional[PromoRequestInfo]):
        self.request_info = request_info

    def choose_promo_id(self) -> Optional[str]:
        promo_ids = self.request_info.promo_ids
        if not promo_ids:
            return None
        return random.choice(promo_ids)

    @staticmethod
    def find_promo(promo_type: str, promo_id: str) -> Optional[Promo]:
        promos = get_promos()
        for promo in promos:
            if promo.promo_type == promo_type and promo.promo_id == promo_id:
                return promo
        return None

    def get_result(self) -> dict:
        if not self.request_info:
            return {}
        promo = self.find_promo(self.request_info.promo_type, self.choose_promo_id())
        if not promo:
            return {}
        result = {'auto_fields': {'promo_type': promo.promo_type}}
        if promo:
            result.update(PromoSerializer(promo).data)
        return result


class RandomPromoFromCategory(DataSource):
    """
    Этот датасорс выбирает одно рандомное промо из всех с заданным promo_type
    """
    required_fields = ('title', 'subtitle', 'imageUrl', 'infoUrl')

    def __init__(self, request_info: Optional[PromoRequestInfo]):
        self.request_info = request_info

    def get_result(self) -> dict:
        if not self.request_info or not self.request_info.promo_type:
            logger.debug('Random promo not requested')
            return {}

        category = self.request_info.promo_type  # ex: olympics

        all_promos = get_promos()
        matched = []
        for p in all_promos:
            if p.promo_type == category and p.enabled:
                matched.append(p)

        logger.info('Found %d promos in random promo category %s', len(matched), category)
        if not matched:
            return {}

        promo = random.choice(matched)
        data = PromoSerializer(promo).data

        if promo.is_olympics():
            # fixing bug on client, see smarttvbackend-1077 for details
            data['content_id'] = promo.fixed_content_id

        # по этому полю дальше определяется сериализатор
        result = {'auto_fields': {'promo_type': category}}
        result.update(data)

        return result


class TrailerSignDataSource(DataSource):

    ALLOWED_STREAMS = ('HLS', 'DASH')  # Order matters

    def __init__(self, trailer_vh_uuid: Optional[str], document_onto_id: Optional[str], headers: dict, request):
        super().__init__()
        self.trailer_vh_uuid = trailer_vh_uuid
        self.document_onto_id = document_onto_id
        self.headers = headers
        self.request = request

    def get_result(self):
        if not self.trailer_vh_uuid:
            return None
        response = api.vh.client.content_detail(
            initial_request=self.request,
            headers=self.headers,
            content_id=self.trailer_vh_uuid
        )
        if not response:
            logger.info('No streams found for trailer: %s, film id: %s', self.trailer_vh_uuid, self.document_onto_id)
            return None

        try:
            streams = response['content']['streams']
        except (TypeError, KeyError):
            logger.info('No streams found for trailer: %s, film id: %s', self.trailer_vh_uuid, self.document_onto_id)
            return None
        if not streams:
            logger.info('No streams found for trailer: %s, film id: %s', self.trailer_vh_uuid, self.document_onto_id)
            return None

        streams = {stream['stream_type']: stream['url'] for stream in streams if stream['stream_type'] in self.ALLOWED_STREAMS}

        for stream_type in self.ALLOWED_STREAMS:
            if stream_type in streams:
                return streams[stream_type]

        logger.error('No supported streams for trailer %s, film onto id: %s', self.trailer_vh_uuid, self.document_onto_id)
        return None


class FilmsToWatchDataSource(DataSource):
    def __init__(self, content_id, headers, content_detail, request):
        self.content_id = content_id
        self.headers = headers
        self.content_detail = content_detail
        self.request = request

    def get_result(self) -> dict:
        content_type = get_content_type(self.content_detail)
        if AUTHORIZATION_HEADER not in self.headers:
            logger.debug('Skipping will watch request for user without authorization')
            return {}

        if api.vh.is_ott_content(content_type):
            return api.ott.client.films_to_watch(content_id=self.content_id, initial_request=self.request,
                                                 headers=self.headers)

        logger.debug('{} is not OTT content, skip checking films to watch'.format(self.content_id))
        return {}
