import logging
from typing import Iterable, Union

from django.core.files.uploadedfile import UploadedFile
from django.db import transaction
from smarttv.droideka.proxy.models import ValidIdentifier
from smarttv.droideka.proxy.serializers.serializers import IdentifierSerializer, IdentifierValidator
from smarttv.utils.machelpers import normalize_mac, validate_mac

logger = logging.getLogger(__name__)
BULK_CREATE_BATCH_SIZE = 1000


class ImportResult:
    def __init__(self):
        self.status = ''
        self.imported = 0
        self.imported_unique = 0
        self.skipped = 0
        self.warnings = []

    def as_dict(self):
        return {
            'status': self.status,
            'imported': self.imported,
            'imported_unique': self.imported_unique,
            'skipped': self.skipped,
            'warnings': self.warnings,
        }


class IdentifiersImporter:

    class InvalidMacType(ValueError):
        pass

    class NoMacs(ValueError):
        pass

    @classmethod
    def check_identifiers_type(cls, identifier_type: str) -> None:
        if identifier_type not in ValidIdentifier.TYPES:
            logger.error('Unknown identifier type "%s". Must be one of: %s', identifier_type,
                         ', '.join(ValidIdentifier.TYPES))
            raise IdentifiersImporter.InvalidMacType()
        logger.info('Identifier type: %s', type)

    @classmethod
    def read_file(cls, file: UploadedFile):
        result = []
        for i, line in enumerate(file):
            if i == 0:  # first line
                _type = line.decode().strip()
                cls.check_identifiers_type(_type)
                continue

            result.append({'type': _type, 'value': line.decode().strip()})

        if not result:
            raise IdentifiersImporter.NoMacs()

        return result

    @classmethod
    def create_models(cls, raw_identifiers, raise_on_error=True):
        if not raw_identifiers:
            raise IdentifiersImporter.NoMacs()
        serializer = IdentifierSerializer(instance=raw_identifiers, many=True)
        validator = IdentifierValidator(data=serializer.data, many=True)
        validator.is_valid(raise_exception=raise_on_error)

        return validator.data

    @staticmethod
    def save_identifiers(identifiers: list) -> int:
        before = ValidIdentifier.objects.count()
        with transaction.atomic():
            ValidIdentifier.objects.bulk_create(identifiers, batch_size=BULK_CREATE_BATCH_SIZE, ignore_conflicts=True)
            after = ValidIdentifier.objects.count()
        return after - before  # сколько добавилось (исключая дубли)

    @staticmethod
    def import_from_string(lines: Iterable[Union[bytes, str]]) -> ImportResult:
        """
        Импортирует маки из списка строк
        """

        result = ImportResult()
        models = []

        for line in lines:
            if isinstance(line, bytes):
                line = line.decode()
            line = line.strip()
            if not line:
                continue

            # валидация
            mac = normalize_mac(line)
            if not validate_mac(mac):
                result.skipped += 1
                result.warnings.append(f'Incorrect MAC "{line}" was skipped')
                continue

            models.append(ValidIdentifier(type=ValidIdentifier.ETHERNET_MAC, value=mac))

        logger.debug('Inserting %d identifiers', len(models))
        if result.warnings:
            logger.info('Warnings: %r', result.warnings)

        result.imported_unique = IdentifiersImporter.save_identifiers(models)

        result.imported = len(models)
        result.status = 'ok'

        return result
