import logging
from typing import Optional
from collections.abc import Iterable

logger = logging.getLogger(__name__)


FIELD_LICENSES = 'licenses'
FIELD_OTT_PARAMS = 'ottParams'
FIELD_INCLUDES = 'includes'
FIELD_MONETIZATION_MODEL = 'monetizationModel'
FIELD_PURCHASE_TAG = 'purchaseTag'
FIELD_ACTIVE = 'active'


class LicenseInfo:

    # Mapping for legacy 'ya_plus' values from 'purchase_tag' field for old clients
    svod_purchase_tag_legacy_mapping = {
        'plus': ('YA_PLUS',),
        'kp-basic': ('KP_BASIC',),
        'kp-amediateka': ('YA_PREMIUM',),
        'super-plus': ('YA_PLUS_SUPER',),
    }

    def __init__(self, licenses: Optional[Iterable] = None):
        self._licenses = licenses or []

        if licenses:
            self._monetization_models = (
                [item[FIELD_MONETIZATION_MODEL] for item in licenses if FIELD_MONETIZATION_MODEL in item])
        else:
            self._monetization_models = []

    def get_license_with_key(self, key_name) -> Optional[dict]:
        for licence in self._licenses:
            if key_name in licence:
                return licence
        return None

    @property
    def licenses(self) -> Iterable:
        return self._licenses

    @property
    def monetization_models(self) -> list:
        return self._monetization_models

    def get_primary(self) -> Optional[dict]:
        return self.get_license_with_key('primary')

    def get_legacy_ya_plus_array(self) -> Optional[Iterable]:
        """
        This method generates `ya_plus` array from `ottParams` field for backwards compatibility, since VH will not send
        `ya_plus`

        This field is required only for `SVOD` content
        """
        if not self._licenses:
            return None
        primary_license = self.get_primary()
        monetization_model = primary_license.get(FIELD_MONETIZATION_MODEL)
        if monetization_model == 'SVOD':
            purchase_tag = primary_license.get(FIELD_PURCHASE_TAG)
            ya_plus_value = self.svod_purchase_tag_legacy_mapping.get(purchase_tag)
            if not ya_plus_value:
                logger.warning('Unknown purchase tag: %s', purchase_tag)
            return ya_plus_value
        return None


def get_monetization_model_from(licenses: Iterable) -> list:
    if licenses:
        return [item[FIELD_MONETIZATION_MODEL] for item in licenses if item.get(FIELD_MONETIZATION_MODEL)]
    return []


def get_licenses(obj) -> LicenseInfo:
    """
    Tries to get license from object or from first episode of series
    """
    if not obj:
        return LicenseInfo()
    ott_params = None
    if FIELD_OTT_PARAMS in obj:
        ott_params = obj[FIELD_OTT_PARAMS]
    elif obj.get(FIELD_INCLUDES) and FIELD_OTT_PARAMS in obj[FIELD_INCLUDES][0]:
        ott_params = obj[FIELD_INCLUDES][0][FIELD_OTT_PARAMS]
    if not ott_params:
        return LicenseInfo()
    if FIELD_LICENSES in ott_params:
        licenses = ott_params[FIELD_LICENSES]
    else:
        return LicenseInfo()
    return LicenseInfo(licenses)
