import json
import logging
import os
import shutil
import tempfile
import time

from django.core.management.base import BaseCommand

from smarttv.utils.headers import USER_AGENT_HEADER

from smarttv.droideka.proxy.api import vh
from smarttv.droideka.proxy.serializers.fast_response import ProgramsSerializer
from smarttv.droideka.utils import caching_date

logger = logging.getLogger(__name__)


class Command(BaseCommand):
    help = 'Periodic job for downloading programs from VH'

    ARG_SLEEP = 'sleep'

    HEADERS = {
        USER_AGENT_HEADER: 'com.yandex.tv.input.efir/1.4.0 (Realtek SmartTV; Android 9)'
    }
    CHANNELS_LIMIT = 300

    TEMP_JSON_FILE_NAME = 'programs.json'

    programs_serializer = ProgramsSerializer()

    @classmethod
    def _get_channel_ids(cls):
        channels_response = vh.channel_client.channels_regions(
            initial_request=None,
            headers=cls.HEADERS,
            with_filtration=0)
        channel_content_ids = \
            set(channel['content_id'] for channel in channels_response['set'] if channel.get('has_schedule'))
        logger.info('Got %s channels', len(channel_content_ids))
        return channel_content_ids

    @classmethod
    def load_programs_for_interval(cls, channel_id, from_time, to_time):
        program_response = vh.channel_client.episodes(
            initial_request=None,
            headers=cls.HEADERS,
            from_time=from_time,
            to_time=to_time,
            parent_id=channel_id,
            limit=cls.CHANNELS_LIMIT,
        )
        vh.fake_stream_url(program_response)
        result = program_response['set']
        logger.info('Loaded %s for channel %s, interval %s - %s', len(result), channel_id, from_time, to_time)
        return result

    def load_programs(self, channel_content_ids, target_intervals, sleep_time):
        result = {}
        logger.info('Start loading programs')
        for content_id in channel_content_ids:
            programs = []
            for interval_start, interval_end in target_intervals:
                response_programs = self.load_programs_for_interval(content_id, interval_start, interval_end)
                if sleep_time:
                    time.sleep(sleep_time)
                serialized_programs = self.programs_serializer.serialize_many(response_programs)
                if serialized_programs:
                    programs.extend(serialized_programs)
            result[content_id] = programs
            logger.info('Loaded %s programs, for %s', len(programs), content_id)
        logger.info('Programs loaded')
        return result

    def add_arguments(self, parser):
        parser.add_argument(f'--{self.ARG_SLEEP}', help='Sleep time between requests(in seconds)', required=False)

    @classmethod
    def save_programs(cls, programs):
        json_file_name = cls.TEMP_JSON_FILE_NAME
        json_dir_path = tempfile.mkdtemp()
        json_file_path = os.path.join(json_dir_path, json_file_name)
        with open(json_file_path, 'w') as f:
            f.write(json.dumps(programs, ensure_ascii=False))
        return shutil.make_archive(
            os.path.join(json_dir_path, 'programs'),
            'zip',
            json_dir_path,
            cls.TEMP_JSON_FILE_NAME)

    def handle(self, *args, **options):
        sleep_time = options.get(self.ARG_SLEEP, None)
        channel_content_ids = self._get_channel_ids()
        target_intervals = caching_date.split_time_interval(caching_date.yesterday_start(), caching_date.tomorrow_end())
        programs = self.load_programs(channel_content_ids, target_intervals, sleep_time)
        programs_zip_file = self.save_programs(programs)

        self.stdout.write(programs_zip_file)
