import json
import logging
from typing import Iterable

import cityhash
import requests

from django.core.management.base import BaseCommand
from furl import furl

logger = logging.getLogger(__name__)


# see arcadia/extsearch/video/kernel/recommender/util/base26/base26.h
class Base26:
    @staticmethod
    def encode_digit(digit: int) -> str:
        assert digit <= 25
        return chr(ord('a') + digit)

    @staticmethod
    def encode(value: int, decorator: str) -> str:
        result = decorator
        while value > 0:
            result = result + Base26.encode_digit(value % 26)
            value //= 26

        return result + decorator


class Carousel:
    def __init__(self, selection_id: str, window_id: str):
        self.selection_id = selection_id
        self.window_id = window_id

    @property
    def id(self):
        return '%s_%s' % (self.window_id, self.selection_id)


kp_carousels = (
    Carousel(selection_id='5e3b4065a77eac00232c2d49', window_id='EXTERNAL_PLUS'),
    Carousel(selection_id='5e3bfb22a77eac00232c2d5b', window_id='EXTERNAL')
)


def get_ott_url(carousel: Carousel) -> str:
    f = furl('https://api.ott.yandex.ru/v9/selections')
    f.add(path=carousel.selection_id)
    f.args['serviceId'] = 167
    f.args['offset'] = 0
    f.args['limit'] = 50
    f.args['selectionWindowId'] = carousel.window_id

    return f.url


def get_ott_response(url: str, headers: dict) -> dict:
    response = requests.get(url, headers=headers)
    response.raise_for_status()

    return json.loads(response.content.decode('utf-8'))


def gen_carousel_term(carousel_id):
    # see https://a.yandex-team.ru/arc/trunk/arcadia/extsearch/video/kernel/tags/tags.cpp?rev=6575813#L120
    h = cityhash.hash64(carousel_id.encode()) >> 1
    return Base26.encode(h, 'hh')


def get_result(carousels_data: Iterable[Carousel]) -> dict:
    carousels = []
    headers = {'Accept': 'application/json'}

    for item in carousels_data:
        ott_response = get_ott_response(get_ott_url(item), headers)

        carousel = {
            'title': ott_response['title'],
            'carousel_id': item.id,
            'carousel_term': gen_carousel_term(item.id),
            'set': [{'uuid': item['filmId']} for item in ott_response['data']]
        }

        carousels.append(carousel)

    return {'carousels': carousels}


class Command(BaseCommand):
    help = 'Generate json file with kinopoisk carousels stubs data to upload to vh-team'

    def add_arguments(self, parser):
        parser.add_argument('--output', help='Path to file with result', required=True)

    def handle(self, *args, **options):
        logger.info('Starting...')
        filename = options['output']

        with open(filename, 'w') as fp:
            result = get_result(kp_carousels)
            fp.write(json.dumps(result, ensure_ascii=False, indent=4))

        logger.info('Done')
