import logging
from django.core import exceptions, validators
from django.core.management.base import BaseCommand
from django.db import transaction

from smarttv.droideka.proxy.models import ValidIdentifier


logger = logging.getLogger(__name__)


BATCH_SIZE = 1000
mac_address_validator = validators.RegexValidator(r'([0-9a-f]{2}:){5}([0-9a-f]{2})')


class Command(BaseCommand):

    help = 'Upload valid identifiers (in lowercase)'

    def add_arguments(self, parser):
        parser.add_argument('--file', help='Path to file with identifiers to upload', required=True)

    def handle(self, *args, **options):
        logger.info('Start uploading identifiers to database')

        filename = options['file'].strip()
        logger.info('Reading data from file: %s', filename)

        with open(filename) as f:
            # reading first line with type of identifier
            identifier = f.readline().strip().lower()
            if identifier not in ValidIdentifier.TYPES:
                logger.error('Unknown identifier "%s". Must be one of: %s', identifier, ', '.join(ValidIdentifier.TYPES))
                return
            logger.info('Identifier type: %s', identifier)

            object_list = []
            for row in f:
                value = row.strip().lower()

                if identifier in (ValidIdentifier.WIFI_MAC, ValidIdentifier.ETHERNET_MAC):
                    try:
                        mac_address_validator(value)
                    except exceptions.ValidationError:
                        logger.error('Invalid mac address "%s". Must be like ff:ff:ff:ff:ff:ff', value)
                        return

                object_list.append(ValidIdentifier(type=identifier, value=value))

            with transaction.atomic():
                ValidIdentifier.objects.bulk_create(object_list, batch_size=BATCH_SIZE)

            logger.info('Successfully uploaded %s identifiers', len(object_list))
