import logging
import hashlib
import uuid
import json

from datetime import datetime
from dateutil import tz
from typing import Optional
from cache_memoize import cache_memoize

from smarttv.droideka.proxy import cache
from django.db import models
from django.utils import timezone
from django.conf import settings
from django.core.validators import RegexValidator, MinValueValidator

from smarttv.droideka.utils import PlatformType, PlatformInfo
from smarttv.droideka.utils.serialization import SerializableMixin
from smarttv.droideka.proxy.constants.carousels import KpCarousel, KpCategory, MusicCarousel, MusicCategory, VhFeed, \
    VhCarousel, EMBEDDED_SECTION_TYPE, COMPOSITE_FEED, CarouselType

logger = logging.getLogger(__name__)

DEFAULT_TZ = tz.gettz('Europe/Moscow')


def get_entity_actual_version(entity):
    version = Versions.objects.using(settings.DB_REPLICA).filter(entity=entity).first()
    if not version:
        logger.error("Actual version not set for %s", entity)
    else:
        logger.debug("Actual version for %s is %d", version.entity, version.version)
    return version


class EditableModelMixin:

    def to_publishable_model(self):
        return None


class Versions(models.Model):
    """
    Contains actual data versions for every table
    """
    entity = models.CharField('Entity', max_length=128, db_index=True)
    version = models.IntegerField('Version')

    class Meta:
        unique_together = [['entity', 'version'], ]
        verbose_name_plural = 'Versions'


class EntityMixin:
    """
    Basic model which provides it's entity name
    By the entity - the actual version can be obtained from 'Versions' table
    """

    @classmethod
    def get_entity(cls):
        return str(cls.__name__)


class PlatformModel(models.Model, EntityMixin, SerializableMixin):
    """
    Represents a target platform - client with specific characteristics
    Attributes:
        platform_type (str): type of client (android)
        platform_name (str): string that represents this type of platform
        platform_version (str): version of platform
        app_version (str): the version of apk
        quasar_platform (str): platform in quasar
    """
    PLATFORM_VERSION_PATTERN = r'^(\d+(\.\d+)(\.\d+)?)?$'
    APP_VERSION_PATTERN = r'^(\d+(.\d+))?$'

    PLATFORM_VERSION_VALIDATOR = RegexValidator(regex=PLATFORM_VERSION_PATTERN)
    APP_VERSION_VALIDATOR = RegexValidator(regex=APP_VERSION_PATTERN)
    PLATFORM_CHOICES = (
        (PlatformType.ANDROID, PlatformType.ANDROID),
        (PlatformType.ANY, PlatformType.ANY),
        (PlatformType.NONE, PlatformType.NONE),
    )
    platform_type = models.CharField('Тип платформы', choices=PLATFORM_CHOICES, max_length=32, null=False, blank=False,
                                     unique=False)
    platform_name = models.CharField('Название платформы', max_length=64, blank=True, unique=False, default='')
    platform_version = models.CharField('Версия платформы', max_length=64, blank=True, unique=False,
                                        validators=[PLATFORM_VERSION_VALIDATOR])
    app_version = models.CharField('Версия APK (version code)', max_length=64, blank=True, unique=False, default='',
                                   validators=[APP_VERSION_VALIDATOR])
    device_manufacturer = models.CharField('Производитель устройства', max_length=64, blank=True, unique=False,
                                           default='')
    device_model = models.CharField('Модель устройства', max_length=64, blank=True, unique=False,
                                    default='')
    quasar_platform = models.CharField('Платформа в квазаре', max_length=64, blank=True, unique=False,
                                       default='')

    def to_json(self) -> dict:
        result = {
            PlatformType.KEY: self.platform_type
        }
        mapping = (
            (PlatformInfo.KEY_PLATFORM_VERSION, self.platform_version),
            (PlatformInfo.KEY_APP_VERSION, self.app_version),
            (PlatformInfo.KEY_DEVICE_MANUFACTURER, self.device_manufacturer),
            (PlatformInfo.KEY_DEVICE_MODEL, self.device_model),
            (PlatformInfo.KEY_QUASAR_PLATFORM, self.quasar_platform),
        )
        for key, value in mapping:
            if value:
                result[key] = value

        return result

    def __str__(self):
        if self.platform_type == PlatformType.ANY:
            return 'Any'
        elif self.platform_type == PlatformType.NONE:
            return 'None'
        else:
            return (f'type: {self.platform_type}, '
                    f'name: {self.platform_name}, '
                    f'version: {self.platform_version}, '
                    f'app version: {self.app_version}, '
                    f'device manufacturer: {self.device_manufacturer}, '
                    f'device model: {self.device_model}'
                    f'quasar platform: {self.quasar_platform}')

    class Meta:
        ordering = ('platform_type', 'platform_name', 'platform_version')
        verbose_name_plural = 'Platforms'


class CategoryExtendedDataMixin(models.Model):
    CATEGORY_ID_MAX_LENGTH = 64

    CONTENT_TYPE_CHOICES = [
        ('', 'Side menu item'),
        (KpCarousel.TYPE, 'Kinopoisk carousel'),
        (KpCategory.TYPE, 'Kinopoisk category'),
    ]

    category_id = models.CharField('ID категории', max_length=CATEGORY_ID_MAX_LENGTH, primary_key=True)
    title = models.CharField('Имя категории', max_length=128)
    icon_s3_key = models.TextField('Key S3 иконки', blank=True, default='')
    rank = models.IntegerField('Целое число для задания порядка элементов(чтобы можно было посортировать)',
                               null=True, blank=True, validators=[MinValueValidator(1)])
    position = models.IntegerField('Целое число для задания абсолютной позиции в выдаче, куда встраивается категория',
                                   null=True, blank=True, validators=[MinValueValidator(1)])
    parent_category = models.ForeignKey('self', on_delete=models.CASCADE, null=True, default=None, blank=True)
    thumbnail_s3_key = models.TextField('Ключ S3 превью', null=True, blank=True)
    logo_s3_key = models.TextField('Ключ S3 логотипа', blank=True, default='')
    banner_S3_key = models.TextField('Key S3 баннера', blank=True, default='')
    description = models.TextField('Описание категории', blank=True, default='')
    content_type = models.TextField('Тип категории', blank=True, default='', choices=CONTENT_TYPE_CHOICES)
    exclude_platforms = models.ManyToManyField(PlatformModel,
                                               related_name='%(app_label)s_%(class)s_exclude_platforms',
                                               related_query_name='%(app_label)s_%(class)ss_exclude')
    include_platforms = models.ManyToManyField(PlatformModel,
                                               related_name='%(app_label)s_%(class)s_include_platforms',
                                               related_query_name='%(app_label)s_%(class)ss_include')
    above_platforms = models.ManyToManyField(PlatformModel,
                                             related_name='%(app_label)s_%(class)s_above_platforms',
                                             related_query_name='%(app_label)s_%(class)ss_above')
    below_platforms = models.ManyToManyField(PlatformModel,
                                             related_name='%(app_label)s_%(class)s_below_platforms',
                                             related_query_name='%(app_label)s_%(class)ss_below')

    @property
    def internal_position(self):
        """
        Actual position in result
        field 'position' in admin start from 1
        This field starts from 0(since objects in python's list numerated from zero, and we need to specify position
            in python's list, but for human it's not comfortable)
        """
        if self.position:
            return self.position - 1
        return self.position

    class Meta:
        abstract = True


class CategoryExtended(CategoryExtendedDataMixin):

    def __str__(self):
        return f'Category id: {self.category_id}, parent id: {self.parent_category}'

    class Meta:
        ordering = ('position', 'rank')
        verbose_name_plural = 'Extended published categories'


class CategoryExtendedEditable(CategoryExtendedDataMixin, EditableModelMixin):
    visible = models.BooleanField(null=False, default=True)

    def to_publishable_model(self):
        return CategoryExtended(
            category_id=self.category_id,
            title=self.title,
            icon_s3_key=self.icon_s3_key,
            rank=self.rank,
            position=self.position,
            parent_category_id=self.parent_category_id,
            thumbnail_s3_key=self.thumbnail_s3_key,
            logo_s3_key=self.logo_s3_key,
            banner_S3_key=self.banner_S3_key,
            description=self.description,
            content_type=self.content_type,
        )

    def __str__(self):
        parent_category_id = self.parent_category.category_id if self.parent_category else None
        return f'CategoryExtendedEditable[category_id:{self.category_id}, parent_category:{parent_category_id}]'

    class Meta:
        ordering = ('position', 'rank')
        verbose_name_plural = 'Extended editable categories'


class CategoryExperiment(models.Model):
    value = models.CharField('Значение флага `category_experiments`', null=False, primary_key=True, max_length=64)
    description = models.TextField('Описание', blank=True, default='')


class Category2DataMixin(models.Model):
    """
    Contains modifications of 'CategoryExtendedDataMixin':
    - created auto-generated primary key: 'id'

    Created for backwards compatibility purpose
    """
    CATEGORY_ID_MAX_LENGTH = 64

    CONTENT_TYPE_CHOICES = [
        (VhFeed.TYPE, 'VH category item'),
        (VhCarousel.TYPE, 'VH carousel'),
        (KpCarousel.TYPE, 'Kinopoisk carousel'),
        (KpCategory.TYPE, 'Kinopoisk category'),
        (EMBEDDED_SECTION_TYPE, 'Embedded section'),
        (COMPOSITE_FEED, 'Composite feed'),  # feed, with fixed amount of carousels, specified on admin page
        (MusicCarousel.TYPE, 'Music main carousel'),
        (MusicCategory.TYPE, 'Music category'),
    ]

    CAROUSEL_TYPE_CHOICES = [
        ('', ''),
        (CarouselType.TYPE_SQUARE, CarouselType.TYPE_SQUARE),
    ]

    category_id = models.CharField('ID категории в сторонней системе', max_length=CATEGORY_ID_MAX_LENGTH)
    title = models.CharField('Имя категории', max_length=128)
    icon_s3_key = models.TextField('Key S3 иконки', blank=True, default='')
    rank = models.IntegerField('Целое число для задания порядка элементов(чтобы можно было посортировать)',
                               null=True, blank=True, validators=[MinValueValidator(1)])
    position = models.IntegerField('Целое число для задания абсолютной позиции в выдаче, куда встраивается категория',
                                   null=True, blank=True, validators=[MinValueValidator(1)])
    parent_category = models.ForeignKey('self', on_delete=models.CASCADE, null=True, default=None, blank=True)
    thumbnail_s3_key = models.TextField('Ключ S3 превью', null=True, blank=True)
    logo_s3_key = models.TextField('Ключ S3 логотипа', blank=True, default='')
    banner_S3_key = models.TextField('Key S3 баннера', blank=True, default='')
    description = models.TextField('Описание категории', blank=True, default='')
    content_type = models.TextField('Тип категории', blank=True, default='', choices=CONTENT_TYPE_CHOICES)
    carousel_type = models.TextField('Тип карусели(внешний вид)', blank=True, default='', choices=CAROUSEL_TYPE_CHOICES)
    authorization_required = models.BooleanField('Только авторизованным пользователям', default=False, null=False)
    show_in_tandem = models.BooleanField('Показывать эту категорию на модуле в режиме тандема', default=True, null=False)
    category_experiments = models.ManyToManyField(CategoryExperiment, blank=True)
    category_disable_experiments = models.ManyToManyField(
        CategoryExperiment,
        blank=True,
        related_name='%(app_label)s_%(class)s_category_disable_experiments',
        related_query_name='%(app_label)s_%(class)ss_negative_experiments')
    persistent_client_category_id = models.TextField('Неизменяемый ID категории для клиента', blank=True, default='')
    exclude_platforms = models.ManyToManyField(PlatformModel,
                                               related_name='%(app_label)s_%(class)s_exclude_platforms',
                                               related_query_name='%(app_label)s_%(class)ss_exclude')
    include_platforms = models.ManyToManyField(PlatformModel,
                                               related_name='%(app_label)s_%(class)s_include_platforms',
                                               related_query_name='%(app_label)s_%(class)ss_include')
    above_platforms = models.ManyToManyField(PlatformModel,
                                             related_name='%(app_label)s_%(class)s_above_platforms',
                                             related_query_name='%(app_label)s_%(class)ss_above')
    below_platforms = models.ManyToManyField(PlatformModel,
                                             related_name='%(app_label)s_%(class)s_below_platforms',
                                             related_query_name='%(app_label)s_%(class)ss_below')

    @property
    def internal_position(self):
        """
        Actual position in result
        field 'position' in admin start from 1
        This field starts from 0(since objects in python's list numerated from zero, and we need to specify position
            in python's list, but for human it's not comfortable)
        """
        if self.position:
            return self.position - 1
        return self.position

    class Meta:
        abstract = True


class Category2(Category2DataMixin, SerializableMixin):
    KEY_CATEGORY_ID = 'category_id'
    KEY_TITLE = 'title'
    KEY_ICON_S3_KEY = 'icon_s3_key'
    KEY_RANK = 'rank'
    KEY_POSITION = 'position'
    KEY_PARENT_CATEGORY_ID = 'parent_category_id'
    KEY_PARENT_CATEGORY = 'parent_category'
    KEY_THUMBNAIL_S3_KEY = 'thumbnail_s3_key'
    KEY_LOGO_S3_KEY = 'logo_s3_key'
    KEY_BANNER_S3_KEY = 'banner_S3_key'
    KEY_DESCRIPTION = 'description'
    KEY_CONTENT_TYPE = 'content_type'
    KEY_CAROUSEL_TYPE = 'carousel_type'
    KEY_AUTHORIZATION_REQUIRED = 'authorization_required'
    KEY_SHOW_IN_TANDEM = 'show_in_tandem'
    KEY_CATEGORY_EXPERIMENTS = 'category_experiments'
    KEY_CATEGORY_DISABLE_EXPERIMENTS = 'category_disable_experiments'
    KEY_EXCLUDE_PLATFORMS = 'exclude_platforms'
    KEY_INCLUDE_PLATFORMS = 'include_platforms'
    KEY_ABOVE_PLATFORMS = 'above_platforms'
    KEY_BELOW_PLATFORMS = 'below_platforms'
    KEY_PERSISTENT_CLIENT_CATEGORY_ID = 'persistent_client_category_id'

    EMBEDDED_CAROUSEL_TYPE = 'embedded_carousel'
    EMBEDDED_CATEGORY_TYPE = 'embedded_category'

    display_content_type_mapping = {
        VhFeed.TYPE: EMBEDDED_CATEGORY_TYPE,
        KpCategory.TYPE: EMBEDDED_CATEGORY_TYPE,
        COMPOSITE_FEED: EMBEDDED_CATEGORY_TYPE,
        VhCarousel.TYPE: EMBEDDED_CAROUSEL_TYPE,
        KpCarousel.TYPE: EMBEDDED_CAROUSEL_TYPE,
    }

    id = models.IntegerField('Внутренний ID категории', primary_key=True)

    @property
    def display_content_type(self) -> Optional[str]:
        return self.display_content_type_mapping.get(self.content_type)

    def to_json(self) -> dict:
        result = {
            self.KEY_CATEGORY_ID: self.category_id,
            self.KEY_TITLE: self.title,
            self.KEY_ICON_S3_KEY: self.icon_s3_key,
            self.KEY_RANK: self.rank,
            self.KEY_POSITION: self.position,
            self.KEY_PARENT_CATEGORY_ID: self.parent_category.category_id if self.parent_category else None,
            self.KEY_THUMBNAIL_S3_KEY: self.thumbnail_s3_key,
            self.KEY_LOGO_S3_KEY: self.logo_s3_key,
            self.KEY_BANNER_S3_KEY: self.banner_S3_key,
            self.KEY_DESCRIPTION: self.description,
            self.KEY_CONTENT_TYPE: self.content_type,
            self.KEY_CAROUSEL_TYPE: self.carousel_type,
            self.KEY_AUTHORIZATION_REQUIRED: self.authorization_required,
            self.KEY_SHOW_IN_TANDEM: self.show_in_tandem,
            self.KEY_CATEGORY_EXPERIMENTS: [experiment.value for experiment in self.category_experiments.all()],
            self.KEY_CATEGORY_DISABLE_EXPERIMENTS: [experiment.value for experiment in self.category_disable_experiments.all()],
            self.KEY_EXCLUDE_PLATFORMS: [platform.to_json() for platform in self.exclude_platforms.all()],
            self.KEY_INCLUDE_PLATFORMS: [platform.to_json() for platform in self.include_platforms.all()],
            self.KEY_ABOVE_PLATFORMS: [platform.to_json() for platform in self.above_platforms.all()],
            self.KEY_BELOW_PLATFORMS: [platform.to_json() for platform in self.below_platforms.all()],
            self.KEY_PERSISTENT_CLIENT_CATEGORY_ID: self.persistent_client_category_id,
        }
        return result

    def __str__(self):
        return f'Category id: {self.category_id}, parent id: {self.parent_category_id}'

    class Meta:
        ordering = ('position', 'rank')
        verbose_name_plural = 'Published categories'


class Category2Editable(Category2DataMixin):
    id = models.AutoField('Внутренний ID категории', primary_key=True)
    visible = models.BooleanField(null=False, default=True)

    def to_publishable_model(self):
        return Category2(
            id=self.id,
            category_id=self.category_id,
            title=self.title,
            icon_s3_key=self.icon_s3_key,
            rank=self.rank,
            position=self.position,
            parent_category_id=self.parent_category_id,
            thumbnail_s3_key=self.thumbnail_s3_key,
            logo_s3_key=self.logo_s3_key,
            banner_S3_key=self.banner_S3_key,
            description=self.description,
            content_type=self.content_type,
            carousel_type=self.carousel_type,
            authorization_required=self.authorization_required,
            show_in_tandem=self.show_in_tandem,
            persistent_client_category_id=self.persistent_client_category_id,
        )

    def __str__(self):
        parent_category_id = self.parent_category.category_id if self.parent_category else None
        return f'Category2Editable[category_id:{self.category_id}, parent_category:{parent_category_id}]'

    class Meta:
        ordering = ('position', 'rank')
        verbose_name_plural = 'Editable categories'


class ScreenSaver(models.Model, EntityMixin):
    """
    Represents the screen to be shown when TV in standby mode

    Attributes:
        title (str): human readable string to be displayed as name of screensaver in the UI
        type (str): human readable string that describes the type of screensaver(image | video)
        resolution (str): digital characteristic type of content (HD | FHD | QHD)
        s3_key (str): S3 key of the image or video
        rank (int): soring key for client

        visible (bool): indicates is content visible to clients or not
        version (int): version of record. Related to 'Versions' table
    """
    TYPE_CHOICES = [
        ('image', 'image'),
        ('video', 'video'),
    ]

    RESOLUTION_CHOICES = [
        ('HD', 'HD'),
        ('FHD', 'FHD'),
        ('QHD', 'QHD'),
    ]

    title = models.CharField('Имя объекта', max_length=256, null=True)
    type = models.CharField('Тип контента', max_length=32, null=False, choices=TYPE_CHOICES)
    resolution = models.CharField('Тип разрешения', max_length=32, null=False, choices=RESOLUTION_CHOICES)
    s3_key = models.CharField('Ключ контента на S3', max_length=256, null=False)
    rank = models.IntegerField('Позиция в выдаче', null=False)
    visible = models.BooleanField(null=False, default=True)
    version = models.IntegerField('Version')

    class Meta:
        ordering = ('rank',)
        unique_together = [['s3_key', 'version']]
        verbose_name_plural = 'Screensavers'


def calculate_hardware_id(ethernet_mac, wifi_mac):
    def prepare(mac):
        return mac.replace(':', '').lower() if mac else 'null'

    return hashlib.md5(f'{prepare(ethernet_mac)}{prepare(wifi_mac)}'.encode('ascii')).hexdigest()


class Device(models.Model):
    hardware_id = models.CharField('ID телевизора', max_length=256, null=False, unique=True)
    serial_number = models.CharField('Серийный номер', max_length=256, null=True, unique=False)
    wifi_mac = models.CharField('WiFi MAC адрес', max_length=256, null=False)
    ethernet_mac = models.CharField('Ethernet MAC адрес', max_length=256, null=False)
    subscription_puid = models.BigIntegerField('PUID подарочной подписки', null=True, default=None)
    promocode = models.CharField('Промокод подарочной подписки', max_length=256, null=True, default=None)
    created_at = models.DateTimeField(default=timezone.now)
    kp_gifts_id = models.UUIDField(null=False, default=uuid.uuid4)
    kp_gifts_given = models.BooleanField(default=False)

    def is_gift_available(self):
        return bool(self.subscription_puid is None and not self.kp_gifts_given)

    def get_kp_gifts_id(self):
        return self.kp_gifts_id.hex

    def save(self, *args, **kwargs):
        if not self.hardware_id:
            self.hardware_id = calculate_hardware_id(self.ethernet_mac, self.wifi_mac)
        return super(Device, self).save(*args, **kwargs)

    class Meta:
        indexes = [
            models.Index(fields=['hardware_id']),
            models.Index(fields=['serial_number']),
            models.Index(fields=['wifi_mac']),
            models.Index(fields=['ethernet_mac']),
            models.Index(fields=['kp_gifts_id']),
        ]


class SupportDeviceProxy(Device):
    class Meta:
        proxy = True


class ValidIdentifier(models.Model):
    ETHERNET_MAC = 'ethernet_mac'
    SERIAL_NUMBER = 'serial_number'
    WIFI_MAC = 'wifi_mac'
    TYPES = (ETHERNET_MAC, SERIAL_NUMBER, WIFI_MAC)

    TYPE_CHOICES = (
        (ETHERNET_MAC, 'Ethernet MAC'),
        (SERIAL_NUMBER, 'Serial number'),
        (WIFI_MAC, 'Wi-Fi MAC'),
    )

    type = models.CharField('Тип индентификатора', choices=TYPE_CHOICES, max_length=32, null=False)
    value = models.CharField('Значение', max_length=256, null=False)

    def save(self, force_insert=False, force_update=False, using=None, update_fields=None):
        self.value = self.value.lower()
        super().save(force_insert, force_update, using, update_fields)

    class Meta:
        unique_together = [['type', 'value']]

    def __str__(self):
        return f'{self.type}: {self.value}'


class SharedPreferences(models.Model):
    DATETIME_PREFIX = 'datetime:'

    current_proc_cache = dict()

    key = models.CharField('Ключ', max_length=512, null=False, unique=True, primary_key=True)
    tag = models.CharField('Тег', max_length=512, help_text='Тег для группировки флагов', default='', blank=True)
    int_value = models.IntegerField('int value', null=True, blank=True)
    bool_value = models.NullBooleanField('boolean value', null=True, blank=True)
    char_value = models.TextField('string value', default='', blank=True)
    datetime_value = models.DateTimeField('date time value', null=True, blank=True)
    is_json = models.BooleanField('Is JSON', null=False, blank=False, default=False)

    class Meta:
        verbose_name_plural = 'SharedPreferences'

    @classmethod
    def _save_in_cache(cls, key, int_value=None, bool_value=None, char_value='', datetime_value=None):
        if int_value is not None:
            value = int_value
        elif bool_value is not None:
            value = bool_value
        elif datetime_value is not None:
            value = cls._get_datetime_redis_representation(datetime_value)
        elif char_value:
            value = char_value
        else:
            value = None
        if value is not None:
            # тут баг: булево value сохраняется как текст
            cache.unversioned.set(key=key, value=str(value))
        else:
            cache.unversioned.delete(key)

    def save(self, force_insert=False, force_update=False, using=None, update_fields=None):
        self._save_in_cache(
            key=self.key,
            int_value=self.int_value,
            bool_value=self.bool_value,
            char_value=self.char_value,
            datetime_value=self.datetime_value
        )
        super().save(force_insert, force_update, using, update_fields)

    def delete(self, using=None, keep_parents=False):
        cache.unversioned.delete(self.key)
        return super().delete(using, keep_parents)

    @classmethod
    def _get_datetime_redis_representation(cls, dt_value):
        return cls.DATETIME_PREFIX + str(int(dt_value.timestamp()))

    @classmethod
    def _get_datetime_from_string_representation(cls, dt_str_value):
        return datetime.fromtimestamp(int(dt_str_value.replace(cls.DATETIME_PREFIX, '')))

    @classmethod
    def _get_datetime_pref(cls, key, pref_value):
        if pref_value is str:
            if pref_value.startswith(cls.DATETIME_PREFIX):
                return cls._get_datetime_from_string_representation(pref_value)
            else:
                return None
        else:
            cache.unversioned.set(
                key,
                cls._get_datetime_redis_representation(pref_value))
            return pref_value

    @classmethod
    def _get_existing_pref(cls, key) -> Optional['SharedPreferences']:
        return SharedPreferences.objects.filter(key=key).first()

    @classmethod
    def _save_safe(cls, key, int_value=None, bool_value=None, char_value='', datetime_value=None, is_json=False):
        if not char_value and is_json:
            is_json = False
        existing_pref = cls._get_existing_pref(key)
        if not existing_pref:
            SharedPreferences(
                key=key,
                int_value=int_value,
                char_value=char_value,
                bool_value=bool_value,
                datetime_value=datetime_value,
                is_json=is_json
            ).save()
        else:
            existing_pref.int_value = int_value
            existing_pref.char_value = char_value
            existing_pref.bool_value = bool_value
            existing_pref.datetime_value = datetime_value
            existing_pref.is_json = False
            existing_pref.save()

    @classmethod
    def get_pref_value_from_db(cls, key, type_cast):
        pref = SharedPreferences.objects.using(settings.DB_REPLICA).filter(key=key).first()
        if pref is None:
            return None
        if type_cast == int:
            return pref.int_value
        elif type_cast == bool:
            return pref.bool_value
        elif type_cast == str:
            return pref.char_value
        elif type_cast == datetime:
            return pref.datetime_value.astimezone(DEFAULT_TZ).replace(tzinfo=None)
        elif type_cast is not None:
            raise ValueError(f"Unknown 'type_cast': {type_cast}")
        return None

    @classmethod
    @cache_memoize(timeout=settings.DEFAULT_RESPONSE_CACHE_TIME, cache_alias='local')
    def get_pref_value_from_cache(cls, key):
        pref_value = cache.unversioned.get(key)
        return pref_value

    @classmethod
    def delete_pref(cls, key):
        db_objects = SharedPreferences.objects.filter(key=key)
        if db_objects:
            db_objects.delete()
        cache.unversioned.delete(key)

    @classmethod
    def get_pref(cls, key, type_cast=None):
        pref_value = cls.get_pref_value_from_cache(key)
        logger.debug("Preference value for key '%s' from cache: '%s'", key, pref_value)
        if pref_value is None:
            pref_value = cls.get_pref_value_from_db(key, type_cast)
            logger.debug("Preference value for key '%s' from DB: '%s'", key, pref_value)
            if pref_value is None:
                return None
            if type_cast == datetime:
                return cls._get_datetime_pref(key, pref_value)
            cache.unversioned.set(key, pref_value)
        if type_cast:
            if type_cast == datetime:
                return cls._get_datetime_from_string_representation(pref_value)
            try:
                return type_cast(pref_value)
            except ValueError:
                logger.error('Can not cast %s with key %s to %s', pref_value, key, type_cast)
                return None
        return pref_value

    @classmethod
    def get_int(cls, key):
        return cls.get_pref(key, type_cast=int)

    @classmethod
    def get_bool(cls, key):
        return cls.get_pref(key, type_cast=bool)

    @classmethod
    def get_datetime(cls, key):
        return cls.get_pref(key, type_cast=datetime)

    @classmethod
    def get_string(cls, key):
        value = cls.get_pref(key, type_cast=str)
        if value:
            return value
        return None

    @classmethod
    def get_json(cls, key):
        value = cls.get_string(key)
        if not value:
            return None
        try:
            return json.loads(value)
        except json.decoder.JSONDecodeError:
            logger.warning('Can not parse shared preference "%s" as JSON', key)
            return None

    @classmethod
    def put_int(cls, key, value):
        cls._save_safe(key, int_value=value)

    @classmethod
    def put_bool(cls, key, value):
        cls._save_safe(key, bool_value=value)

    @classmethod
    def put_string(cls, key, value):
        cls._save_safe(key, char_value=value)

    @classmethod
    def put_datetime(cls, key, value):
        cls._save_safe(key, datetime_value=value)

    @classmethod
    def put_json(cls, key, value):
        cls._save_safe(key, char_value=json.dumps(value, ensure_ascii=False), is_json=True)


class Promo(models.Model):
    ZALOGIN = 'zalogin'
    GIFT = 'gift'
    MUSIC = 'music'
    OLYMPICS = 'olympics'

    TYPE_CHOICES = (
        (ZALOGIN, 'Login'),
        (GIFT, 'Unused gift'),
        (MUSIC, 'Music'),
        (OLYMPICS, 'Olympic games 2022')
    )

    CONTENT_TYPE_CHOICES = (
        ('action', 'Диплинк(другое)'),
    )

    # значение из action_value имеет приоритет над этим мапингом
    action_mapping = {
        ZALOGIN: 'home-app://profile',
        GIFT: 'home-app://gift_promo',
        MUSIC: 'home-app://category?category_id=music',
    }

    enabled = models.BooleanField(default=True)
    promo_id = models.CharField('ID промо', max_length=64, primary_key=True)
    promo_type = models.CharField('Тип промо', choices=TYPE_CHOICES, max_length=64, null=False, blank=False)
    content_type = models.CharField('Клиентское поле, чтобы клиент понимал как работать с этой карточкой',
                                    max_length=64, null=False, blank=False, choices=CONTENT_TYPE_CHOICES)
    thumbnail = models.CharField('thumbnail карточки(в аватарнице)', null=False, blank=False, max_length=512)
    title = models.CharField('Главный текст карточки', null=False, blank=True, default='', max_length=64)
    subtitle = models.CharField('Второстепенный текст карточки', null=False, blank=True, default='', max_length=64)
    action_value = models.CharField('Поле action', null=False, blank=True, default='', max_length=512)
    fallback_action = models.CharField('Поле fallback_action', null=False, blank=True, default='', max_length=512)

    @property
    def content_id(self):
        return f'{self.promo_type}/{self.promo_id}'

    @property
    def action(self):
        if self.action_value:
            return self.action_value
        return self.action_mapping[self.promo_type]

    def is_olympics(self):
        return self.promo_type == self.OLYMPICS

    @property
    def fixed_content_id(self):
        return f'{self.promo_type}/fixed'


class RecommendedChannel(models.Model):
    title = models.CharField('Название (видит только админ)', max_length=64)
    channel_id = models.IntegerField('ID канала в VH')
    rank = models.IntegerField('Ключ сортировки', default=0)
    visible = models.BooleanField(default=True)

    def __str__(self):
        return self.title


class SmotreshkaChannel(models.Model):
    CATEGORY_CHOICES = (
        ('inform', 'Новостные'),
        ('films', 'Кино и сериалы'),
        ('educate', 'Познавательные'),
        ('child', 'Детские'),
        ('entertain', 'Развлекательные'),
        ('music', 'Музыкальные'),
        ('sport', 'Спортивные'),
        ('federal', 'Федеральные'),
        ('foreign', 'Международные'),
        ('region', 'Региональные'),
        ('smotreshka', 'Эфирные (Смотрёшка)'),
    )

    def __str__(self):
        return self.title

    title = models.CharField('Название', max_length=255)
    enabled = models.BooleanField('Включен', default=False)
    smotreshka_channel_id = models.CharField('ID канала в смотрешке', max_length=100)
    channel_id = models.PositiveIntegerField('ID канала (как в вх)')
    category = models.CharField('Категория', max_length=64, blank=False, choices=CATEGORY_CHOICES)
    number = models.PositiveIntegerField('Номер')
    main_color = models.CharField('Цвет заливки иконки', max_length=10, blank=True)
    thumbnail = models.CharField('Картинка для карточки', max_length=500, blank=True)
    smarttv_icon = models.CharField('Иконка', max_length=500, blank=True)


class Cinema(models.Model):
    def __str__(self):
        return self.code

    code = models.CharField('Код', max_length=255, primary_key=True)
    enabled = models.BooleanField('Включен', default=False)
    icon = models.CharField('Иконка', max_length=500, blank=True)
    created_at = models.DateTimeField('Создан', null=True, auto_now_add=True)
