from threading import Timer
import gevent
import logging
from geobase6 import Lookup

from django.conf import settings

from smarttv.droideka.proxy import cache
from smarttv.droideka.proxy import api
from smarttv.droideka.proxy import common
from smarttv.droideka.proxy import mem_cache


logger = logging.getLogger(__name__)


def log_lifecycle(target: str):
    def decorator(function):
        def wrapper(*args, **kwargs):
            logger.info('Start loading %s', target)
            result = function(*args, **kwargs)
            logger.info('End loading %s', target)
            return result
        return wrapper
    return decorator


class CacheUpdateTask:
    timer = None
    is_started = False
    cache_filled = False

    @log_lifecycle('budapest ids')
    def load_budapest_device_ids(self):
        cache.budapest_device_ids = api.alice.client.devices()

    @log_lifecycle('geobase')
    def load_geobase(self):
        common.geobase = Lookup(settings.GEODATA_BIN_PATH)

    @log_lifecycle('content id mapping')
    def load_content_id_mapping(self):
        mem_cache.onto_id_mapping = mem_cache.read_json('ONTO_ID_MAPPING_PATH')

    @log_lifecycle('content type mapping')
    def load_content_type_mapping(self):
        mem_cache.content_type_mapping = mem_cache.read_json('CONTENT_TYPE_MAPPING_PATH')

    def update_cache(self):
        logger.info('Started periodic task')
        greenlets = [
            gevent.Greenlet.spawn(self.load_budapest_device_ids),
            gevent.Greenlet.spawn(self.load_geobase),
            gevent.Greenlet.spawn(self.load_content_id_mapping),
            gevent.Greenlet.spawn(self.load_content_type_mapping),
        ]
        gevent.wait(greenlets, timeout=settings.PERIODIC_TASK_WAIT_TIMEOUT)
        for greenlet in greenlets:
            if not greenlet.ready() or not greenlet.successful():
                raise ValueError('Not all periodic tasks are successful')
        logger.debug('Cache updated successfully')
        if not self.cache_filled:
            self.cache_filled = True

    def timer_callback(self):
        """
        Container for all other tasks
        """
        logger.debug('Timer callback called')
        try:
            self.update_cache()
        finally:
            self.timer = Timer(interval=settings.PERIODIC_TASK_INTERVAL, function=self.timer_callback)
            self.timer.daemon = True
            self.timer.start()

    def start(self):
        if self.is_started:
            logger.debug('Timer already started')
            return
        self.is_started = True
        self.update_cache()
        self.timer = Timer(interval=settings.PERIODIC_TASK_INTERVAL, function=self.timer_callback)
        self.timer.daemon = True
        self.timer.start()


cache_update_task = CacheUpdateTask()
