import logging
from collections import defaultdict

from smarttv.droideka.proxy.licenses import get_licenses
from smarttv.droideka.proxy.api import vh
from smarttv.droideka.proxy.vh.constants import FIELD_LEGAL, FIELD_VH_LICENSES, FIELD_CONTENT_TYPE, \
    TRAILER_CONTENT_TYPES

logger = logging.getLogger(__name__)


class EntitySearchFieldClip:
    FIELD_PLAYER_ID = 'PlayerId'
    FIELD_IS_AVOD = 'is_avod'
    FIELD_IS_TVOD = 'is_tvod'
    FIELD_IS_SVOD = 'is_svod'
    FIELD_IS_EST = 'is_est'


class PlayerDetector:
    """
    Detects which player to use to play requested episode
    """
    KINOPOISK_PLAYER = 'kinopoisk'
    OTT_PLAYER = 'ott'
    VH_PLAYER = 'vh'
    WEB_VIEW_PLAYER = 'web'
    SMOTRESHKA_PLAYER = 'smotreshka'

    KEY_PLAYER_ID = 'player_id'

    oo_clips_player_mapping = defaultdict(lambda: PlayerDetector.WEB_VIEW_PLAYER)
    oo_clips_player_mapping['ott'] = OTT_PLAYER
    oo_clips_player_mapping['vh'] = VH_PLAYER

    SUBSCRIPTION_MONETIZATION_MODELS = {'AVOD', 'SVOD'}

    @classmethod
    def _has_monetization_models(cls, target_monetization_models, licenses):
        for monetization_model in licenses:
            if monetization_model in target_monetization_models:
                return True
        return False

    @classmethod
    def _get_licences_from_reach_info(cls, rich_info):
        if rich_info is None:
            return None
        if 'vh_meta' not in rich_info:
            return None
        vh_meta = rich_info['vh_meta']
        if 'content_groups' not in vh_meta:
            return None
        content_groups = vh_meta['content_groups']
        if not content_groups:
            return None
        content_group = content_groups[0]
        if content_group.get(FIELD_CONTENT_TYPE) in TRAILER_CONTENT_TYPES:
            return cls.VH_PLAYER
        licences = content_group.get('child_licenses') or content_group.get('licenses')
        if not licences:
            return None
        for licence in licences:
            if licence['monetization_model'] == 'AVOD':
                return cls.OTT_PLAYER
        return cls.KINOPOISK_PLAYER

    @classmethod
    def _is_kinopoisk_monetization_model(cls, monetization_models: list) -> bool:
        return monetization_models and ('TVOD' in monetization_models or 'EST' in monetization_models or
                                        'SVOD' in monetization_models)

    @classmethod
    def get_player_id_for_clip(cls, **kwargs):
        """
        Result of video search has next structure:

        {
          'entity_data': {...},
          'clips': [
            {
            ...,
            'PlayerId',
            ...
            },
            ...
          ]
        }

        Target field is 'PlayerId'. It can have many values. For example: ott, vh, youtube, ok,...
        So, this values must be mapped in next way:
        ott   -> ott
        vh    -> vh
        other -> web

        To determine that object is a clip, field 'PlayerId' must be presented
        If object is not a clip - 'None' returned
        """
        if EntitySearchFieldClip.FIELD_PLAYER_ID not in kwargs:
            return None
        player_id = cls.oo_clips_player_mapping[kwargs[EntitySearchFieldClip.FIELD_PLAYER_ID]]
        if player_id == cls.OTT_PLAYER or player_id == cls.VH_PLAYER:
            if kwargs.get(
                EntitySearchFieldClip.FIELD_IS_SVOD
            ) or kwargs.get(
                EntitySearchFieldClip.FIELD_IS_TVOD
            ) or kwargs.get(
                EntitySearchFieldClip.FIELD_IS_EST
            ):
                # For any paid content return kinopoisk's player id
                return cls.KINOPOISK_PLAYER
            if kwargs.get(EntitySearchFieldClip.FIELD_IS_AVOD):
                return cls.OTT_PLAYER
        return player_id

    @classmethod
    def get_player_id_for_assoc(cls, **kwargs):
        """
        Associations objects stored in:
        (oo response / search response) -> entity_data -> related_object -> [type: assoc] -> object ->
        [i]

        If it has 'legal' field, 'legal' -> 'vh_licenses' object has field 'avod' - then OTT player should be used
        if it has 'legal' field, and no avod(i.e. it has any other, like 'est', 'svod', 'tvod', ...) - kinopoisk
        otherwise - 'None' is returned
        """
        try:
            vh_licenses = kwargs[FIELD_LEGAL][FIELD_VH_LICENSES]
        except KeyError:
            return None
        if vh_licenses.get('avod'):
            return cls.OTT_PLAYER
        return cls.KINOPOISK_PLAYER

    @classmethod
    def get_player_id_for_entity_data(cls, rich_info=None, **kwargs):
        player_id = cls._get_licences_from_reach_info(rich_info)
        if player_id:
            return player_id
        if kwargs:
            return cls.get_player_id_for_assoc(**kwargs.get('base_info', {}))
        return None

    @classmethod
    def get_player_id_for_carousel_item(cls,
                                        content_type_name=None,
                                        content_id=None,
                                        object_response=None,
                                        **kwargs):
        if content_type_name and content_id and object_response is None:
            if content_type_name in (vh.CONTENT_TYPE_SERIES, vh.CONTENT_TYPE_EPISODE):
                license_info = get_licenses(kwargs)
                monetization_models = license_info.monetization_models
                if cls._is_kinopoisk_monetization_model(monetization_models):
                    return cls.KINOPOISK_PLAYER
                elif 'blogger_id' in kwargs or 'blogger' in kwargs or 'onto_category' not in kwargs:
                    # if response of VH's hook 'player' doesn't contain 'onto_category' field - means that it's not
                    # licensed document(it VH clip), for this document VH player should be used
                    # if response of VH's hook 'player' has fields 'blogger_id' or 'blogger'. Here, also use - VH player
                    return cls.VH_PLAYER
                return cls.OTT_PLAYER
            elif content_type_name in ('channel', 'episode'):
                return cls.VH_PLAYER
        return PlayerDetector.WEB_VIEW_PLAYER

    @classmethod
    def get_player_id_for_card_detail(cls,
                                      content_id=None,
                                      onto_id=None,
                                      content_detail=None,
                                      object_response=None,
                                      ott_metadata=None,
                                      **kwargs):
        logger.debug('[for_card_detail] %s, vh: %s, ott: %s', content_id, bool(content_detail), bool(ott_metadata))
        if not content_detail and ott_metadata and not object_response:
            return cls.KINOPOISK_PLAYER
        player_id = None
        if content_detail:
            player_id = cls.get_player_id_for_carousel_item(**content_detail)
        if player_id and (player_id != PlayerDetector.WEB_VIEW_PLAYER or not object_response):
            return player_id
        if object_response:
            player_id = cls.get_player_id_for_entity_data(**object_response)
        if player_id and player_id != PlayerDetector.WEB_VIEW_PLAYER:
            return player_id

        content_type_name = kwargs.get('content_type_name') if kwargs else None
        logger.info("Can't detect required player (%s, %s, %s)",
                    content_id,
                    onto_id,
                    content_type_name)
        return cls.WEB_VIEW_PLAYER

    @classmethod
    def get_player_id_for_kp_carousel_item(self,
                                           monetizationModels,
                                           **kwargs):
        if 'AVOD' in monetizationModels:
            return self.OTT_PLAYER
        return self.KINOPOISK_PLAYER


class SearchRequestPlayerIdFiller:
    """
    Fills all the items in search request that requires player id field
    """

    @staticmethod
    def fill_player_id_for_base_info(entity_data):
        if entity_data:
            player_id = PlayerDetector.get_player_id_for_entity_data(**entity_data)
            if not player_id:
                return
            base_info = entity_data.get('base_info')
            if not base_info:
                return
            base_info[PlayerDetector.KEY_PLAYER_ID] = player_id

    @staticmethod
    def fill_player_id_for_assocs(assocs):
        assocs = assocs or []
        for association in assocs:
            player_id = PlayerDetector.get_player_id_for_assoc(**association)
            if player_id:
                association[PlayerDetector.KEY_PLAYER_ID] = player_id

    @staticmethod
    def fill_player_id_for_clips(clips):
        if not clips:
            return
        for clip in clips:
            logger.debug('PlayerId: %s', clip.get('PlayerId'))
            player_id = PlayerDetector.get_player_id_for_clip(**clip)
            if player_id:
                clip[PlayerDetector.KEY_PLAYER_ID] = player_id
            else:
                logger.warning("Can't detect required player")

    @staticmethod
    def fill_player_id_for_parent_collection(parent_collection):
        if parent_collection:
            for obj in parent_collection:
                player_id = PlayerDetector.get_player_id_for_assoc(**obj)
                if player_id:
                    obj[PlayerDetector.KEY_PLAYER_ID] = player_id
