import logging
from abc import ABC, abstractmethod
from typing import Optional
from collections import defaultdict

from smarttv.droideka.proxy.constants.carousels import VhFeed, KpCarousel, DroidekaCarousel, DroidekaCategory

logger = logging.getLogger(__name__)


class PaginationData:
    """
    Represents some range of data in "endless" feed with unique id

    :param offset:           defines the start of the range
    :param limit:            defines how much data to retrieve from feed
    :param feed_identifier:  optional parameter, to stabilize the feed. Since feed - usually some recommended list - it
                             can be changed from time to time. This parameter helps to query certain feed with fixed for
                             this id ranking
    """

    def __init__(self, offset: int, limit: int, feed_identifier: Optional[str] = None):
        self.limit = limit
        self.offset = offset
        self.feed_identifier = feed_identifier


class MixedCarouselsPaginationData(PaginationData):
    """
    Has additional pagination parameter to allow to paginate over mixed carousel's data sources
    """

    def __init__(self, offset: int, limit: int, external_carousel_offset: int, feed_identifier: Optional[str] = None):
        super().__init__(offset, limit, feed_identifier)
        self.external_carousel_offset = external_carousel_offset


class IdentificationData:
    """
    Represents unique identifiers of some feed

    :param primary_id:    usually only this identifier is being used
    :param secondary_id:  some types of feed requires second identifier. But it's optional, and usually - never used
    """

    def __init__(self, primary_id: str, secondary_id: Optional[str] = None):
        self.primary_id = primary_id
        self.secondary_id = secondary_id


class MoreUrlRequestData:
    """
    Contains general data for building "more_url" - url for retrieving next portion of data
    """

    def __init__(self, request, next_url_name: str):
        self.request = request
        self.next_url_name = next_url_name


class MoreUrlResponseData:
    """
    Information for building "more_url", which can be obtained only from response

    :param loaded_items_size:    indicates how much items were loaded
    :param new_feed_identifier:  sometimes, after retrieving new portion of feed, it's required to update feed
                                 identifier
    """

    def __init__(self, loaded_items_size: int, new_feed_identifier: str):
        self.loaded_items_size = loaded_items_size
        self.new_feed_identifier = new_feed_identifier


class MoreUrlMixedResponseData(MoreUrlResponseData):
    """
    Additional information for building "more_url" for specific case, when we have multiple data sources

    :param external_loaded_items_size:
    """

    def __init__(self, loaded_items_size: int, new_feed_identifier: str, external_loaded_items_size: int):
        super().__init__(loaded_items_size, new_feed_identifier)
        self.external_loaded_items_size = external_loaded_items_size


class NextUrlInfo(ABC):
    @property
    @abstractmethod
    def next_url_name(self) -> str:
        pass

    @property
    @abstractmethod
    def nested_next_url_name(self) -> str:
        pass


class VhCarouselsInfo(NextUrlInfo, ABC):

    @property
    @abstractmethod
    def category_id(self) -> str:
        pass

    @property
    @abstractmethod
    def offset(self) -> Optional[int]:
        pass

    @property
    @abstractmethod
    def limit(self) -> int:
        pass

    @property
    @abstractmethod
    def cache_hash(self) -> Optional[str]:
        pass

    @property
    @abstractmethod
    def max_item_count(self) -> int:
        pass

    @property
    @abstractmethod
    def headers(self) -> dict:
        pass

    @property
    @abstractmethod
    def request(self):
        pass

    @property
    @abstractmethod
    def restriction_age(self) -> int:
        pass


class VhCarouselsInfoImpl(VhCarouselsInfo):

    def __init__(self,
                 serialized_request_data: dict,
                 next_url_name: str,
                 nested_next_url_name: str,
                 request,
                 headers: dict):
        self._identification_data = IdentificationData(primary_id=serialized_request_data['category_id'])
        self._pagination_data = PaginationData(
            offset=serialized_request_data['offset'],
            limit=serialized_request_data['limit'],
            feed_identifier=serialized_request_data.get(VhFeed.FIELD_CACHE_HASH)
        )
        self._restriction_age = serialized_request_data.get('restriction_age')
        self._max_item_count = serialized_request_data['max_items_count']
        self._more_url_request_data = MoreUrlRequestData(request, next_url_name)
        self._nested_next_url_name = nested_next_url_name
        self._headers = headers

    @property
    def category_id(self) -> str:
        return self._identification_data.primary_id

    @property
    def offset(self) -> Optional[int]:
        return self._pagination_data.offset

    @property
    def limit(self) -> int:
        return self._pagination_data.limit

    @property
    def cache_hash(self) -> Optional[str]:
        return self._pagination_data.feed_identifier

    @property
    def max_item_count(self) -> int:
        return self._max_item_count

    @property
    def headers(self) -> dict:
        return self._headers

    @property
    def request(self):
        return self._more_url_request_data.request

    @property
    def next_url_name(self) -> str:
        return self._more_url_request_data.next_url_name

    @property
    def nested_next_url_name(self) -> str:
        return self._nested_next_url_name

    @property
    def restriction_age(self) -> Optional[int]:
        return self._restriction_age


class CarouselTypeMixin(ABC):
    @property
    @abstractmethod
    def carousel_type(self) -> str:
        pass


class ExternalCarouselInfo(NextUrlInfo, ABC):

    @property
    @abstractmethod
    def external_carousel_offset(self) -> int:
        pass

    @property
    @abstractmethod
    def current_external_carousel_amount(self) -> str:
        pass

    @property
    @abstractmethod
    def external_categories_mapping(self):
        pass


class MixedCarouselsInfo(ExternalCarouselInfo, VhCarouselsInfoImpl):
    FIELD_KEY = 'mixed_carousels_info'

    def __init__(self,
                 serialized_data: dict,
                 request,
                 headers: dict,
                 next_url_name: str,
                 nested_next_url_name: str,
                 **kwargs):
        super().__init__(serialized_data, next_url_name, nested_next_url_name, request, headers)
        self._external_carousel_offset = serialized_data['external_carousel_offset']
        self.categories = []
        self._external_categories_mapping = defaultdict(list)
        self.carousels_next_url_name = kwargs.get('carousels_next_url_name')
        self.carousel_next_url_name = kwargs.get('carousel_next_url_name')
        self.category_content_type = kwargs.get('category_content_type')
        self.purchases_available_only = serialized_data.get('purchases_available_only')
        self.external_categories_limit = 0

    @property
    def external_carousel_offset(self) -> int:
        return self._external_carousel_offset

    @property
    def vh_offset(self) -> int:
        result = self.offset - self.external_carousel_offset
        if result < 0:
            raise ValueError('Offset can not be less than 0')
        return result

    @property
    def vh_limit(self) -> int:
        result = self.limit - len(self.categories)
        if result < 0:
            raise ValueError('Limit can not be less than 0')
        return result

    @property
    def current_external_carousel_amount(self) -> int:
        return len(self.categories)

    @property
    def external_categories_mapping(self):
        return self._external_categories_mapping

    def extend_categories(self, categories: list):
        self.categories.extend(categories)
        mapping = self._external_categories_mapping
        self.external_categories_limit += len(categories)
        for category in categories:
            content_type = category.content_type
            mapping[content_type].append(category)


class VhCarouselInfo:
    def __init__(self, serialized_data: dict, request, headers: dict, next_url_name: str):
        self.identification_data = IdentificationData(serialized_data[DroidekaCarousel.FIELD_CAROUSEL_ID])
        self.pagination_data = PaginationData(
            serialized_data[DroidekaCarousel.FIELD_OFFSET],
            serialized_data[DroidekaCarousel.FIELD_LIMIT],
            serialized_data.get(DroidekaCarousel.FIELD_CACHE_HASH)
        )
        self.more_url_data = MoreUrlRequestData(request, next_url_name)
        self._headers = headers
        self._restriction_age = serialized_data.get(DroidekaCarousel.FIELD_RESTRICTION_AGE)
        self.filter = serialized_data.get(DroidekaCarousel.FIELD_FILTER)
        self.tag = serialized_data.get(DroidekaCarousel.FIELD_TAG)
        self.should_have_promo = serialized_data.get('should_have_promo', False)

    @property
    def carousel_id(self) -> str:
        return self.identification_data.primary_id

    @property
    def limit(self) -> int:
        return self.pagination_data.limit

    @property
    def offset(self) -> int:
        return self.pagination_data.offset

    @property
    def docs_cache_hash(self) -> str:
        return self.pagination_data.feed_identifier

    @property
    def headers(self) -> dict:
        return self._headers

    @property
    def request(self):
        return self.more_url_data.request

    @property
    def next_url_name(self) -> str:
        return self.more_url_data.next_url_name

    @property
    def restriction_age(self) -> Optional[int]:
        return self._restriction_age


class KpCarouselInfo(CarouselTypeMixin):

    def __init__(self, serialized_data: dict, request, headers: dict, next_url_name: str):
        self.carousel_id = serialized_data[DroidekaCarousel.FIELD_CAROUSEL_ID]
        carousel_id_parts = serialized_data[DroidekaCarousel.FIELD_CAROUSEL_ID].split('/')
        self.window_id = carousel_id_parts[0]
        self.selection_id = carousel_id_parts[1]
        self.limit = serialized_data[DroidekaCarousel.FIELD_LIMIT]
        self.offset = serialized_data[DroidekaCarousel.FIELD_OFFSET]
        self.session_id = serialized_data.get(DroidekaCarousel.FIELD_CACHE_HASH)
        self.request = request
        self.next_url_name = next_url_name
        self.headers = headers
        self.restriction_age = serialized_data.get(DroidekaCarousel.FIELD_RESTRICTION_AGE)
        self.more_url_limit = serialized_data.get(DroidekaCarousel.FIELD_MORE_URL_LIMIT)
        self.should_have_promo = serialized_data.get('should_have_promo', False)

    @property
    def carousel_type(self) -> str:
        return KpCarousel.TYPE


class PurchasesCarouselInfo:

    def __init__(self, carousel_id, limit, offset, request, headers, next_url_name, available_only,
                 external_categories_mapping):
        self.carousel_id = carousel_id
        self.limit = limit
        self.offset = offset
        self.request = request
        self.next_url_name = next_url_name
        self.headers = headers
        self.available_only = available_only
        self.external_categories_mapping = external_categories_mapping

    @staticmethod
    def from_mixed_carousels_info(carousels_info: MixedCarouselsInfo) -> 'PurchasesCarouselInfo':
        return PurchasesCarouselInfo(
            None, carousels_info.max_item_count, 0, carousels_info.request, carousels_info.headers,
            carousels_info.nested_next_url_name, carousels_info.purchases_available_only,
            carousels_info.external_categories_mapping)

    @staticmethod
    def from_serialized_data(serialized_data, request, headers, next_url_name) -> 'PurchasesCarouselInfo':
        carousel_id = serialized_data.get(DroidekaCarousel.FIELD_CAROUSEL_ID)
        available_only = serialized_data.get(DroidekaCarousel.FIELD_AVAILABLE_ONLY)
        limit = serialized_data.get(DroidekaCarousel.FIELD_LIMIT)
        offset = serialized_data.get(DroidekaCarousel.FIELD_OFFSET)
        return PurchasesCarouselInfo(carousel_id, limit, offset, request, headers, next_url_name, available_only, None)


class KpCategoryInfo:
    def __init__(self, serialized_data, request, headers, next_url_name, nested_next_url_name, **kwargs):
        self.window_id = serialized_data[DroidekaCategory.FIELD_CATEGORY_ID]
        self.selections_limit = serialized_data[DroidekaCategory.FIELD_LIMIT]
        self.selections_offset = serialized_data[DroidekaCategory.FIELD_OFFSET]
        self.session_id = serialized_data[DroidekaCategory.FIELD_CACHE_HASH]
        self.items_limit = serialized_data[DroidekaCategory.FIELD_MAX_ITEMS_COUNT]
        self.request = request
        self.headers = headers
        self.next_url_name = next_url_name
        self.nested_next_url_name = nested_next_url_name
        self.category_content_type = kwargs.get('category_content_type')
        self.purchases_available_only = kwargs.get('purchases_available_only')
