import logging
from typing import Optional, Type, List
from rest_framework.exceptions import NotFound

from smarttv.droideka.proxy import data_source

from smarttv.droideka.proxy.constants.base import PaginationInfo
from smarttv.droideka.proxy.constants.carousels import CarouselsExternal, VhFeed, VhCarousel, CAROUSELS_WITH_FILTERS, \
    DroidekaCategory, COMPOSITE_FEED, MusicCarousel
from smarttv.droideka.proxy.exceptions import BadGatewayError
from smarttv.droideka.proxy.request.carousels import MixedCarouselsInfo, VhCarouselInfo, KpCarouselInfo, \
    KpCategoryInfo, PurchasesCarouselInfo
from smarttv.droideka.proxy.response.carousels import MixedCarouselsResponse, KpMultiSelectionsResponseFields, \
    VhFeedResponse
from smarttv.droideka.utils.chaining import DataSource
from smarttv.droideka.utils.chaining import ResultBuilder, ResultBuilderExtension
from smarttv.droideka.utils.url import build_view_url
from smarttv.droideka.proxy.serializers.fast_response import CarouselsV7Serializer, KpCarouselSerializer, \
    EmbeddedSectionSerializer, KpMultiSelectionCarouselSerializer, MusicInfiniteFeedSerializer
from smarttv.droideka.utils.default_values import DEFAULT_DICT
from smarttv.droideka.proxy.vh.constants import WHAT_TO_SEE_OTT_SELECTION_ID
from smarttv.droideka.proxy.constants.home_app_versions import VER_3_120
from yaphone.utils.parsed_version import ParsedVersion
from smarttv.droideka.proxy.categories_provider import categories_provider
from smarttv.droideka.proxy.models import Category2
from smarttv.droideka.proxy.exceptions import GatewayTimeoutError

logger = logging.getLogger(__name__)


class ContentDetailResultBuilder(ResultBuilder):
    """
    Provides details about content by id
    """

    def _create_data_source(self):
        return data_source.ContentDetailDataSource(
            content_id=self.content_id,
            headers=self.headers,
            request=self.request,
        )

    def __init__(self, content_id, headers, request):
        self.request = request
        self.content_id = content_id
        self.headers = headers
        super().__init__(data_source=self._create_data_source(),
                         raise_exception=True)


class ContentDetailResultBuilderExtension(ResultBuilderExtension):
    """
    Provides details about content identity data
    """
    RESULT_FIELD_NAME = 'content_detail'

    def _create_data_source(self, identity_data):
        return data_source.ContentDetailDataSource(
            content_id=identity_data['content_id'],
            headers=identity_data['headers'],
            request=self.request,
        )

    def __init__(self, request):
        self.request = request
        super().__init__(data_source_provider=self._create_data_source,
                         name=self.RESULT_FIELD_NAME,
                         raise_exception=False)


class ContentDetailV4ResultBuilder(ResultBuilder):

    def _create_data_source(self) -> DataSource:
        return data_source.ContentDetailIdentityInfoDataSource(
            self.content_id,
            self.onto_id,
            self.content_type,
            self.headers,
            self.request,
        )

    def __init__(self, headers, content_id, request, onto_id=None, content_type=None):
        self.content_id = content_id
        self.onto_id = onto_id
        self.content_type = content_type
        self.headers = headers
        self.request = request
        super().__init__(self._create_data_source(), raise_exception=True)


class EntitySearchSingleItemExtension(ResultBuilderExtension):
    """
    Provides object response by content_id
    """
    RESULT_FIELD_NAME = 'object_response'

    def __init__(self, request, headers=None, passport_uid=None):
        self.headers = headers
        self.passport_uid = passport_uid
        self.request = request

        super().__init__(
            data_source_provider=self._create_data_source,
            raise_exception=True,
            name=self.RESULT_FIELD_NAME
        )

    @staticmethod
    def _get_onto_id_list(data):
        onto_id = data.get('onto_id')
        if not onto_id and data.get(ContentDetailResultBuilderExtension.RESULT_FIELD_NAME):
            onto_id = data[ContentDetailResultBuilderExtension.RESULT_FIELD_NAME].get('onto_id')
        return [onto_id] if onto_id else []

    def _create_data_source(self, content_detail):
        if not content_detail:
            raise ValueError("Can't create ES data source without content detail")
        if not self.headers:
            self.headers = content_detail.get('headers')

        return data_source.EntitySearchSingleItemDataSource(
            onto_id_list=self._get_onto_id_list(content_detail),
            headers=self.headers,
            request=self.request,
            passport_uid=self.passport_uid
        )


class OttMetadataExtension(ResultBuilderExtension):
    RESULT_FIELD_NAME = 'ott_metadata'

    def _create_data_source(self, identity_data_container):
        if not self.content_id and not self.headers and not identity_data_container:
            raise ValueError("Can't create OttMetadataDataSource without identity data")

        if not self.content_id and not self.headers:
            self.headers = identity_data_container.get('headers')
            self.content_id = identity_data_container.get('content_id')

        return data_source.OttMetadataDataSource(content_id=self.content_id, headers=self.headers,
                                                 content_detail=identity_data_container,
                                                 request=identity_data_container.get('request'))

    def __init__(self, content_id=None, headers=None):
        self.headers = headers
        self.content_id = content_id
        super().__init__(
            data_source_provider=self._create_data_source,
            raise_exception=False,
            name=self.RESULT_FIELD_NAME,
        )


class OttFilmsToWatchExtension(ResultBuilderExtension):
    RESULT_FIELD_NAME = 'ott_films_to_watch'

    def _create_data_source(self, identity_data_container):
        if not self.content_id or not self.headers or not self.request:
            raise ValueError("Can't create FilmsToWatchDataSource without identity data")

        return data_source.FilmsToWatchDataSource(content_id=self.content_id, headers=self.headers,
                                                  content_detail=identity_data_container,
                                                  request=self.request)

    def __init__(self, content_id=None, headers=None, request=None):
        self.headers = headers
        self.content_id = content_id
        self.request = request
        super().__init__(
            data_source_provider=self._create_data_source,
            raise_exception=False,
            name=self.RESULT_FIELD_NAME,
        )


class OttSeriesStructureExtension(ResultBuilderExtension):
    RESULT_FIELD_NAME = 'ott_series'

    @staticmethod
    def _create_data_source(identity_data_container):
        if not identity_data_container:
            raise ValueError("Can't create OttSeriesStructureDataSource without identity data")
        return data_source.OttSeriesStructureDataSource(
            content_id=identity_data_container.get('content_id'),
            content_type=identity_data_container.get('content_type_name'),
            headers=identity_data_container.get('headers'),
            request=identity_data_container.get('request'),
        )

    def __init__(self):
        super().__init__(data_source_provider=self._create_data_source,
                         raise_exception=False,
                         name=self.RESULT_FIELD_NAME)


class FeedResultBuilder(ResultBuilder):
    datasource_type = data_source.CarouselsDataSource

    def _create_data_source(self):
        return self.datasource_type(self.category_id,
                                    self.next_url_name,
                                    self.nested_next_url_name,
                                    cache_hash=self.cache_hash,
                                    max_items_count=self.max_items_count,
                                    offset=self.offset,
                                    limit=self.limit,
                                    requested_amount=self.limit,
                                    restriction_age=self.restriction_age,
                                    headers=self.headers,
                                    request=self.request,)

    def __init__(self, category_id, next_url_name, nested_next_url_name, max_items_count, offset, limit, cache_hash,
                 restriction_age, headers, request):
        self.category_id = category_id
        self.max_items_count = max_items_count
        self.offset = offset
        self.limit = limit
        self.cache_hash = cache_hash
        self.headers = headers
        self.request = request
        self.next_url_name = next_url_name
        self.nested_next_url_name = nested_next_url_name
        self.restriction_age = restriction_age

        super().__init__(self._create_data_source(), True)


class MixedCarouselsMoreUrlBuilder:

    def __init__(self, category_id, feed: dict, injectable_carousels: list, feed_identifier: str,
                 request_info: MixedCarouselsInfo):
        self.category_id = category_id
        self.feed = feed
        self.injectable_carousels = injectable_carousels
        self.feed_identifier = feed_identifier
        self.limit = request_info.limit
        self.offset = request_info.offset
        self.external_carousel_offset = request_info.external_carousel_offset
        self.request = request_info.request
        self.next_url_name = request_info.next_url_name
        self.max_items_count = request_info.max_item_count
        self.injectable_carousels_limit = request_info.external_categories_limit

    def build_more_url(self) -> Optional[str]:
        feed_carousels_size = len(self.feed['carousels'])
        injectable_carousels_size = len(self.injectable_carousels)
        if feed_carousels_size < (self.limit - self.injectable_carousels_limit):
            return None
        args = {
            VhFeed.FIELD_CATEGORY_ID: self.category_id,
            PaginationInfo.FIELD_LIMIT: self.limit,
            PaginationInfo.FIELD_OFFSET: self.offset + feed_carousels_size + injectable_carousels_size,
            VhFeed.FIELD_CACHE_HASH: self.feed_identifier,
            VhFeed.FIELD_MAX_ITEMS_COUNT: self.max_items_count,
            CarouselsExternal.FIELD_EXTERNAL_CAROUSEL_OFFSET: (self.external_carousel_offset +
                                                               injectable_carousels_size)
        }

        return build_view_url(args, self.next_url_name, self.request)


class InjectableCarouselContainer:
    def __init__(self, position, carousel):
        self.position = position
        self.carousel = carousel


class FeedResultBuilderV7(ResultBuilder):
    vh_carousels_serializer = CarouselsV7Serializer()
    kp_carousel_serializer = KpCarouselSerializer()
    embedded_section_serializer = EmbeddedSectionSerializer()
    music_serializer = MusicInfiniteFeedSerializer()

    def __init__(self, data: MixedCarouselsInfo):
        super().__init__(data_source.FeedDataSource(data), True)
        if data.category_content_type != COMPOSITE_FEED:
            self.add_extension(VhFeedV7Extension())
        self.add_extension(PurchasesExtension())  # TODO(alex-garmash): do not add it on every request
        self.add_extension(KpSelectionsFeedV7Extension())
        self.add_extension(SectionExtension())
        self.add_extension(MusicHomeScreenExtension(headers=data.headers))
        self.request_info = data

    def serializer_vh_carousels(self, category_id, carousels: dict):
        context = {'category_id': category_id, 'platform_info': self.request_info.request.platform_info,
                   'request': self.request_info.request,
                   'carousel_view_name': self.request_info.nested_next_url_name,
                   'carousels_view_name': self.request_info.next_url_name}

        serialized_result = self.vh_carousels_serializer.serialize(
            VhFeedResponse(carousels),
            context)
        return serialized_result

    def get_category_position(self, category: dict):
        if self.request_info.category_id == 'main' and 'music_response' in category:
            music_main_carousel_position = self.request_info.request.request_info.experiments.get_value_or_default(
                data_source.EXP_MUSIC_CAROUSEL_POSITION)
            if music_main_carousel_position is not None and isinstance(music_main_carousel_position, int) \
                    and music_main_carousel_position >= 0:
                return music_main_carousel_position - 1
        return category['position']

    def get_result(self) -> MixedCarouselsResponse:
        raw_result = super().get_result()

        vh_feed = self.serializer_vh_carousels(
            self.request_info.category_id, raw_result.get(VhFeedV7Extension.FIELD_NAME, DEFAULT_DICT))
        injectable_carousels = []

        total_offset = self.request_info.offset + self.request_info.external_carousel_offset
        raw_kp_carousels = raw_result.get(KpSelectionsFeedV7Extension.FIELD_NAME)
        context = {'platform_info': self.request_info.request.platform_info,
                   'request': self.request_info.request,
                   'carousel_view_name': self.request_info.nested_next_url_name,
                   'carousels_view_name': self.request_info.next_url_name}
        if raw_kp_carousels:
            for kp_carousel in raw_kp_carousels:
                injectable_carousels.append(
                    InjectableCarouselContainer(kp_carousel['position'] - total_offset,
                                                self.kp_carousel_serializer.serialize(kp_carousel, context=context)))
        purchases = raw_result.get(PurchasesExtension.FIELD_NAME)
        if purchases:
            injectable_carousels.append(
                InjectableCarouselContainer(purchases['position'] - total_offset,
                                            self.kp_carousel_serializer.serialize(purchases, context=context)))
        embedded_sections = raw_result.get(SectionExtension.FIELD_NAME)
        if embedded_sections:
            for section in embedded_sections:
                injectable_carousels.append(
                    InjectableCarouselContainer(section.internal_position - total_offset,
                                                self.embedded_section_serializer.serialize(section, context=context)))

        music = raw_result.get(MusicHomeScreenExtension.FIELD_NAME)
        if music:
            for index, carousel in enumerate(music):
                carousel_position = self.get_category_position(carousel)
                position = carousel_position - total_offset
                medium = index == 0
                carousel_id = carousel['category_id'] if self.should_use_admin_carousel_id() else None
                data = self.music_serializer.serialize_single_row(carousel['music_response'], medium, carousel['admin_title'],
                                                                  carousel_id)
                if data:
                    injectable_carousels.append(InjectableCarouselContainer(position, data))

        more_url = MixedCarouselsMoreUrlBuilder(
            self.request_info.category_id,
            vh_feed,
            injectable_carousels,
            vh_feed.get(VhFeed.FIELD_CACHE_HASH),
            self.request_info
        ).build_more_url()
        return MixedCarouselsResponse(vh_feed, injectable_carousels, more_url)

    def should_use_admin_carousel_id(self):
        """
        Client sometimes duplicates music carousel if carousel id is unique.
        To fix this behaviour, we use category_id from admin interface as carousel id.
        This will be fixed on the client someday
        """
        current = ParsedVersion(self.request_info.request.platform_info.app_version)
        return current < VER_3_120


class MusicCategoryResultBuilder(ResultBuilder):
    serializer = MusicInfiniteFeedSerializer()

    def __init__(self, request_info: MixedCarouselsInfo):
        super().__init__(data_source.MusicSectionDataSource(request_info), raise_exception=True)

    def get_result(self) -> MixedCarouselsResponse:
        try:
            raw_result = super().get_result()
            feed = self.serializer.serialize(raw_result, raise_exception=True)
        except (TypeError, KeyError):
            raise BadGatewayError()

        return MixedCarouselsResponse(feed, [], None)


class VhFeedV7Extension(ResultBuilderExtension):
    FIELD_NAME = 'vh_carousels'
    datasource_type = data_source.CarouselsV7DataSource

    def _create_data_source(self, feed_data):
        return self.datasource_type(feed_data[MixedCarouselsInfo.FIELD_KEY])

    def __init__(self):
        super().__init__(data_source_provider=self._create_data_source,
                         raise_exception=False,
                         name=self.FIELD_NAME)


class KpSelectionsFeedV7Extension(ResultBuilderExtension):
    FIELD_NAME = 'kp_carousels'

    def _create_data_source(self, feed_data):
        return data_source.KpSelectionsDataSource(feed_data[MixedCarouselsInfo.FIELD_KEY])

    def __init__(self):
        super().__init__(data_source_provider=self._create_data_source,
                         raise_exception=False,
                         name=self.FIELD_NAME)


class SectionExtension(ResultBuilderExtension):
    FIELD_NAME = 'sections'

    def _create_data_source(self, feed_data) -> DataSource:
        return data_source.EmbeddedSectionDataSource(feed_data[MixedCarouselsInfo.FIELD_KEY])

    def __init__(self):
        super().__init__(data_source_provider=self._create_data_source, name=self.FIELD_NAME, raise_exception=False)


class MusicHomeScreenExtension(ResultBuilderExtension):
    FIELD_NAME = 'music'

    def _create_data_source(self, result) -> DataSource:
        request_info = result[MixedCarouselsInfo.FIELD_KEY]
        return data_source.MainMusicCarouselDataSource(request_info)

    def __init__(self, headers):
        self.headers = headers
        super().__init__(data_source_provider=self._create_data_source, name=self.FIELD_NAME, raise_exception=False)


class CarouselVideohubResultBuilder(ResultBuilder):

    def has_filters(self):
        return self.carousel_info.carousel_id in CAROUSELS_WITH_FILTERS and self.carousel_info.offset == 0

    def __init__(self, carousel_info: VhCarouselInfo, **kwargs):
        super().__init__(data_source.CarouselRequestInfoDataSource(carousel_info), True)
        self.carousel_info = carousel_info
        self.add_extension(CarouselVideohubExtension())
        if self.has_filters():
            self.add_extension(CarouselContentFiltersExtension())

    def get_result(self) -> dict:
        intermediate_result = super().get_result()
        if CarouselVideohubExtension.RESULT_FIELD_NAME not in intermediate_result:
            raise GatewayTimeoutError(f"Timeout while loading carousel '{self.carousel_info.carousel_id}'")
        result = intermediate_result[CarouselVideohubExtension.RESULT_FIELD_NAME]
        if self.has_filters():
            filters_field_name = CarouselContentFiltersExtension.RESULT_FIELD_NAME
            result[filters_field_name] = intermediate_result[filters_field_name]

        return result


class CarouselVideohubExtension(ResultBuilderExtension):
    RESULT_FIELD_NAME = 'carousel'

    def _create_data_source(self, carousel_info_container):
        carousel_info = carousel_info_container['carousel_info']
        carousel_data = {
            VhCarousel.FIELD_CAROUSEL_ID: carousel_info.carousel_id,
            VhCarousel.FIELD_DOCS_CACHE_HASH: carousel_info.docs_cache_hash
        }

        return data_source.FilterableCarouselDataSource(
            carousel_data, carousel_info.next_url_name, carousel_info.offset, carousel_info.limit,
            carousel_info.limit, carousel_info.restriction_age, carousel_info.headers, carousel_info.request,
            filter_key=carousel_info.filter, tag=carousel_info.tag)

    def __init__(self):
        super().__init__(self._create_data_source, name=self.RESULT_FIELD_NAME, raise_exception=True)


class KpCarouselExtension(ResultBuilderExtension):
    RESULT_FIELD_NAME = 'carousel'

    def _create_data_source(self, carousel_info_container: dict):
        return data_source.KpSelectionDataSource(carousel_info_container['carousel_info'])

    def __init__(self):
        super().__init__(self._create_data_source,
                         self.RESULT_FIELD_NAME,
                         raise_exception=True)


class KpCarouselResultBuilder(ResultBuilder):
    KP_DOCUMENTS_CONTAINER_KEY = 'data'

    def _create_data_source(self) -> DataSource:
        return data_source.CarouselRequestInfoDataSource(self.carousel_info)

    def __init__(self, carousel_info: KpCarouselInfo, **kwargs):
        self.carousel_info = carousel_info
        super().__init__(self._create_data_source(), True)
        self.add_extension(KpCarouselExtension())
        if carousel_info.should_have_promo and carousel_info.offset == 0:
            self.add_extension(TopCarouselPromoExtension())

    def get_result(self) -> dict:
        intermediate_result = super().get_result()
        carousel = intermediate_result.get(KpCarouselExtension.RESULT_FIELD_NAME)
        TopCarouselPromoExtension.fill_promos(intermediate_result, carousel, self.KP_DOCUMENTS_CONTAINER_KEY)
        return carousel


class PurchasesResultBuilder(ResultBuilder):

    def _create_data_source(self) -> DataSource:
        return data_source.PurchasesDataSource(self.carousel_info)

    def __init__(self, carousel_info: PurchasesCarouselInfo, **kwargs):
        self.carousel_info = carousel_info
        super().__init__(self._create_data_source(), True)


class PromoExtension(ResultBuilderExtension):
    def _create_data_source(self, request_info_container: dict) -> DataSource:
        return self.data_source_class(request_info_container.get(self.request_info_key))

    def __init__(self, name: str, data_source_class: Type[DataSource], request_info_key: str):
        self.data_source_class = data_source_class
        self.request_info_key = request_info_key
        super().__init__(self._create_data_source, name=name, raise_exception=False)


class TopCarouselPromoExtension(ResultBuilderExtension):
    NAME = 'promos'

    KEY_HOTEL_PROMO = 'hotel_promo'
    KEY_ZALOGIN_PROMO = 'zalogin_promo'
    KEY_GIFT_PROMO = 'gift_promo'
    KEY_MUSIC_PROMO = 'music_promo'
    KEY_OLYMPICS_PROMO = 'olympics_promo'

    PROMO_KEYS = [KEY_HOTEL_PROMO, KEY_ZALOGIN_PROMO, KEY_GIFT_PROMO, KEY_MUSIC_PROMO, KEY_OLYMPICS_PROMO]

    def _create_data_source(self, request_data_container: dict) -> DataSource:
        return data_source.TopCarouselPromoDataSource(request_data_container['carousel_info'])

    def __init__(self):
        super().__init__(self._create_data_source, name=self.NAME, raise_exception=False)
        self.add_extension(
            PromoExtension(
                self.KEY_HOTEL_PROMO,
                data_source.HotelDataSource,
                data_source.TopCarouselPromoDataSource.HOTEL_PROMO_REQUEST_INFO)
        )
        self.add_extension(
            PromoExtension(
                self.KEY_ZALOGIN_PROMO,
                data_source.CustomPromoDataSource,
                data_source.TopCarouselPromoDataSource.ZALOGIN_PROMO_REQUEST_INFO)
        )
        self.add_extension(
            PromoExtension(
                self.KEY_GIFT_PROMO,
                data_source.CustomPromoDataSource,
                data_source.TopCarouselPromoDataSource.GIFT_PROMO_REQUEST_INFO)
        )
        self.add_extension(
            PromoExtension(
                self.KEY_MUSIC_PROMO,
                data_source.CustomPromoDataSource,
                data_source.TopCarouselPromoDataSource.MUSIC_PROMO_REQUEST_INFO)
        )
        self.add_extension(
            PromoExtension(
                self.KEY_OLYMPICS_PROMO,
                data_source.RandomPromoFromCategory,
                data_source.TopCarouselPromoDataSource.OLYMPICS_PROMO_REQUEST_INFO)
        )

    @staticmethod
    def get_promos(promos_result: dict) -> Optional[List[dict]]:
        if not promos_result:
            return None
        result = []
        for key in TopCarouselPromoExtension.PROMO_KEYS:
            promo = promos_result.get(key)
            if promo:
                result.append(promo)
        return result

    @staticmethod
    def fill_promos(intermediate_result: dict, carousel: dict, document_container_key: str):
        promos = TopCarouselPromoExtension.get_promos(intermediate_result.get(TopCarouselPromoExtension.NAME))
        if promos:
            parent_includes = carousel.pop(document_container_key, [])
            carousel[document_container_key] = promos
            carousel[document_container_key].extend(parent_includes)


class RecomendationsCarouselResultBuilder(ResultBuilder):
    """
    Promo carousel with optional Hotel item on the first position + recommendations
    """

    def __init__(self, carousel_info: VhCarouselInfo, **kwargs):
        super().__init__(data_source.CarouselRequestInfoDataSource(carousel_info), True)
        self.carousel_info = carousel_info
        if carousel_info.should_have_promo and carousel_info.offset == 0:
            self.add_extension(TopCarouselPromoExtension())
        self.add_extension(CarouselVideohubExtension())

    def get_result(self) -> dict:
        intermediate_result = super().get_result()
        carousel = intermediate_result[CarouselVideohubExtension.RESULT_FIELD_NAME]
        TopCarouselPromoExtension.fill_promos(intermediate_result, carousel, data_source.FIELD_INCLUDES)
        return carousel


class SeriesSeasonsV6ResultBuilder(ResultBuilder):
    """
    Provides requested amount of seasons with 1st episode in every season
    """

    @staticmethod
    def _create_data_source(request, series_id, limit, offset, headers):
        return data_source.SeriesSeasonsV6DataSource(
            request=request,
            series_id=series_id,
            limit=limit,
            offset=offset,
            headers=headers,
        )

    def __init__(self, request, series_id, limit, offset, headers):
        super().__init__(
            self._create_data_source(
                request,
                series_id,
                limit,
                offset,
                headers,
            ), True)


class SeriesEpisodesV6ResultBuilder(ResultBuilder):
    """
    Provides requested amount of episodes for corresponding season
    """

    @staticmethod
    def _create_data_source(request, season_id, limit, offset, headers):
        return data_source.FilterableSeriesEpisodesDataSource(
            request=request,
            season_id=season_id,
            limit=limit,
            offset=offset,
            headers=headers,
        )

    def __init__(self, request, season_id, limit, offset, headers):
        super().__init__(self._create_data_source(
            request,
            season_id,
            limit,
            offset,
            headers
        ), True)


class MultiSelectionsResultBuilder(ResultBuilder):
    """
    Provides KP multi selections
    """

    ALLOWED_OTT_CONTENT_TYPES = (
        KpMultiSelectionsResponseFields.OttContentType.MOVIE,
        KpMultiSelectionsResponseFields.OttContentType.SERIES,
        KpMultiSelectionsResponseFields.OttContentType.SELECTION
    )

    kp_carousel_serializer = KpCarouselSerializer()
    kp_multi_selection_serializer = KpMultiSelectionCarouselSerializer()
    music_serializer = MusicInfiniteFeedSerializer()

    def _create_data_source(self, kp_category_info, music_category) -> DataSource:
        return data_source.KpMultiSelectionsDataSource(kp_category_info, music_category)

    def get_music_carousel(self, kp_category_info: KpCategoryInfo, child_categories: Optional[List[Category2]]) -> Optional[Category2]:
        if not child_categories:
            return None
        category: Category2
        for category in child_categories:
            ge_start = category.internal_position >= kp_category_info.selections_offset
            le_end = category.internal_position <= \
                     kp_category_info.selections_offset + kp_category_info.selections_limit
            is_music_main_carousel = category.content_type == MusicCarousel.TYPE
            if ge_start and le_end and is_music_main_carousel:
                return category
        return None

    def __init__(self, kp_category_info: KpCategoryInfo):
        self.kp_category_info = kp_category_info
        child_categories = categories_provider.get_categories(
            kp_category_info.request.platform_info,
            kp_category_info.window_id,
            kp_category_info.request.request_info
        )
        music_carousel = self.get_music_carousel(kp_category_info, child_categories)
        super().__init__(self._create_data_source(kp_category_info, music_carousel), True)
        self.request = kp_category_info.request
        if music_carousel:
            self.add_extension(MusicHomeScreenExtension(kp_category_info.headers))

    @classmethod
    def get_carousel_with_allowed_items_only(cls, carousel: dict) -> Optional[dict]:
        if not carousel or not carousel.get(KpMultiSelectionsResponseFields.FIELD_DATA):
            return None
        data = carousel[KpMultiSelectionsResponseFields.FIELD_DATA]
        filtered_data = []
        for item in data:
            item_content_type = KpMultiSelectionsResponseFields.OttContentType.value_of(
                item.get(KpMultiSelectionsResponseFields.FIELD_CONTENT_TYPE) or
                item.get(KpMultiSelectionsResponseFields.FIELD_TYPE))
            if item_content_type in cls.ALLOWED_OTT_CONTENT_TYPES:
                filtered_data.append(item)
        if not filtered_data:
            return None
        carousel[KpMultiSelectionsResponseFields.FIELD_DATA] = filtered_data
        return carousel

    def get_result(self) -> MixedCarouselsResponse:
        raw_result = super().get_result()
        carousels = []
        context = {'platform_info': self.request.platform_info}
        if KpMultiSelectionsResponseFields.FIELD_COLLECTIONS in raw_result:
            for _carousel in raw_result[KpMultiSelectionsResponseFields.FIELD_COLLECTIONS]:
                carousel = self.get_carousel_with_allowed_items_only(_carousel)
                if not carousel:
                    logger.info('Filtered carousel: %s',
                                _carousel.get(KpMultiSelectionsResponseFields.FIELD_TITLE))
                    continue
                carousel_type = carousel[KpMultiSelectionsResponseFields.FIELD_TYPE]
                if carousel_type in KpMultiSelectionsResponseFields.FLAT_TYPES and \
                        carousel[KpMultiSelectionsResponseFields.FIELD_SELECTION_ID] != WHAT_TO_SEE_OTT_SELECTION_ID:
                    carousels.append(self.kp_carousel_serializer.serialize(carousel, context=context))
                elif carousel_type == KpMultiSelectionsResponseFields.TYPE_MULTI_SELECTION:
                    carousels.append(self.kp_multi_selection_serializer.serialize(carousel, context=context))
        music_carousels = raw_result.get(MusicHomeScreenExtension.FIELD_NAME)
        if music_carousels:
            for carousel in music_carousels:
                position = carousel['position'] - self.kp_category_info.selections_offset
                carousel_id = carousel['category_id']
                data = self.music_serializer.serialize_single_row(carousel['music_response'], True,
                                                                  carousel['admin_title'],
                                                                  carousel_id)
                if data:
                    carousels.insert(position, data)
        result = {'carousels': carousels}
        return MixedCarouselsResponse(result, None, raw_result.get(DroidekaCategory.FIELD_MORE_URL))


class CarouselContentFiltersExtension(ResultBuilderExtension):
    RESULT_FIELD_NAME = 'filter'

    def _create_data_source(self, carousel_request_info_container: dict):
        carousel_request_info = carousel_request_info_container['carousel_info']
        return data_source.CarouselContentFiltersDataSource(carousel_request_info)

    def __init__(self):
        super().__init__(self._create_data_source, name=self.RESULT_FIELD_NAME, raise_exception=True)


class PurchasesExtension(ResultBuilderExtension):
    FIELD_NAME = 'purchases'

    def _create_data_source(self, feed_data) -> DataSource:
        return data_source.PurchasesDataSource(PurchasesCarouselInfo.from_mixed_carousels_info(
            feed_data[MixedCarouselsInfo.FIELD_KEY]))

    def __init__(self, ):
        super().__init__(self._create_data_source, name=self.FIELD_NAME, raise_exception=False)


class UnboundEpisodesExtension(ResultBuilderExtension):
    FIELD_NAME = 'raw_episodes'

    def _create_data_source(self, seasons) -> DataSource:
        return data_source.UnboundEpisodesDataSource(seasons, self.headers, self.request)

    def __init__(self, request, headers):
        super().__init__(self._create_data_source, self.FIELD_NAME, raise_exception=True)
        self.request = request
        self.headers = headers


class UnboundEpisodesResultBuilder(ResultBuilder):

    def _create_data_source(self, data: dict, headers: dict, request):
        return data_source.UnboundSeasonsDataSource(data, headers, request)

    def __init__(self, data: dict, headers: dict = None, request=None):
        super().__init__(self._create_data_source(data, headers, request), True)
        self.add_extension(UnboundEpisodesExtension(request, headers))

    def get_result(self) -> list:
        result = super().get_result()
        if UnboundEpisodesExtension.FIELD_NAME not in result:
            series_id = result['series_id']
            raise NotFound(f'No episodes found for series "{series_id}"')
        return result[UnboundEpisodesExtension.FIELD_NAME]


class FullCardDetail(ResultBuilder):

    def _create_data_source(self, content_id, headers, passport_uid, request):
        return data_source.ContentDetailIdentityInfoDataSource(content_id, None, None, headers, request,
                                                               passport_uid=passport_uid)

    def __init__(self, content_id, headers, passport_uid, request):
        super().__init__(self._create_data_source(content_id, headers, passport_uid, request), True)

    def get_result(self) -> dict:
        result = super().get_result()
        content_detail = result.get(ContentDetailResultBuilderExtension.RESULT_FIELD_NAME)
        if content_detail:
            if content_detail.get(EntitySearchSingleItemExtension.RESULT_FIELD_NAME):
                result[EntitySearchSingleItemExtension.RESULT_FIELD_NAME] = content_detail.pop(
                    EntitySearchSingleItemExtension.RESULT_FIELD_NAME)
            if content_detail.get(OttMetadataExtension.RESULT_FIELD_NAME):
                result[OttMetadataExtension.RESULT_FIELD_NAME] = content_detail.pop(
                    OttMetadataExtension.RESULT_FIELD_NAME)
            if content_detail.get(OttFilmsToWatchExtension.RESULT_FIELD_NAME):
                result[OttFilmsToWatchExtension.RESULT_FIELD_NAME] = content_detail.pop(
                    OttFilmsToWatchExtension.RESULT_FIELD_NAME)
        return result


class TrailersSignExtension(ResultBuilderExtension):
    RESULT_FIELD_NAME = 'trailers_sign'

    def __init__(self, document_onto_id, request, headers: dict):
        self.document_onto_id = document_onto_id
        self.headers = headers
        self.request = request

        super().__init__(
            data_source_provider=self._create_data_source,
            raise_exception=False,
            name=self.RESULT_FIELD_NAME
        )

    def _get_trailer_vh_uuid(self, oo: dict):
        try:
            return oo['view']['Gallery']['Trailer']['vh_uuid']
        except (KeyError, TypeError):
            return None

    def _create_data_source(self, oo: dict):
        return data_source.TrailerSignDataSource(self._get_trailer_vh_uuid(oo), self.document_onto_id, self.headers, self.request)
