import base64
import hashlib
import logging
import threading

import boto3
import botocore.config as boto_config
import botocore.exceptions as boto_exceptions
from django.conf import settings

logger = logging.getLogger(__name__)


class S3Client:
    """
    Borrowed from https://a.yandex-team.ru/arc/trunk/arcadia/plus/gift/gift/s3mds.py
    """
    extension_name = 's3mds2'

    class AlreadyExists(Exception):
        pass

    class ConnectionError(Exception):
        pass

    def __init__(self):
        self._thread_data = threading.local()

    @property
    def host(self):
        return settings.S3_MDS_HOST

    @property
    def public_host(self):
        return settings.S3_MDS_PUBLIC_HOST

    @property
    def bucket_name(self):
        return settings.S3_MDS_ANDROID_BUCKET_NAME

    @property
    def access_key_id(self):
        return settings.S3_MDS_ACCESS_KEY_ID

    @property
    def access_secret_key(self):
        return settings.S3_MDS_ACCESS_SECRET_KEY

    def _get_connection(self):
        try:
            session = boto3.session.Session(
                aws_access_key_id=self.access_key_id,
                aws_secret_access_key=self.access_secret_key
            )

            def inject_header(params, **kwargs):
                params['headers']['connection'] = 'Keep-Alive'

            session.events.register('before-call.s3', inject_header)

            config = boto_config.Config(signature_version='s3', retries={'max_attempts': 1})
            return session.resource('s3', endpoint_url=f'https://{self.host}', config=config)
        except Exception as exc:
            msg = f'Cannot connect to s3 server: {exc}'
            logger.error(msg)
            raise self.ConnectionError(msg)

    @property
    def connection(self):
        if hasattr(self._thread_data, 'connection'):
            return self._thread_data.connection
        self._thread_data.connection = self._get_connection()
        return self._thread_data.connection

    @property
    def bucket(self):
        return self.connection.Bucket(self.bucket_name)

    def get_object(self, key_name):
        return self.connection.Object(self.bucket_name, key_name)

    def save(self, key_name, data, content_type='application/octet-stream', replace=False):
        md5 = hashlib.md5(data)
        md5_hex = md5.hexdigest()
        obj = self.get_object(key_name)
        try:
            obj.load()
            current_md5 = obj.e_tag.strip('"')
            if md5_hex == current_md5:
                # File is the same
                logger.info('S3 object [%s] hash [%s] already exists.', key_name, md5_hex)
                return obj
            if not replace:
                msg = 'S3 object [%s] hash [%s] already exists. Cannot replace with [%s]'
                logger.warning(msg, key_name, current_md5, md5_hex)
                raise self.AlreadyExists(msg % (key_name, current_md5, md5_hex))
        except boto_exceptions.ClientError as e:
            if e.response.get('Error', {}).get('Code') != '404':
                raise
        obj.put(Body=data, ContentMD5=base64.b64encode(md5.digest()).decode(), ContentType=content_type)
        return obj

    def get_url(self, key_name):
        return f'https://{self.bucket_name}.{self.public_host}/{key_name}'

    def get_all_keys(self, prefix):
        return [obj_sum.key for obj_sum in self.bucket.objects.filter(Prefix=prefix).all()]


s3_client = S3Client()
