from rest_framework import serializers


class HiddenFieldsMixin:
    """
    Mixin for a serializers.Serializer with basic logic of omitting None fields
    """
    hidden_fields = ()

    def __init__(self, *args, **kwargs):
        self.discard_empty_fields = kwargs.pop('discard_empty_fields', True)
        super().__init__(*args, **kwargs)

    def del_if_empty_or_hidden(self, result, key):
        if (self.discard_empty_fields and key in result and result[key] is None) or key in self.hidden_fields:
            result.pop(key, None)

    def to_representation(self, instance):
        result = super().to_representation(instance)

        for key in self.get_fields():
            self.del_if_empty_or_hidden(result, key)

        return result


class BaseSerializer(HiddenFieldsMixin, serializers.Serializer):
    pass


class BaseModelSerializer(HiddenFieldsMixin, serializers.ModelSerializer):
    pass
