# -*- coding: utf-8 -*-
"""
This module is made to provide fast serialization of responses, since built-in serializers are too slow
"""
import json
import logging
import re
from abc import ABC, abstractmethod
from typing import Any, Iterable, Optional
from enum import Enum, auto
from collections import defaultdict
from urllib.parse import quote

from django.conf import settings
from django.contrib.admin.utils import flatten

from smarttv.droideka.proxy import api
from smarttv.droideka.proxy.licenses import get_licenses
from smarttv.droideka.proxy.api import vh, es
from smarttv.droideka.proxy.common import fix_schema
from smarttv.droideka.proxy.common import join_string_items
from smarttv.droideka.proxy.player_detection import PlayerDetector
from smarttv.droideka.proxy.s3mds import s3_client
from smarttv.droideka.proxy.serializers.response import get_actors, get_directors
from smarttv.droideka.utils import merge_dict
from smarttv.droideka.utils.default_values import DEFAULT_DICT
from smarttv.droideka.proxy.constants.carousels import VhFeed, VhCarousel, KpMultiSelection, DroidekaCarousel, \
    CarouselsExternal, CarouselType
from smarttv.droideka.proxy.response.carousels import VhFeedResponse, KpMultiSelectionsResponseFields
from smarttv.droideka.proxy.models import Category2
from smarttv.droideka.proxy import licenses as licenses_helper
from smarttv.droideka.proxy.avatars_image_generator import fill_images_from_url, fill_images_from_pattern
from smarttv.droideka.utils import PlatformInfo
from smarttv.droideka.utils.url import build_view_url
from smarttv.droideka.proxy.constants.card import NEED_PAID_CHANNELS_STUB, PAID_CHANNEL_STUB_HEADER_IMAGE, \
    PAID_CHANNEL_STUB_BODY_TEXT

logger = logging.getLogger(__name__)

FIELD_INCLUDES = 'includes'
FIELD_MONETIZATION_MODEL = 'monetizationModel'
FIELD_EPISODE_NUMBER = 'episode_number'
FIELD_SEASON_NUMBER = 'season_number'
FIELD_SEASON = 'season'
FIELD_EPISODE_NAME = 'episode_name'

FIELD_THUMBNAILS = 'thumbnails'
FIELD_POSTER = 'poster'
FIELD_POSTERS = 'posters'
FIELD_LOGO = 'logo'
FIELD_LOGOS = 'logos'
FIELD_COVERS = 'covers'
FIELD_COVER = 'cover'
FIELD_DOCUMENT_COVER = 'document_cover'

NEED_MAP_URL = 'need_map_url'

FIELD_SKIPPABLE_FRAGMENTS = 'skippableFragments'


class MonetizationModelType(Enum):
    SVOD = "SVOD"
    TVOD = "TVOD"
    EST = "EST"
    AVOD = "AVOD"


def get_episode_title(raw_episode):
    episode_name = raw_episode.get(FIELD_EPISODE_NAME)
    episode_number = raw_episode.get(FIELD_EPISODE_NUMBER)
    if episode_name and episode_number:
        return f'{episode_number}. {episode_name}'
    elif episode_number:
        return f'Эпизод {episode_number}'
    else:
        return episode_name


def get_carousel_more_icon(has_vertical):
    if has_vertical:
        return s3_client.get_url('temp/v.png')
    else:
        return s3_client.get_url('temp/h.png')


def fill_ya_plus_if_necessary(license_info, destination):
    if license_info:
        ya_plus = license_info.get_legacy_ya_plus_array()
        if ya_plus:
            destination['ya_plus'] = ya_plus


def get_platform_info(context: dict) -> Optional[PlatformInfo]:
    if not context:
        return None
    return context.get('platform_info')


class Serializer(ABC):
    @abstractmethod
    def serialize(self, obj, context) -> Any:
        pass


class BaseContextedSerializer(Serializer):
    fields_to_copy: Iterable = None
    fields_to_rename: dict = None
    fields_to_modify: dict = None

    def filter_none_items(self, obj):
        none_keys = [k for k, v in obj.items() if v is None]
        for key in none_keys:
            del obj[key]

        return obj

    def serialize(self, obj, context) -> dict:
        result = {}
        if obj:
            if self.fields_to_copy:
                result = {key: obj[key] for key in self.fields_to_copy if key in obj}

            if self.fields_to_rename:
                update = ((target_key, obj[origin_key]) for origin_key, target_key in self.fields_to_rename.items() if
                          origin_key in obj)
                result.update(update)

            if self.fields_to_modify:
                update = ((key, modify_func(obj[key])) for key, modify_func in self.fields_to_modify.items() if
                          key in obj)
                result.update(update)

        return result


class BaseSerializer(BaseContextedSerializer):

    def serialize(self, obj, context=None) -> dict:
        return super().serialize(obj, context)


class BaseContentInfo(BaseSerializer):
    FIELD_CONTENT_ID = 'content_id'
    FIELD_CONTENT_TYPE = 'content_type'
    FIELD_CONTENT_TYPE_NAME = 'content_type_name'
    FIELD_RESTRICTION_AGE = 'restriction_age'
    FIELD_TITLE = 'title'
    FIELD_THUMBNAIL = 'thumbnail'
    FIELD_DEEP_HD = 'deep_hd'
    FIELD_LIKES = 'likes'
    FIELD_DISLIKES = 'dislikes'
    FIELD_YA_PLUS = 'ya_plus'
    FIELD_LICENSES = 'licenses'
    FIELD_OWNED = 'owned'

    fields_to_copy = [FIELD_CONTENT_ID, FIELD_RESTRICTION_AGE, FIELD_TITLE, FIELD_DEEP_HD, FIELD_DISLIKES, FIELD_LIKES]
    fields_to_rename = {FIELD_CONTENT_TYPE_NAME: FIELD_CONTENT_TYPE}
    fields_to_modify = {FIELD_THUMBNAIL: fix_schema}

    def get_ownership_info(self, licenses: list):
        if not licenses:
            return None
        ownership_info = {}
        for _license in licenses:
            try:
                monetization_model = _license[FIELD_MONETIZATION_MODEL]
            except (KeyError, TypeError):
                continue
            ownership_info[monetization_model] = ownership_info.get(monetization_model, False) | _license.get(licenses_helper.FIELD_ACTIVE, False)
        return ownership_info

    def get_license_container(self, obj: dict):
        try:
            return obj[licenses_helper.FIELD_OTT_PARAMS][licenses_helper.FIELD_LICENSES]
        except (TypeError, KeyError):
            try:
                return obj[FIELD_INCLUDES][0][licenses_helper.FIELD_LICENSES]
            except (TypeError, KeyError, IndexError):
                return None

    def get_owned(self, obj: dict) -> Optional[dict]:
        licenses = self.get_license_container(obj)
        return self.get_ownership_info(licenses)

    def serialize(self, obj, context=None):
        result = super().serialize(obj, context)
        license_info = get_licenses(obj)
        fill_ya_plus_if_necessary(license_info, result)
        if license_info.licenses:
            result[self.FIELD_LICENSES] = license_info.monetization_models
        owned = self.get_owned(obj)
        if owned:
            result[self.FIELD_OWNED] = owned
        if context and context.get(NEED_MAP_URL, False):
            fill_images_from_url(result.get(self.FIELD_THUMBNAIL), FIELD_THUMBNAILS, result, get_platform_info(context))
        return result


class SkippableFragmentsSerializer(BaseSerializer):
    FIELD_SERIES = 'series'

    allowed_skippable_fragment_types = ('intro', 'credits', 'recap')

    def is_valid_fragment(self, fragment):
        return fragment.get('type') in self.allowed_skippable_fragment_types and fragment.get('startTime') is not None \
            and fragment.get('endTime') is not None

    def serialize(self, skippable_fragments, context=None) -> Optional[list]:
        if not skippable_fragments:
            return None
        result = []
        for fragment in flatten(skippable_fragments):
            if self.is_valid_fragment(fragment):
                result.append(fragment)
        return result

    def serialize_and_fill(self, container: Optional[dict], result: dict):
        if not container or not container.get(FIELD_SKIPPABLE_FRAGMENTS) or self.FIELD_SERIES not in container:
            return
        valid_skippable_fragments = self.serialize(container[FIELD_SKIPPABLE_FRAGMENTS])
        if not valid_skippable_fragments:
            return
        result[FIELD_SKIPPABLE_FRAGMENTS] = valid_skippable_fragments


class VideoInfo(BaseContentInfo):
    FIELD_ONTO_CATEGORY = 'onto_category'
    FIELD_RATING_KP = 'rating_kp'
    FIELD_GENRES = 'genres'
    FIELD_STREAMS = 'streams'
    FIELD_RELEASE_YEAR = 'release_year'
    FIELD_COUNTRIES = 'countries'
    FIELD_PERCANTAGE_SCORE = 'percentage_score'
    FIELD_OTT_PARAMS = 'ottParams'
    FIELD_DURATION = 'duration'
    FIELD_DURATION_S = 'duration_s'
    FIELD_SHOW_TV_PROMO = 'show_tv_promo'
    FIELD_PROGRESS = 'progress'
    FIELD_START_AT = 'start_at'

    fields_to_copy = BaseContentInfo.fields_to_copy + [FIELD_ONTO_CATEGORY, FIELD_RATING_KP,
                                                       FIELD_STREAMS, FIELD_RELEASE_YEAR, FIELD_COUNTRIES,
                                                       FIELD_PERCANTAGE_SCORE, FIELD_OTT_PARAMS, FIELD_SHOW_TV_PROMO,
                                                       FIELD_PROGRESS, FIELD_START_AT]
    fields_to_rename = merge_dict({FIELD_DURATION: FIELD_DURATION_S}, BaseContentInfo.fields_to_rename)
    fields_to_modify = merge_dict(BaseContentInfo.fields_to_modify, {FIELD_GENRES: join_string_items})


class ChannelInfo(BaseContentInfo):
    FIELD_CHANNEL_ID = 'channel_id'
    FIELD_CHANNEL_TYPE = 'channel_type'
    FIELD_HAS_CATCHUP = 'has_cachup'
    FIELD_CATCHUP_AGE = 'catchup_age'
    FIELD_CHANNEL_ICON = 'smarttv_icon'
    FIELD_SMARTTV_ICON_COLOR = 'smarttv_icon_color'
    FIELD_CHANNEL_CATEGORY = 'channel_category'
    FIELD_AUTO_FIELDS = 'auto_fields'
    FIELD_CHANNEL_SMARTTV_NUMBER = 'channel_smarttv_number'

    AUTO_FIELDS_KEYS = (FIELD_CHANNEL_ICON, FIELD_CHANNEL_SMARTTV_NUMBER, FIELD_SMARTTV_ICON_COLOR)

    fields_to_copy = BaseContentInfo.fields_to_copy + [FIELD_CHANNEL_ID, FIELD_CHANNEL_TYPE, FIELD_HAS_CATCHUP,
                                                       FIELD_CATCHUP_AGE, FIELD_CHANNEL_CATEGORY]

    def copy_field_if_presented(self, field_name: str, src: dict, dest: dict, need_schema: bool):
        if field_name in src:
            if not need_schema:
                dest[field_name] = src[field_name]
            else:
                dest[field_name] = fix_schema(src[field_name])

    def get_auto_fields(self, obj):
        result = {}
        auto_fields = obj.get(self.FIELD_AUTO_FIELDS)
        if auto_fields:
            for key in self.AUTO_FIELDS_KEYS:
                self.copy_field_if_presented(field_name=key,
                                             src=auto_fields,
                                             dest=result,
                                             need_schema=key == self.FIELD_CHANNEL_ICON)

        return result

    def serialize(self, obj, context=None):
        result = super().serialize(obj, context)
        result.update(self.get_auto_fields(obj))

        return result


class BloggerInfo(BaseSerializer):
    FIELD_BLOGGER_ID = 'blogger_id'
    FIELD_BLOGGER = 'blogger'
    FIELD_BLOGGER_NAME = 'blogger_name'
    FIELD_AVATAR = 'avatar'
    FIELD_RELEASE_DATE = 'release_date'
    FIELD_RELEASE_DATE_UT = 'release_date_ut'

    FIELD_BLOGGER_TITLE = 'title'

    fields_to_copy = [FIELD_BLOGGER_ID, FIELD_BLOGGER, FIELD_RELEASE_DATE, FIELD_RELEASE_DATE_UT]

    def serialize(self, obj, context=None):
        result = super().serialize(obj, context)
        result.update({
            self.FIELD_BLOGGER_NAME: obj.get(self.FIELD_BLOGGER, {}).get(self.FIELD_BLOGGER_TITLE),
            self.FIELD_AVATAR: fix_schema(obj.get(self.FIELD_BLOGGER, {}).get(self.FIELD_AVATAR)),
        })
        return result


class BroadcastInfo(BaseSerializer):
    FIELD_START_TIME = 'start_time'
    FIELD_END_TIME = 'end_time'

    fields_to_copy = [FIELD_START_TIME, FIELD_END_TIME]


class EpisodeInfo(VideoInfo, ChannelInfo, BloggerInfo, BaseContentInfo, BroadcastInfo, BaseSerializer):
    FIELD_CONTENT_URL = 'content_url'
    FIELD_EPISODE_NUMBER = 'episode_number'
    FIELD_PARENT_ID = 'parent_id'
    FIELD_PARENT_VOD_LIB_ID = 'parent_vod_lib_id'
    FIELD_MAIN_COLOR = 'main_color'
    FIELD_DESCRIPTION = 'description'
    FIELD_POSTER = 'poster'
    FIELD_ONTO_POSTER = 'onto_poster'
    FIELD_ONTO_ID = 'onto_id'
    FIELD_SEASONS_COUNT = 'seasons_count'
    FIELD_SHORT_DESCRIPTION = 'short_description'
    FIELD_IS_NEW_DELAYED_EPISODE = 'is_new_delayed_episode'
    FIELD_IS_NEXT_DELAYED_EPISODE = 'is_next_delayed_episode'
    FIELD_LEGAL_LOGOS = 'legal_logos'

    LEGAL_LOGO_SIZE_PRIORITY = ('L', 'M', 'S')

    # todo: тут больше уместет tuple, а не list
    fields_to_copy = BroadcastInfo.fields_to_copy + VideoInfo.fields_to_copy + ChannelInfo.fields_to_copy \
        + BloggerInfo.fields_to_copy + [FIELD_EPISODE_NUMBER, FIELD_PARENT_ID, FIELD_PARENT_VOD_LIB_ID,
                                        FIELD_MAIN_COLOR, FIELD_DESCRIPTION, FIELD_SEASONS_COUNT, FIELD_INCLUDES,
                                        FIELD_ONTO_ID, FIELD_LOGO, FIELD_SHORT_DESCRIPTION,
                                        FIELD_IS_NEW_DELAYED_EPISODE, FIELD_IS_NEXT_DELAYED_EPISODE]

    def choose_main_color(self, result):
        if self.FIELD_SMARTTV_ICON_COLOR in result:
            result[self.FIELD_MAIN_COLOR] = result.pop(self.FIELD_SMARTTV_ICON_COLOR)

    def get_legal_logo(self, obj):
        if not obj:
            return None
        thumbnail_logs = obj.get('thumbnail_logos')
        if not thumbnail_logs:
            return None

        logo_map = {}
        for item in thumbnail_logs:
            if 'formFactor' not in item or 'url' not in item:
                continue
            logo_map[item['formFactor']] = item['url']

        for priority in self.LEGAL_LOGO_SIZE_PRIORITY:
            if priority in logo_map:
                return fix_schema(logo_map[priority])
        return None

    def serialize(self, obj, context=None):
        result = super().serialize(obj, context)
        result[self.FIELD_CONTENT_URL] = 'https://fakeurl.com'
        result[self.FIELD_POSTER] = fix_schema(obj.get(self.FIELD_ONTO_POSTER))
        result[PlayerDetector.KEY_PLAYER_ID] = PlayerDetector.get_player_id_for_carousel_item(**obj)
        legal_logo = self.get_legal_logo(obj)
        if legal_logo:
            result[self.FIELD_LEGAL_LOGOS] = {'orig': legal_logo}
        if obj.get(FIELD_LOGO) and context and 'platform_info' in context:
            fill_images_from_url(result[FIELD_LOGO], FIELD_LOGOS, result, context['platform_info'])
        if obj.get(FIELD_COVER) and context and 'platform_info' in context:
            fill_images_from_url(obj[FIELD_COVER], FIELD_DOCUMENT_COVER, result, context['platform_info'], 'cover')
        self.filter_none_items(result)
        self.choose_main_color(result)

        if context and context.get(NEED_MAP_URL, False):
            fill_images_from_url(result.get(self.FIELD_POSTER), FIELD_POSTERS, result, get_platform_info(context))

        return result


class EpisodeInfoV8(EpisodeInfo):
    FIELD_CHANNEL_PROVIDER = 'channel_provider'
    fields_to_copy = EpisodeInfo.fields_to_copy + [FIELD_CHANNEL_PROVIDER]

    def serialize(self, obj, context=None):
        result = super().serialize(obj, context)
        if self.FIELD_CHANNEL_PROVIDER not in result:
            result[self.FIELD_CHANNEL_PROVIDER] = 'vh'  # default
        return result


class SmotreshkaChannel(BaseSerializer):
    FIELD_CHANNEL_ID = 'channel_id'
    FIELD_CONTENT_ID = 'content_id'
    FIELD_TITLE = 'title'
    FIELD_HAS_CATCHUP = 'has_cachup'
    FIELD_CATCHUP_AGE = 'catchup_age'
    FIELD_CATEGORY = 'channel_category'  # list
    FIELD_MAIN_COLOR = 'main_color'
    FIELD_DESCRIPTION = 'description'
    FIELD_LOGO = 'logo'
    FIELD_CONTENT_TYPE = 'content_type'
    FIELD_SMARTTV_ICON = 'smarttv_icon'
    FIELD_THUMBNAIL = 'thumbnail'
    FIELD_AUTO_FIELDS = 'auto_fields'
    FIELD_CHANNEL_SMARTTV_NUMBER = 'channel_smarttv_number'
    FIELD_PLAYER_ID = 'player_id'
    FIELD_CHANNEL_PROVIDER = 'channel_provider'

    fields_to_copy = [
        FIELD_CHANNEL_ID, FIELD_CONTENT_ID, FIELD_TITLE, FIELD_HAS_CATCHUP, FIELD_CATCHUP_AGE,
        FIELD_CATEGORY, FIELD_MAIN_COLOR, FIELD_DESCRIPTION, FIELD_LOGO, FIELD_CONTENT_TYPE, FIELD_SMARTTV_ICON,
        FIELD_CHANNEL_SMARTTV_NUMBER, FIELD_PLAYER_ID, FIELD_CHANNEL_PROVIDER, FIELD_THUMBNAIL
    ]

    def serialize(self, obj, context=None):
        result = super().serialize(obj, context)
        if self.FIELD_AUTO_FIELDS in obj:
            result[self.FIELD_CHANNEL_SMARTTV_NUMBER] \
                = obj[self.FIELD_AUTO_FIELDS][self.FIELD_CHANNEL_SMARTTV_NUMBER]
        return result


class FicherField:
    CAROUSEL = 'thematic_selection_tag'
    CATEGORY = 'thematic_selection_supertag'


class BaseEmbeddedItemSerializer(BaseSerializer, ABC):
    FIELD_THUMBNAIL = 'thumbnail'
    FIELD_TITLE = 'title'
    FIELD_CAROUSEL_ID = 'carousel_id'

    fields_to_copy = [FIELD_THUMBNAIL, FIELD_TITLE]
    fields_to_modify = {FIELD_THUMBNAIL: fix_schema}

    @property
    @abstractmethod
    def identifier_field_name(self):
        pass

    @property
    @abstractmethod
    def content_type(self):
        pass

    @classmethod
    def build_url(cls, item, context) -> str:
        return build_view_url(
            {'carousel_id': item['auto_fields'][cls.identifier_field_name]},
            context['carousel_view_name'],
            context['request'])

    def serialize(self, obj, context=None) -> dict:
        result = super().serialize(obj, context)
        fill_images_from_url(
            result.get(self.FIELD_THUMBNAIL), FIELD_THUMBNAILS, result, get_platform_info(context))
        result['url'] = self.build_url(obj, context)
        result['content_type'] = self.content_type
        return result


class EmbeddedCarouselSerializer(BaseEmbeddedItemSerializer):
    identifier_field_name = FicherField.CAROUSEL
    content_type = 'embedded_carousel'


class EmbeddedCarouselsSerializer(BaseEmbeddedItemSerializer):
    identifier_field_name = FicherField.CATEGORY
    content_type = 'embedded_category'


class PromoType(Enum):
    UNKNOWN = auto()
    ZALOGIN = auto()
    HOTEL = auto()
    MUSIC = auto()
    OLYMPICS = auto()


class CarouselItemType(Enum):
    DOCUMENT = auto()
    CATEGORY_FICHER = auto()
    CAROUSEL_FICHER = auto()
    ZALOGIN_PROMO = auto()
    HOTEL_PROMO = auto()
    MUSIC_PROMO = auto()
    OLYMPICS_PROMO = auto()


class DBPromoSerializer(BaseSerializer):
    FIELD_SUBTITLE = 'subtitle'

    fields_to_copy = [BaseContentInfo.FIELD_CONTENT_ID, BaseContentInfo.FIELD_TITLE, FIELD_SUBTITLE]
    fields_to_rename = {EpisodeInfo.FIELD_DESCRIPTION: 'subtitle'}
    content_type = 'action'

    @abstractmethod
    def get_action(self, obj):
        pass

    def serialize(self, obj, context: dict) -> dict:
        result = super().serialize(obj, context)
        result['content_type'] = self.content_type
        result['action'] = self.get_action(obj)
        if self.FIELD_SUBTITLE not in result:
            # hack for client, since this field is required(at least empty)
            result[self.FIELD_SUBTITLE] = ''
        fill_images_from_url(fix_schema(obj.get(BaseContentInfo.FIELD_THUMBNAIL)), FIELD_THUMBNAILS, result,
                             context['platform_info'])
        return result


class PromoZaloginSerializer(DBPromoSerializer):
    def get_action(self, obj):
        return 'home-app://profile'


class PromoMusicSerializer(DBPromoSerializer):
    def get_action(self, obj):
        return obj['action']


class ActionPromoWithFallbackSerializer(DBPromoSerializer):
    """
    Промо-карточка, у которой в админке задаются поля action и fallback_action

    Пример результата:
    ```
        {
            "content_id": "olympics/olympics",
            "title": "Игры в Пекине в приложении Смотрим",
            "subtitle": "",
            "content_type": "action",
            "action": "1tv://project?id=403",
            "thumbnails": {
                "orig": "xx"
            },
            "fallback_action": "home-app://market_item?package=ru.tv1.android.tv"
        }
    ```
    """
    def serialize(self, obj, context: dict) -> dict:
        result = super().serialize(obj, context)
        result['fallback_action'] = obj['fallback_action']
        return result

    def get_action(self, obj):
        return obj['action']


class PromoHotelSerializer(BaseSerializer):
    """
    Budapest promo card in the personal recomendations carousel
    """

    AVATARS_CONFIG_KEY = 'hotels'
    FIELD_CONTENT_ID = 'content_id'
    FIELD_CONTENT_TYPE = 'content_type'
    FIELD_ACTION = 'action'
    FIELD_THUMBNAILS = 'thumbnails'
    FIELD_TITLE = 'title'
    FIELD_SUBTITLE = 'subtitle'

    fields_to_copy = (FIELD_TITLE, FIELD_SUBTITLE)

    content_type = 'action'

    def serialize(self, obj, context=None) -> dict:
        """
        obj['result'] example: {
            'imageUrl': 'xxx',
            'infoUrl': 'yyy',
            'title': 'xxx',
            'subtitle': 'xxx',
        }
        """
        data = obj.get('result', {})
        result = super().serialize(data, context)
        result[self.FIELD_CONTENT_ID] = 'hotel_promo'
        result[self.FIELD_CONTENT_TYPE] = 'action'

        try:
            result[self.FIELD_ACTION] = self.action_deeplink(data['infoUrl'])
            self.fill_thumbnails(result, data)
        except (KeyError, TypeError) as err:
            logging.error(f'Invalid device info. Error: {err}, entry: {data}')
            return {}

        return result

    def action_deeplink(self, target_url):
        return f'home-app://open_url?url={quote(target_url, safe="")}'

    def fill_thumbnails(self, result, obj):
        fill_images_from_pattern(obj['imageUrl'], self.FIELD_THUMBNAILS, result, self.AVATARS_CONFIG_KEY)

    def thumbnails(self, image):
        return {
            'orig': image,
        }


class PromoSerializerMixin:
    FIELD_AUTO_FIELDS = 'auto_fields'
    FIELD_PROMO_TYPE = 'promo_type'

    #  мапа из auto_fields.promo_type
    PROMO_TYPE_MAPPING = defaultdict(lambda: PromoType.UNKNOWN, {
        'zalogin': PromoType.ZALOGIN,
        'hotel': PromoType.HOTEL,
        'music': PromoType.MUSIC,
        'olympics': PromoType.OLYMPICS,
    })

    items_mapping = {
        CarouselItemType.CAROUSEL_FICHER: EmbeddedCarouselSerializer(),
        CarouselItemType.CATEGORY_FICHER: EmbeddedCarouselsSerializer(),
        CarouselItemType.ZALOGIN_PROMO: PromoZaloginSerializer(),
        CarouselItemType.HOTEL_PROMO: PromoHotelSerializer(),
        CarouselItemType.MUSIC_PROMO: PromoMusicSerializer(),
        CarouselItemType.OLYMPICS_PROMO: ActionPromoWithFallbackSerializer(),
    }

    DEFAULT_DOCUMENT_SERIALIZER = None

    FICHERS_TYPE = (CarouselItemType.CATEGORY_FICHER, CarouselItemType.CAROUSEL_FICHER)

    @classmethod
    def is_carousel_grid(cls, item):
        return False

    @classmethod
    def is_category_grid(cls, item):
        return False

    @classmethod
    def get_auto_fields(cls, item):
        if not item:
            return None
        return item.get(cls.FIELD_AUTO_FIELDS)

    @classmethod
    def is_grid_item(cls, item, grid_key):
        auto_fields = cls.get_auto_fields(item)
        return auto_fields and auto_fields.get(grid_key)

    @classmethod
    def is_promo(cls, item):
        auto_fields = cls.get_auto_fields(item)
        return auto_fields and auto_fields.get(cls.FIELD_PROMO_TYPE)

    @classmethod
    def get_promo_type(cls, item) -> Optional[PromoType]:
        auto_fields = cls.get_auto_fields(item)
        if not auto_fields or cls.FIELD_PROMO_TYPE not in auto_fields:
            return None
        return cls.PROMO_TYPE_MAPPING[auto_fields[cls.FIELD_PROMO_TYPE]]

    @classmethod
    def get_item_type(cls, item) -> CarouselItemType:
        promo_type = cls.get_promo_type(item)
        if cls.is_carousel_grid(item):
            return CarouselItemType.CAROUSEL_FICHER
        elif cls.is_category_grid(item):
            return CarouselItemType.CATEGORY_FICHER
        elif promo_type == PromoType.ZALOGIN:
            return CarouselItemType.ZALOGIN_PROMO
        elif promo_type == PromoType.HOTEL:
            return CarouselItemType.HOTEL_PROMO
        elif promo_type == PromoType.MUSIC:
            return CarouselItemType.MUSIC_PROMO
        elif promo_type == PromoType.OLYMPICS:
            return CarouselItemType.OLYMPICS_PROMO
        else:
            return CarouselItemType.DOCUMENT

    @classmethod
    def is_fichers_carousel(cls, items):
        if not items:
            return False
        return cls.get_item_type(items[0]) in cls.FICHERS_TYPE

    @classmethod
    def get_serializer(cls, document_type: CarouselItemType) -> Serializer:
        serializer = cls.items_mapping.get(document_type)
        if serializer:
            return serializer
        return cls.DEFAULT_DOCUMENT_SERIALIZER


class CinemaSerializer(BaseSerializer):
    class UnsupportedCinema(ValueError):
        pass

    fields_to_copy = (
        'tv_deeplink',
        'tv_fallback_link',
        'tv_package_name',
        'link',
        'duration',
        'code',
        'favicon',
        'hide_price',
        'cinema_name',
    )
    variants_order = {'tvod': 1, 'est': 2, 'avod': 3, 'svod': 4, 'fvod': 5}
    app_package_mapping = {
        'okko': 'ru.more.play',
        'kion': 'ru.mts.mtstv',
        '1tv': 'ru.tv1.android.tv',
        'ivi': 'ru.ivi.client',
        'kp': 'ru.kinopoisk.yandex.tv'
    }

    def get_tv_package_name(self, code):
        """
        Пока объектный ответ не научился давать нам название пакета для приложения
        кинотеатра, дройдека добавляет эти данные сама
        """
        package = self.app_package_mapping.get(code, '')
        if not package:
            logger.debug('Can not find tv_package_name for cinema code %s', code)

        return package

    def filter_unknown_payment_variants(self, variants):
        result = []
        for item in variants:
            if item['type'] in self.variants_order:
                result.append(item)

        return result

    def serialize_variants(self, obj):
        result = []
        variants = obj.get('variants', [])

        # неизвестные варианты оплаты отфильтровываются
        variants = self.filter_unknown_payment_variants(variants)

        # оставшиеся сортируются как в variants_order
        variants.sort(key=lambda v: self.variants_order.get(v['type']) or 999)

        for var in variants:
            result.append(
                {
                    'price': var['price'],
                    'type': var['type'],
                    'quality': var['quality'],
                }
            )
        return result

    def serialize(self, obj, context=None) -> dict:
        result = super().serialize(obj, context)
        result['variants'] = self.serialize_variants(obj)
        result['favicon'] = fix_schema(result.get('favicon', ''))
        result['tv_fallback_link'] = result.get('tv_fallback_link', '')

        if 'tv_package_name' not in result:
            result['tv_package_name'] = self.get_tv_package_name(obj['code'])

        if not result['tv_package_name']:
            raise self.UnsupportedCinema(f"Cinema {obj['code']}: empty tv_package_name")

        if not result.get('tv_deeplink'):
            raise self.UnsupportedCinema(f"Cinema {obj['code']}: empty tv_deeplink")

        return result


class KpDocumentSerializer(BaseSerializer):
    FIELD_HORIZONTAL_POSTER = 'horizontalPoster'

    class UnknownOttContentType(Exception):
        pass

    ott2vh_content_type_mapping = {
        'ott-movie': api.vh.CONTENT_TYPE_EPISODE,
        'tv-series': api.vh.CONTENT_TYPE_SERIES,
    }

    # it's ok to map only to 'film' и 'series'(without 'anim_film' and 'anim_series'), since we have no data to
    # distinguish these types, but client doesn't need them anyway
    # https://st.yandex-team.ru/TVANDROID-2764#5fbe461c1e9a9e7f68fe1cd9
    ott2vh_onto_category = {
        'ott-movie': api.vh.OntoCategory.FILM.value,
        'tv-series': api.vh.OntoCategory.SERIES.value,
    }

    def map_ott_content_type_to_vh(self, content_type: str) -> str:
        vh_content_type = self.ott2vh_content_type_mapping.get(content_type)
        if not vh_content_type:
            raise self.UnknownOttContentType()
        return vh_content_type

    def map_ott_content_type_to_vh_onto_category(self, content_type: str) -> str:
        try:
            return self.ott2vh_onto_category.get(content_type)
        except KeyError:
            raise self.UnknownOttContentType()

    fields_to_copy = [BaseContentInfo.FIELD_TITLE]

    fields_to_rename = {
        'filmId': EpisodeInfo.FIELD_CONTENT_ID,
        'kpRating': EpisodeInfo.FIELD_RATING_KP,
        'posterUrl': EpisodeInfo.FIELD_POSTER,
        'years': EpisodeInfo.FIELD_RELEASE_YEAR,
        'restrictionAge': EpisodeInfo.FIELD_RESTRICTION_AGE,
        'medianColor': EpisodeInfo.FIELD_MAIN_COLOR,
        'duration': EpisodeInfo.FIELD_DURATION_S
    }

    fields_to_modify = {
        'genres': join_string_items,
        'countries': join_string_items
    }

    monetization_model_to_watching_option_type = {
        MonetizationModelType.SVOD: 'SUBSCRIPTION',
        MonetizationModelType.AVOD: 'FREE',
        MonetizationModelType.EST: 'PAID',
        MonetizationModelType.TVOD: 'PAID_MULTIPLE',
    }

    def try_fill_licence_params(self, obj: dict, result: dict, monetization_model: MonetizationModelType) -> bool:
        watching_option = obj.get('watchingOption')
        monetization_models = obj.get("monetizationModels")
        if not watching_option or not monetization_models:
            return False
        if watching_option.get('type') != self.monetization_model_to_watching_option_type.get(monetization_model) \
                or monetization_model.value not in monetization_models:
            return False
        is_available = watching_option.get('purchased', False)
        result[EpisodeInfo.FIELD_OWNED] = {monetization_model.value: is_available}
        result[EpisodeInfo.FIELD_LICENSES] = [monetization_model.value]
        result[EpisodeInfo.FIELD_OTT_PARAMS] = {
            'monetizationModel': monetization_model.value,
            'primary': True,
            'active': is_available
        }
        if watching_option.get('subscriptionPurchaseTag'):
            result[EpisodeInfo.FIELD_OTT_PARAMS]['purchaseTag'] = watching_option['subscriptionPurchaseTag']
        if monetization_model == MonetizationModelType.SVOD and 'subscription' in watching_option:
            result[EpisodeInfo.FIELD_YA_PLUS] = [watching_option['subscription']]
        return True

    def fill_licence_info(self, obj: dict, result):
        for monetization_model in MonetizationModelType:
            is_filled = self.try_fill_licence_params(obj, result, monetization_model)
            if is_filled:
                break

    def serialize(self, obj, context=None) -> dict:
        result = super().serialize(obj, context)

        ott_content_type = obj.get('contentType')
        result[EpisodeInfo.FIELD_CONTENT_TYPE] = self.map_ott_content_type_to_vh(ott_content_type)
        result[EpisodeInfo.FIELD_ONTO_CATEGORY] = self.map_ott_content_type_to_vh_onto_category(ott_content_type)
        result[PlayerDetector.KEY_PLAYER_ID] = PlayerDetector.get_player_id_for_kp_carousel_item(obj)
        self.fill_licence_info(obj, result)

        fill_images_from_url(result.get(EpisodeInfo.FIELD_POSTER), FIELD_POSTERS, result, get_platform_info(context))
        fill_images_from_url(obj.get(self.FIELD_HORIZONTAL_POSTER), FIELD_THUMBNAILS, result,
                             get_platform_info(context))

        return result


class OttMetadata(BaseSerializer):
    FIELD_CONTENT = 'ott_metadata'

    FIELD_TITLE = 'title'
    FIELD_YEARS = 'years'
    FIELD_GENRES = 'genres'  # drama, comedy, thriller, etc
    FIELD_DURATION = 'duration'
    FIELD_OTT_PARAMS = 'ottParams'
    FIELD_CONTENT_TYPE = 'contentType'
    FIELD_RESTRICTION_AGE = 'restrictionAge'

    fields_to_copy = [
        FIELD_TITLE,
        FIELD_YEARS,
        FIELD_GENRES,
        FIELD_DURATION,
    ]


class ThinCardDetailsInfo(OttMetadata, BaseSerializer):
    skippable_fragments_serializer = SkippableFragmentsSerializer()

    NEED_AD_CONFIG = 'need_ad_config'

    FIELD_CONTENT = 'content_detail'

    FIELD_SERIES = 'series'
    FIELD_YA_PLUS = 'ya_plus'
    FIELD_OTT_PARAMS = 'ott_params'
    FIELD_CONTENT_ID = 'content_id'
    FIELD_RELEASE_YEAR = 'release_year'
    FIELD_SHOW_TV_PROMO = 'show_tv_promo'
    FIELD_RESTRICTION_AGE = 'restriction_age'
    FIELD_CONTENT_TYPE = 'content_type'
    FIELD_CONTENT_TYPE_NAME = 'content_type_name'
    FIELD_MONETIZATION_MODEL = 'monetization_model'
    FIELD_RELEASE_DATE = 'release_date'
    FIELD_THUMBNAIL = 'thumbnail'
    FIELD_SEASON = 'season'
    FIELD_POSTER = 'poster'
    FIELD_ONTO_POSTER = 'onto_poster'
    FIELD_EPISODE_NUMBER = 'episode_number'
    FIELD_AD_CONFIG = 'adConfig'
    FIELD_VIDEO_CONTENT_ID_RAW = 'video_content_id'
    FIELD_VIDEO_CONTENT_ID = 'video-content-id'
    FIELD_PAID_CHANNEL_STUB = 'paid_channel_stub'
    FIELD_OFFER_TEXT = 'offer_text'
    FIELD_OFFER_SUB_TEXT = 'offer_subtext'
    FIELD_START_AT = 'start_at'
    FIELD_PARTNER_ID = 'partner_id'

    fields_to_copy = OttMetadata.fields_to_copy + [
        FIELD_SERIES,
        FIELD_YA_PLUS,
        FIELD_OTT_PARAMS,  # Client needs this just in case.
        FIELD_CONTENT_ID,
        FIELD_RELEASE_YEAR,
        FIELD_SHOW_TV_PROMO,
        FIELD_RESTRICTION_AGE,
        FIELD_CONTENT_TYPE_NAME,
        FIELD_MONETIZATION_MODEL,
        FIELD_RELEASE_DATE,
        FIELD_SEASON,
        FIELD_EPISODE_NUMBER,
        FIELD_START_AT,
    ]

    fields_to_rename = {
        FIELD_CONTENT_TYPE_NAME: FIELD_CONTENT_TYPE,
        OttMetadata.FIELD_CONTENT_TYPE: FIELD_CONTENT_TYPE,
        OttMetadata.FIELD_OTT_PARAMS: FIELD_OTT_PARAMS,
        OttMetadata.FIELD_RESTRICTION_AGE: FIELD_RESTRICTION_AGE,
    }

    fields_to_modify = {
        FIELD_THUMBNAIL: fix_schema,
    }

    def _get_root_content(self, obj):
        return obj.get(self.FIELD_CONTENT)

    def serialize(self, obj, context=None) -> dict:
        content_detail = self._get_root_content(obj)
        result = super().serialize(content_detail, context)
        result.update({
            self.FIELD_CONTENT_ID: obj[self.FIELD_CONTENT_ID],
            PlayerDetector.KEY_PLAYER_ID: PlayerDetector.get_player_id_for_card_detail(**obj),
        })

        if content_detail and content_detail.get(self.FIELD_ONTO_POSTER):
            result[self.FIELD_POSTER] = fix_schema(content_detail[self.FIELD_ONTO_POSTER])

        self._prepare_monetization_model(result)
        self._prepare_content_type(result)
        self._prepare_release_year(result)
        self._prepare_genres(result)
        self._prepare_title(content_detail, result)
        self._prepare_ad_config(content_detail, result, context)
        self._prepare_paid_channel_stub(result, context)

        fill_images_from_url(result.get(self.FIELD_THUMBNAIL), FIELD_THUMBNAILS, result, get_platform_info(context))
        self.skippable_fragments_serializer.serialize_and_fill(content_detail, result)

        return result

    def _prepare_monetization_model(self, data):
        ott_params = data.get(self.FIELD_OTT_PARAMS)
        if ott_params:
            ott_monetization_model = ott_params.get(FIELD_MONETIZATION_MODEL)
            data[self.FIELD_MONETIZATION_MODEL] = ott_monetization_model

    def _prepare_content_type(self, data):
        data[self.FIELD_CONTENT_TYPE] = data.pop(self.FIELD_CONTENT_TYPE_NAME, None)

    def _prepare_release_year(self, data):
        """
        Gets the first year from result data's years list.
        :param data: result data from the other backend, which contains a list of years

        TODO: handle multiple years correctly.
          Here could be something like this:
            "years": [ 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012 ]
        """
        if data.get(OttMetadata.FIELD_YEARS):
            years = data.pop(OttMetadata.FIELD_YEARS, None)
            year = years[0] if years else None
            data[self.FIELD_RELEASE_YEAR] = year

    def _prepare_genres(self, data):
        raw_genres = data.get(self.FIELD_GENRES)
        if raw_genres:
            data[self.FIELD_GENRES] = join_string_items(raw_genres)

    @classmethod
    def _prepare_title(cls, data, result):
        if data and result:
            if FIELD_EPISODE_NAME in data or FIELD_EPISODE_NUMBER in data:
                result[OttMetadata.FIELD_TITLE] = get_episode_title(data)

    def _prepare_ad_config(self, data, result, context):
        if context and context.get(self.NEED_AD_CONFIG) and data:
            ad_config = data.get(self.FIELD_AD_CONFIG)
            if not ad_config:
                return
            if self.FIELD_VIDEO_CONTENT_ID_RAW in ad_config:
                ad_config[self.FIELD_VIDEO_CONTENT_ID] = ad_config.pop(self.FIELD_VIDEO_CONTENT_ID_RAW)
            if ad_config.get(self.FIELD_PARTNER_ID) != 0:
                result[self.FIELD_AD_CONFIG] = ad_config

    def _prepare_paid_channel_stub(self, obj, context):
        if not context or obj[self.FIELD_CONTENT_TYPE] not in ('channel', 'episode') or \
                not context.get(NEED_PAID_CHANNELS_STUB):
            return
        result = {}
        text = context[PAID_CHANNEL_STUB_BODY_TEXT]
        if text:
            result[PAID_CHANNEL_STUB_BODY_TEXT] = text
        header_image = context[PAID_CHANNEL_STUB_HEADER_IMAGE]
        if header_image:
            result[PAID_CHANNEL_STUB_HEADER_IMAGE] = header_image
        if result:
            ott_params = result.get(self.FIELD_OTT_PARAMS)
            if ott_params:
                licenses = ott_params.get(BaseContentInfo.FIELD_LICENSES)
                if licenses:
                    primary_license = licenses[0]
                    if 'offerText' in primary_license:
                        result[self.FIELD_OFFER_TEXT] = primary_license['offerText']
                    if 'offerSubText' in primary_license:
                        result[self.FIELD_OFFER_SUB_TEXT] = primary_license['offerSubText']
        if result:
            obj[self.FIELD_PAID_CHANNEL_STUB] = result


class CarouselSerializer(PromoSerializerMixin, BaseSerializer):
    FIELD_CATEGORY_ID = 'category_id'
    FIELD_CAROUSEL_ID = 'carousel_id'
    FIELD_HAS_VERTICAL = 'has_vertical'
    FIELD_TITLE = 'title'
    FIELD_CACHE_HASH = 'cache_hash'
    FIELD_AVATAR = 'avatar'
    FIELD_INCLUDES = 'includes'
    FIELD_FILTER = 'filter'

    FIELD_MORE_ICON = 'icon'
    FIELD_RANK = 'rank'

    fields_to_copy = [FIELD_CATEGORY_ID, FIELD_CAROUSEL_ID, FIELD_HAS_VERTICAL, FIELD_TITLE, FIELD_CACHE_HASH,
                      VhFeed.FIELD_MORE, VhCarousel.FIELD_MORE_INFO, FIELD_RANK, FIELD_FILTER,
                      DroidekaCarousel.FIELD_CAROUSEL_TYPE]
    fields_to_modify = merge_dict(BaseContentInfo.fields_to_modify, {FIELD_AVATAR: fix_schema})

    DEFAULT_DOCUMENT_SERIALIZER = EpisodeInfo()

    @classmethod
    def is_carousel_grid(cls, item):
        return cls.is_grid_item(item, 'thematic_selection_tag')

    @classmethod
    def is_category_grid(cls, item):
        return cls.is_grid_item(item, 'thematic_selection_supertag')

    @classmethod
    def prepare_carousel_items(cls, items, context):
        if not items:
            return []
        result = []
        for item in items:
            result.append(cls.get_serializer(cls.get_item_type(item)).serialize(item, context))
        return result

    def serialize(self, obj, context=None):
        result = super().serialize(obj, context)

        carousel_id = obj.get(self.FIELD_CAROUSEL_ID) or context.get(self.FIELD_CAROUSEL_ID)
        result[self.FIELD_CAROUSEL_ID] = carousel_id
        items = obj.get(self.FIELD_INCLUDES)
        result[self.FIELD_INCLUDES] = self.prepare_carousel_items(items, context)
        if self.is_fichers_carousel(items):
            result[CarouselsExternal.FIELD_CAROUSEL_TYPE] = CarouselType.TYPE_SQUARE

        more_info = result.get(VhCarousel.FIELD_MORE_INFO)
        if more_info:
            has_vertical = result.get(self.FIELD_HAS_VERTICAL, False)
            more_info[self.FIELD_MORE_ICON] = get_carousel_more_icon(has_vertical)

        return self.filter_none_items(result)


class KpCarouselSerializer(PromoSerializerMixin, BaseSerializer):

    kp_document_serializer = KpDocumentSerializer()

    fields_to_copy = [CarouselSerializer.FIELD_TITLE, VhCarousel.FIELD_MORE_INFO, VhFeed.FIELD_MORE]

    DEFAULT_DOCUMENT_SERIALIZER = KpDocumentSerializer()

    @classmethod
    def get_kp_carousel_id(cls, raw_kp_carousel: dict) -> str:
        return f'{raw_kp_carousel[api.ott.KEY_SELECTION_WINDOW_ID]}/{raw_kp_carousel[api.ott.KEY_SELECTION_ID]}'

    @classmethod
    def prepare_carousel_items(cls, items, context):
        if not items:
            return []
        return [cls.get_serializer(cls.get_item_type(item)).serialize(item, context) for item in items]

    def serialize(self, obj, context=None) -> dict:
        result = super().serialize(obj, context)
        carousel_id = self.get_kp_carousel_id(obj)
        result[CarouselSerializer.FIELD_CAROUSEL_ID] = carousel_id
        has_vertical = True
        result[CarouselSerializer.FIELD_HAS_VERTICAL] = has_vertical
        result[CarouselSerializer.FIELD_INCLUDES] = self.prepare_carousel_items(
            obj.get('data'), context)
        more_info = result.get(VhCarousel.FIELD_MORE_INFO)
        if more_info:
            more_info[CarouselSerializer.FIELD_MORE_ICON] = get_carousel_more_icon(has_vertical)
        return result


class OttIdentifierSerializerMixin:

    def get_serialized_ott_identifier(self, obj):
        window_id = obj[api.ott.KEY_SELECTION_WINDOW_ID]
        selection_id = obj[api.ott.KEY_SELECTION_ID]
        if not window_id or not selection_id:
            raise ValueError(f"KP carousel must have both '{api.ott.KEY_SELECTION_WINDOW_ID}' "
                             f"and '{api.ott.KEY_SELECTION_ID}'. Actual values: {window_id}/{selection_id}")
        return f'{window_id}/{selection_id}'


class KpMultiSelectionItemSerializer(OttIdentifierSerializerMixin, BaseSerializer):

    fields_to_copy = [EpisodeInfo.FIELD_TITLE, KpMultiSelection.FIELD_MULTISELECTION_URL]
    fields_to_rename = {KpMultiSelectionsResponseFields.FIELD_IMAGE_URL: EpisodeInfo.FIELD_THUMBNAIL}

    def serialize(self, obj, context=None) -> dict:
        result = super().serialize(obj, context)
        result[CarouselSerializer.FIELD_CAROUSEL_ID] = self.get_serialized_ott_identifier(obj)
        result[EpisodeInfo.FIELD_CONTENT_TYPE] = KpMultiSelection.CAROUSEL_CONTENT_TYPE
        return result


class KpMultiSelectionCarouselSerializer(OttIdentifierSerializerMixin, BaseSerializer):

    fields_to_copy = [CarouselSerializer.FIELD_TITLE, DroidekaCarousel.FIELD_MORE_INFO,
                      KpMultiSelectionsResponseFields.FIELD_CAROUSEL_TYPE]

    item_serializer = KpMultiSelectionItemSerializer()

    def serialize_items(self, source_object) -> list:
        includes = []
        if source_object.get(KpMultiSelectionsResponseFields.FIELD_DATA):
            for item in source_object[KpMultiSelectionsResponseFields.FIELD_DATA]:
                includes.append(self.item_serializer.serialize(item))
        return includes

    def serialize(self, obj, context=None) -> dict:
        result = super().serialize(obj, context)
        result[CarouselSerializer.FIELD_CAROUSEL_ID] = self.get_serialized_ott_identifier(obj)
        result[CarouselSerializer.FIELD_INCLUDES] = self.serialize_items(obj)

        return result


class Doc2DocSerializer(BaseSerializer):
    VALID_LICENCE = 'AVOD'
    FIELD_OTT_PARAMS = 'ottParams'
    FIELD_MONETIZATION_MODEL = 'monetizationModel'
    FIELD_LICENCES = 'licenses'

    def serialize(self, obj, context=None) -> dict:
        result = super().serialize(obj, context)

        documents = CarouselSerializer.prepare_carousel_items(items=obj['set'], context=context)

        self.update_result(result, documents, obj['rvb_mapping'], context)
        return self.filter_none_items(result)

    def filter_avod(self, documents):
        result = []
        for document in documents:
            license_info = get_licenses(document)
            primary_licence = license_info.get_primary()
            if not primary_licence or primary_licence[self.FIELD_MONETIZATION_MODEL] == self.VALID_LICENCE:
                result.append(document)

        return result

    @staticmethod
    def update_result(result, documents, rvb_mapping, context):
        result.update({
            'rvb_mapping': rvb_mapping,
            'includes': documents
        })


class Doc2DocV6Serializer(Doc2DocSerializer):
    LONG_CONTENT_CATEGORIES = frozenset({'film', 'series', 'anim_film', 'anim_series'})
    FIELD_ONTO_CATEGORY = 'onto_category'

    def update_result(self, result, documents, rvb_mapping, context):
        long_content = []
        videos = []
        for document in documents:
            platform_info = get_platform_info(context)
            fill_images_from_url(document.get(EpisodeInfo.FIELD_THUMBNAIL), FIELD_THUMBNAILS, document, platform_info)

            if document.get(self.FIELD_ONTO_CATEGORY) in self.LONG_CONTENT_CATEGORIES:
                fill_images_from_url(document.get(EpisodeInfo.FIELD_POSTER), FIELD_POSTERS, document, platform_info)
                long_content.append(document)
            else:
                videos.append(document)
        result.update({
            'rvb_mapping': rvb_mapping,
            'videos': videos,
            'long_content': long_content,
        })


class CarouselsSerializer(Serializer):
    carousel_serializer = CarouselSerializer()

    FIELD_ROOT_LIST = 'items'
    FIELD_RANK = 'rank'

    def serialize(self, obj, context=None):
        if not obj or self.FIELD_ROOT_LIST not in obj:
            return []
        carousels = obj[self.FIELD_ROOT_LIST]

        return [self.carousel_serializer.serialize(item, context) for item in carousels]


class CarouselsV5Serializer(Serializer):
    def serialize(self, obj, context=None) -> dict:
        carousels = CarouselsSerializer().serialize(obj, context)
        result = {'carousels': carousels}
        if VhFeed.FIELD_MORE in obj:
            result[VhFeed.FIELD_MORE] = obj[VhFeed.FIELD_MORE]
        api.vh.propagate_vh_tracking_params(obj, result)
        return result


class CarouselsV7Serializer(Serializer):

    FIELD_CAROUSELS = 'carousels'

    carousel_serializer = CarouselSerializer()

    def serialize(self, obj: VhFeedResponse, context=None) -> dict:
        context = context or {}
        context[NEED_MAP_URL] = True
        result = {
            self.FIELD_CAROUSELS: [self.carousel_serializer.serialize(item, context) for item in obj.carousels],
            VhFeed.FIELD_CACHE_HASH: obj.cache_hash
        }
        api.vh.propagate_vh_tracking_params(obj.tracking_params, result)
        return result


def get_season_number(raw_episode):
    season_info = raw_episode.get(FIELD_SEASON)
    if not season_info:
        logger.warning('No season info in episode: %s', raw_episode.get(BaseContentInfo.FIELD_CONTENT_ID))
        return None
    return season_info.get(SeriesSeasonV6Serializer.FIELD_SEASON_NUMBER)


class SeriesEpisodeV6Serializer(BaseSerializer):
    fields_to_copy = [
        BaseContentInfo.FIELD_CONTENT_ID,
        FIELD_EPISODE_NUMBER,
        VideoInfo.FIELD_SHOW_TV_PROMO,
        BaseContentInfo.FIELD_RESTRICTION_AGE,
    ]
    fields_to_rename = {
        BaseContentInfo.FIELD_CONTENT_TYPE_NAME: BaseContentInfo.FIELD_CONTENT_TYPE,
        VideoInfo.FIELD_DURATION: VideoInfo.FIELD_DURATION_S,
    }
    fields_to_modify = {BaseContentInfo.FIELD_THUMBNAIL: fix_schema}

    def serialize(self, obj, context=None) -> dict:
        result = super().serialize(obj, context)
        result[SeriesSeasonV6Serializer.FIELD_SEASON_NUMBER] = get_season_number(obj)
        result[BaseContentInfo.FIELD_TITLE] = get_episode_title(obj)
        result[PlayerDetector.KEY_PLAYER_ID] = PlayerDetector.get_player_id_for_carousel_item(**obj)

        return result


class SeriesEpisodeV7Serializer(SeriesEpisodeV6Serializer):
    def serialize(self, obj, context=None) -> dict:
        result = super().serialize(obj, context)
        fill_images_from_url(
            result.get(BaseContentInfo.FIELD_THUMBNAIL), FIELD_THUMBNAILS, result, get_platform_info(context))
        return result


class UnboundSeriesEpisodeV7Serializer(SeriesEpisodeV7Serializer):

    def _get_pivot_indexes(self, context) -> tuple:
        if not context:
            return None, None
        return context.get('pivot_season_number'), context.get('pivot_episode_number')

    def serialize(self, obj, context=None) -> dict:
        pivot_season_number, pivot_episode_number = self._get_pivot_indexes(context)
        result = super().serialize(obj, context)
        if result['season_number'] == pivot_season_number and result['episode_number'] == pivot_episode_number:
            result['is_pivot'] = True
        return result


class SeriesSeasonV6Serializer(BaseSerializer):
    FIELD_SEASON_NUMBER = 'season_number'
    FIELD_EPISODE_COUNT = 'episodes_count'
    FIELD_SEASON_ID = 'season_id'
    FIELD_TYPE = 'type'
    FIELD_EPISODES = 'episodes'

    episodes_serializer = SeriesEpisodeV6Serializer()

    fields_to_copy = [FIELD_SEASON_NUMBER, FIELD_EPISODE_COUNT, FIELD_TYPE]
    fields_to_rename = {
        BaseContentInfo.FIELD_CONTENT_ID: FIELD_SEASON_ID,
    }

    def get_episodes(self, raw_episodes, context):
        return [self.episodes_serializer.serialize(raw_episode, context) for raw_episode in raw_episodes]

    def serialize(self, obj, context=None) -> dict:
        result = super().serialize(obj, context)
        result[self.FIELD_EPISODES] = self.get_episodes(obj[self.FIELD_EPISODES], context)
        return result


class SeriesSeasonV7Serializer(SeriesSeasonV6Serializer):
    episodes_serializer = SeriesEpisodeV7Serializer()


# noinspection PyMethodMayBeStatic
class FullCardDetail(BaseContextedSerializer):
    base_content_info_serializer = BaseContentInfo()
    episode_serializer = EpisodeInfo()
    cinema_serializer = CinemaSerializer()

    FIELD_OBJECT_RESPONSE = 'object_response'
    FIELD_OTT_METADATA = 'ott_metadata'
    FIELD_WILL_WATCH_CONTAINER = 'ott_films_to_watch'

    FIELD_BASE_INFO = 'base_info'
    FIELD_RICH_INFO = 'rich_info'
    FIELD_VH_META = 'vh_meta'
    FIELD_CONTENT_GROUPS = 'content_groups'

    FIELD_VIEW = 'view'
    FIELD_GALLERY = 'Gallery'
    FIELD_TRAILER = 'Trailer'

    FIELD_ONTO_CATEGORY = 'onto_category'
    FIELD_GENRES = 'genres'
    FIELD_TITLE = 'title'
    FIELD_DESCRIPTION = 'description'
    FIELD_COUNTRIES = 'countries'
    FIELD_MAIN_COLOR = 'main_color'
    FIELD_SEASON_NUMBER = 'season_number'

    FIELD_PERSONAL_SCORE = 'personal_score'
    FIELD_PERCENTAGE_SCORE = 'percentage_score'

    FIELD_RATINGS = 'ratings'

    FIELD_VH_RESTRICTION_AGE = 'restriction_age'
    FIELD_OTT_RESTRICTION_AGE = 'restrictionAge'
    FIELD_OO_RESTRICTION_AGE = 'age_limit'

    FIELD_DURATION = 'duration'

    content_type_mapping = {
        'vod-library': 'vod-library',
        'TV_SERIES': 'vod-library',
        'vod-episode': 'vod-episode',
        'MOVIE': 'vod-episode',
    }

    SERIES_CONTENT_TYPES = (vh.CONTENT_TYPE_SERIES, es.OO_CONTENT_TYPE_SERIES)
    EPISODE_CONTENT_TYPES = (vh.CONTENT_TYPE_EPISODE, es.OO_CONTENT_TYPE_EPISODE)

    def _get_onto_category_by_content_type(self, content_type: str) -> str:
        if content_type == es.OO_CONTENT_TYPE_EPISODE:
            return 'film'
        elif content_type == es.OO_CONTENT_TYPE_SERIES:
            return 'series'
        return ''

    def get_genres(self, data):
        ott_metadata = data.get(self.FIELD_OTT_METADATA)
        if ott_metadata and self.FIELD_GENRES in ott_metadata:
            return join_string_items(ott_metadata[self.FIELD_GENRES])
        else:
            additional_info_items = data.get(
                self.FIELD_OBJECT_RESPONSE, DEFAULT_DICT
            ).get(
                'wiki_snippet', DEFAULT_DICT
            ).get(
                'item', []
            )

            filtered_items = [item for item in additional_info_items if item['key'].startswith('Genres')]
            if not filtered_items:
                return None
            genre_list = filtered_items[0].get('value')
            if not genre_list:
                return None
            # extract the actual names
            genre_name_list = [genre.get('text') for genre in genre_list]
            return join_string_items(genre_name_list)

    def get_restriction_age(self, content_detail, ott_metadata, base_info):
        if ott_metadata and ott_metadata.get(self.FIELD_OTT_RESTRICTION_AGE):
            return ott_metadata[self.FIELD_OTT_RESTRICTION_AGE]
        elif content_detail and content_detail.get(self.FIELD_VH_RESTRICTION_AGE):
            return content_detail[self.FIELD_VH_RESTRICTION_AGE]
        else:
            age_limit = base_info.get(self.FIELD_OO_RESTRICTION_AGE)
            if not age_limit or not isinstance(age_limit, str) or not age_limit.isdigit():
                return 0
            else:
                return int(age_limit)

    def get_content_type(self, content_type):
        if content_type not in self.content_type_mapping:
            return content_type
        return self.content_type_mapping[content_type]

    def get_release_year_from_oo(self, oo):
        if not oo or 'rich_info' not in oo:
            return None
        rich_info = oo['rich_info']
        if not rich_info or 'vh_meta' not in rich_info:
            return None
        vh_meta = rich_info['vh_meta']
        if not vh_meta or 'content_groups' not in vh_meta:
            return None
        content_groups = vh_meta['content_groups']
        if content_groups:
            return content_groups[0].get('year')
        return None

    def get_release_year(self, ott_metadata, content_detail, oo, content_id):
        if ott_metadata:
            years = ott_metadata.get('years')
            if years:
                return years[0]
        if content_detail:
            year = content_detail.get('release_year')
            if year:
                return year
        year = self.get_release_year_from_oo(oo)
        if year:
            return year
        else:
            logger.error("Can't obtain 'release_year' for id: %s", content_id)
            return None

    def get_trailer(self, oo: dict):
        try:
            trailer = oo[self.FIELD_VIEW][self.FIELD_GALLERY][self.FIELD_TRAILER]
        except (TypeError, KeyError):
            trailer = None
        if not trailer:
            onto_id = oo.get(self.FIELD_BASE_INFO, DEFAULT_DICT).get('id')
            logger.warning('No trailer for %s', onto_id)
            return None
        stream_url = oo.get('trailers_sign')
        if stream_url:
            trailer['stream_url'] = fix_schema(stream_url)
        url = trailer.get('url')
        if url:
            trailer.update({'url': fix_schema(url)})
        thumbnail_obj = trailer.get('thumbnail')
        if thumbnail_obj:
            thumbnail_obj.update({'thmb_href': fix_schema(thumbnail_obj.get('thmb_href'))})
        return trailer

    def get_title(self, raw_episode):
        name = raw_episode.get("title")
        episode_number = raw_episode.get("number")
        if name:
            return f'{episode_number}. {name}'
        else:
            return f'Эпизод {episode_number}'

    def get_poster(self, content_detail):
        if content_detail:
            return fix_schema(content_detail.get('onto_poster', ''))
        return ''

    def serialize_episodes(self, raw_episodes, season_number, player_id=None, context=None):
        assert raw_episodes, season_number
        result = []
        for raw_episode in raw_episodes:
            item = {
                'thumbnail': fix_schema(raw_episode.get('imageUrl')),
                EpisodeInfo.FIELD_CONTENT_TYPE: vh.CONTENT_TYPE_EPISODE,
                PlayerDetector.KEY_PLAYER_ID: player_id or PlayerDetector.get_player_id_for_carousel_item(raw_episode),
                'title': self.get_title(raw_episode),
                EpisodeInfo.FIELD_CONTENT_ID: raw_episode.get('contentId'),
                'duration_s': raw_episode.get(self.FIELD_DURATION),
                self.FIELD_SEASON_NUMBER: season_number,
                'episode_number': raw_episode.get('number')
            }
            license_info = get_licenses(raw_episode)
            fill_ya_plus_if_necessary(license_info, item)

            result.append(item)
        return result

    def get_seasons(self, obj, player_id=None, context=None):
        raw_seasons = obj.get('ott_series', DEFAULT_DICT).get('seasons', [])
        result = []
        for raw_season in raw_seasons:
            season_number = raw_season.get('number')
            episodes = raw_season.get('episodes')
            season = {
                'type': 'season',
                'episodes_count': len(episodes),
                'episodes': self.serialize_episodes(
                    raw_season.get('episodes'), season_number, player_id=player_id, context=context),
                self.FIELD_SEASON_NUMBER: season_number
            }
            result.append(season)
        return result

    def get_onto_category(self, base_info: dict, ott_metadata: dict, content_detail: dict) -> str:
        """
        Can return 3 values: 'film', 'series', None
        Actually, there is more values: 'film', 'anim_film', 'series', 'anim_series', None
        But for client it's OK to have just 'film' / 'series', since,
        in OO 'film' and 'anim_film' is 'MOVIE'
        and 'series' and 'anim_series' is 'TV_SERIES'
        :param base_info: object basic information about requested document(from OO response)
        :param ott_metadata: basic information about film from OTT
        :param content_detail: basic information about film from VH
        """
        onto_category = None
        if base_info:
            content_type = base_info.get('legal', DEFAULT_DICT).get('vh_licenses', DEFAULT_DICT).get('content_type')
            onto_category = self._get_onto_category_by_content_type(content_type)
        if not onto_category and ott_metadata:
            content_type = ott_metadata.get('contentType')
            onto_category = self._get_onto_category_by_content_type(content_type)
        if not onto_category and content_detail:
            onto_category = content_detail.get(self.FIELD_ONTO_CATEGORY)
        return onto_category

    def get_description(self, base_info, content_detail, ott_metadata):
        oo_description = base_info.get(self.FIELD_DESCRIPTION) if base_info else None
        vh_description = content_detail.get(self.FIELD_DESCRIPTION) if content_detail else None
        ott_description = ott_metadata.get(self.FIELD_DESCRIPTION) if ott_metadata else None
        return oo_description or vh_description or ott_description

    def get_thumbnail(self, content_detail, base_info):
        vh_thumbnail = fix_schema(content_detail.get('thumbnail')) if content_detail else None
        oo_thumbnail = base_info.get('image', DEFAULT_DICT).get('original') if base_info else None
        return vh_thumbnail or oo_thumbnail

    def _get_countries_from_oo_wiki_snippet(self, countries_container):
        """
        :param countries_container: list of items with next format:
        [
            {
              "_source_code": 10,
              "entref": "some str",
              "search_request": "some str",
              "text": "США",
              "list_id": "some str"
            },
            {
              "_source_code": 10,
              "entref": "some str",
              "search_request": "some str",
              "text": "Великобритания",
              "list_id": "some str"
            }
        ]
        :return: string with joined values of 'text' field, i.e. 'США, Великобритания'
        """
        country_name_list = []
        for item in countries_container:
            country_name_list.append(item.get('text'))
        if country_name_list:
            return join_string_items(country_name_list)

    def _get_countries_from_oo(self, oo):
        if oo:
            wiki_snippet_container = oo.get('wiki_snippet', DEFAULT_DICT).get('item')
            if wiki_snippet_container:
                for item in wiki_snippet_container:
                    if item.get('key') == 'Countries@on':
                        return self._get_countries_from_oo_wiki_snippet(item.get('value'))

    def get_countries(self, content_detail, ott_metadata, oo):
        if content_detail:
            countries = content_detail.get('countries')
            if countries:
                # No need to join anything here, since 'countries' in content_detail
                # is a string with comma-separated items
                return countries
        if ott_metadata:
            country_list = ott_metadata.get('countries')
            if country_list:
                return join_string_items(country_list)
        countries = self._get_countries_from_oo(oo)
        return countries or ''

    def get_duration(self, content_detail, ott_metadata):
        if content_detail and content_detail.get(self.FIELD_DURATION):
            return content_detail[self.FIELD_DURATION]
        elif ott_metadata and ott_metadata.get(self.FIELD_DURATION):
            return ott_metadata[self.FIELD_DURATION]
        return 0

    def get_cinemas(self, oo):
        cinema_data = oo.get('rich_info', DEFAULT_DICT).get('cinema_data', DEFAULT_DICT)
        if not cinema_data:
            return []

        if not cinema_data.get('cinemas'):
            return []

        cinemas = []
        for _item in cinema_data['cinemas']:
            try:
                item = self.cinema_serializer.serialize(_item)
            except self.cinema_serializer.UnsupportedCinema as err:
                logger.info(f'Skipping cinema: {err}')
                continue
            except Exception:
                logger.exception('Can not serialize cinema item')
                continue

            # не будем добавлять Кинопоиск в список кинотеатров
            # см SMARTTVBACKEND-1237
            if item['code'] == 'kp':
                continue

            cinemas.append(item)

        return cinemas

    def get_cover(self, content_detail, ott_metadata, base_info):
        if content_detail and content_detail.get('cover'):
            return fix_schema(content_detail['cover'])
        elif ott_metadata.get('coverUrl'):
            return fix_schema(ott_metadata['coverUrl'])
        elif base_info.get('thumbnail'):
            return fix_schema(base_info['thumbnail'])

        return None

    def serialize(self, obj: dict, context: dict) -> dict:
        if not context:
            raise ValueError("'context' must not be empty")
        content_detail = obj.get('content_detail')
        result = self.base_content_info_serializer.serialize(content_detail, context)
        oo = obj.get(self.FIELD_OBJECT_RESPONSE, DEFAULT_DICT)
        content_type = context.get('content_type')
        need_series = context.get('need_series')

        content_id = obj.get(BaseContentInfo.FIELD_CONTENT_ID)
        result[BaseContentInfo.FIELD_CONTENT_ID] = content_id
        base_info = oo.get(self.FIELD_BASE_INFO, DEFAULT_DICT)
        ott_metadata = obj.get(self.FIELD_OTT_METADATA, DEFAULT_DICT)
        result[self.FIELD_TITLE] = base_info.get(self.FIELD_TITLE)
        result[self.FIELD_DESCRIPTION] = self.get_description(base_info, content_detail, ott_metadata)
        result[BaseContentInfo.FIELD_CONTENT_TYPE] = self.get_content_type(content_type)
        result[self.FIELD_ONTO_CATEGORY] = self.get_onto_category(base_info, ott_metadata, content_detail)
        result['thumbnail'] = self.get_thumbnail(content_detail, base_info)
        result[self.FIELD_PERCENTAGE_SCORE] = base_info.get(self.FIELD_PERSONAL_SCORE)
        result['release_year'] = self.get_release_year(ott_metadata, content_detail, oo, content_id)
        result['restriction_age'] = self.get_restriction_age(content_detail, ott_metadata, base_info)
        result['genres'] = self.get_genres(obj)
        result['countries'] = self.get_countries(content_detail, ott_metadata, oo)
        result['directors'] = get_directors(obj)
        result['actors'] = get_actors(obj)
        result[FIELD_LOGO] = ott_metadata.get('logo', DEFAULT_DICT).get('url')
        result['cover'] = self.get_cover(content_detail, ott_metadata, base_info)
        result['trailer'] = self.get_trailer(oo)
        result[self.FIELD_RATINGS] = oo.get(self.FIELD_VIEW, DEFAULT_DICT).get(self.FIELD_RATINGS, [])
        result['cinemas'] = self.get_cinemas(oo)
        result['poster'] = self.get_poster(content_detail)

        player_id = PlayerDetector.get_player_id_for_card_detail(**obj)
        result[PlayerDetector.KEY_PLAYER_ID] = player_id

        if need_series and content_type in self.SERIES_CONTENT_TYPES:
            result.update({'seasons': self.get_seasons(obj, player_id=player_id, context=context)})
        if content_type in self.EPISODE_CONTENT_TYPES:
            duration = self.get_duration(content_detail, ott_metadata)
            if not duration:
                logger.warning('No duration for %s', content_id)
            result.update({
                'duration_s': duration,
            })
        if vh.is_ott_content(content_type) and content_detail:
            license_info = get_licenses(content_detail)
            fill_ya_plus_if_necessary(license_info, result)
            show_tv_promo = content_detail.get('show_tv_promo')
            if show_tv_promo:
                result.update({'show_tv_promo': show_tv_promo})

        fill_images_from_url(
            result.get(BaseContentInfo.FIELD_THUMBNAIL), FIELD_THUMBNAILS, result, get_platform_info(context))

        return result


class FullCardDetailV7(FullCardDetail):
    def serialize(self, obj, context=None) -> dict:
        result = super().serialize(obj, context)
        platform_info = get_platform_info(context)
        fill_images_from_url(result[FIELD_LOGO], FIELD_LOGOS, result, platform_info)
        fill_images_from_url(result[FIELD_COVER], FIELD_COVERS, result, platform_info)
        fill_images_from_url(result[FIELD_COVER], 'document_covers', result, platform_info, 'cover')
        fill_images_from_url(result[FIELD_POSTER], FIELD_POSTERS, result, platform_info)
        result['will_watch'] = self._get_will_watch(obj)
        return result

    def _get_will_watch(self, obj: dict) -> bool:
        try:
            return obj[self.FIELD_WILL_WATCH_CONTAINER]['exists']
        except (TypeError, KeyError):
            return False


class RatingsCardDetailSerializer(BaseSerializer):

    def get_percentage_score(self, oo):
        if FullCardDetail.FIELD_BASE_INFO in oo:
            base_info = oo[FullCardDetail.FIELD_BASE_INFO]
            if base_info:
                return base_info.get(FullCardDetail.FIELD_PERSONAL_SCORE, 0)
        return 0

    def get_ratings(self, oo):
        if FullCardDetail.FIELD_VIEW in oo:
            view = oo[FullCardDetail.FIELD_VIEW]
            if view:
                return view.get(FullCardDetail.FIELD_RATINGS)

    def serialize(self, obj, context=None) -> dict:
        oo = obj['object_response']

        return {
            FullCardDetail.FIELD_PERCENTAGE_SCORE: self.get_percentage_score(oo),
            FullCardDetail.FIELD_RATINGS: self.get_ratings(oo)
        }


class ProgressCardDetailSerializer(BaseSerializer):
    fields_to_copy = (EpisodeInfo.FIELD_PROGRESS, EpisodeInfo.FIELD_DURATION,
                      FIELD_EPISODE_NUMBER, FIELD_EPISODE_NAME, FIELD_SEASON_NUMBER,
                      EpisodeInfo.FIELD_CONTENT_ID)

    fields_to_rename = {
        EpisodeInfo.FIELD_CONTENT_TYPE_NAME: EpisodeInfo.FIELD_CONTENT_TYPE,
    }

    def serialize(self, obj, context=None) -> dict:
        result = super().serialize(obj, context)
        fill_images_from_url(
            orig_url=fix_schema(obj.get(VideoInfo.FIELD_THUMBNAIL)),
            target_name=FIELD_THUMBNAILS,
            destination=result,
            platform_info=get_platform_info(context)
        )
        result[FIELD_SEASON_NUMBER] = get_season_number(obj)
        result[EpisodeInfo.FIELD_CONTENT_URL] = 'https://fakeurl.com'
        self.filter_none_items(result)
        return result


class ProgramsSerializer(EpisodeInfo, BaseSerializer):
    END_TIME = 'end_time'
    PROGRAM_ID = 'program_id'
    START_TIME = 'start_time'
    HAS_CACHUP = 'has_cachup'
    HAS_SCHEDULE = 'has_schedule'
    PROGRAM_TITLE = 'program_title'
    TITLE = 'title'
    LIVE_STREAM = 'live_stream'
    COMBINED_TITLE = 'combined_title'

    NEED_MOVE_START = 'need_move_start'
    META = 'meta'
    WITHOUT_TIMELINE = 'without_timeline'

    SERIES_TITLE_PATTERN = re.compile(r'^\d.*сери[ия].*')

    fields_to_copy = [PROGRAM_TITLE, HAS_SCHEDULE, HAS_CACHUP, START_TIME, PROGRAM_ID, END_TIME] \
        + EpisodeInfo.fields_to_copy

    def is_live_stream(self, obj):
        need_move_start = obj.get('need_move_start')
        without_timeline = obj.get('meta', {}).get('without_timeline')
        if need_move_start is None and without_timeline is None:
            result = False
        elif need_move_start is None:
            result = without_timeline
        elif without_timeline is None:
            result = need_move_start
        else:
            result = without_timeline or need_move_start
        return not result

    def get_combined_title(self, data):
        title = data['title']
        program_title = data.get('program_title')
        if not program_title or title == program_title:
            return title
        return f'{program_title}. {title}'

    def get_title(self, data):
        title = data.get(self.FIELD_TITLE)
        if self.SERIES_TITLE_PATTERN.match(title):
            program_title = data.get(self.PROGRAM_TITLE)
            if program_title:
                return f'{program_title} - {title}'
        return title

    @classmethod
    def get_required_fields(cls) -> list:
        return [
            cls.END_TIME,
            cls.START_TIME,
            cls.PROGRAM_ID,
            cls.HAS_SCHEDULE,
            cls.TITLE,
        ]

    def is_valid(self, obj: Optional[dict]) -> bool:
        return obj and \
            obj.get(self.END_TIME) and \
            obj.get(self.START_TIME) and \
            obj.get(self.PROGRAM_ID) and \
            obj.get(self.HAS_SCHEDULE) and \
            obj.get(self.TITLE)

    def serialize(self, obj: Optional[dict], context: Optional[dict] = None) -> Optional[dict]:
        if not self.is_valid(obj):
            return None
        result = super().serialize(obj, context)
        result[self.LIVE_STREAM] = self.is_live_stream(obj)
        result[self.COMBINED_TITLE] = self.get_combined_title(obj)
        result[self.FIELD_TITLE] = self.get_title(obj)
        return result

    def serialize_many(self, episodes: list, context: Optional[dict] = None) -> list:
        result = []
        bad_items = []
        for episode in episodes:
            serialized_item = self.serialize(episode, context=context)
            if serialized_item:
                result.append(serialized_item)
            elif episode:
                bad_items.append(episode)
        if bad_items:
            logger.warning('Error parsing response for programs: %s', bad_items)
        return result


class EmbeddedSectionSerializer(BaseSerializer):
    FIELD_CONTENT_TYPE = 'content_type'
    FIELD_CAROUSEL_ID = 'carousel_id'
    FIELD_INCLUDES = 'includes'

    FIELD_THUMBNAIL = 'thumbnail'
    FIELD_TITLE = 'title'
    FIELD_URL = 'url'
    FIELD_BANNER_URL = 'banner_url'

    def serialize_item(self, item: Category2, context=None):
        result = {
            self.FIELD_TITLE: item.title,
            self.FIELD_URL: item.url
        }
        if item.display_content_type:
            result[self.FIELD_CONTENT_TYPE] = item.display_content_type
        if item.thumbnail_s3_key:
            result[self.FIELD_THUMBNAIL] = s3_client.get_url(item.thumbnail_s3_key)
        if item.banner_S3_key:
            result[self.FIELD_BANNER_URL] = s3_client.get_url(item.banner_S3_key)
        return result

    def serialize(self, obj: Category2, context=None) -> dict:
        result = super().serialize(obj, context)
        result[self.FIELD_CAROUSEL_ID] = obj.category_id
        result[self.FIELD_CONTENT_TYPE] = obj.content_type
        if obj.carousel_type:
            result[KpMultiSelectionsResponseFields.FIELD_CAROUSEL_TYPE] = obj.carousel_type
        result[self.FIELD_INCLUDES] = [self.serialize_item(item) for item in obj.includes]

        return result


class MusicInfiniteFeedSerializer(BaseSerializer):
    """
    Serialize api.music.yandex.net/infinite-feed output

    Check https://paste.yandex-team.ru/4240367/text for example response
    """
    AVATARS_CONFIG_KEY = 'music'
    FIELD_CAROUSEL_ID = 'carousel_id'
    FIELD_CONTENT_ID = 'content_id'
    FIELD_CONTENT_TYPE = 'content_type'
    FIELD_CAROUSEL_TYPE = 'carousel_type'
    FIELD_INCLUDES = 'includes'
    FIELD_ALICE_CALLBACK = 'alice_callback'
    FIELD_TITLE = 'title'
    FIELD_THUMBNAILS = 'thumbnails'

    SQUARE_SMALL = 'square_small'
    SQUARE_MEDIUM = 'square_medium'

    expected_errors = (KeyError, TypeError, ValueError)

    PLAYLIST_TYPES = ('playlist', 'auto-playlist')
    ARTIST_TYPE = 'artist'

    def serialize(self, music_response, raise_exception=False) -> dict:
        try:
            return self._serialize(music_response)
        except self.expected_errors:
            if raise_exception:
                raise
            logger.exception('Can not serialize music response')
            return {}

    def serialize_single_row(self, row, medium_size=False, title=None, carousel_id=None):
        try:
            return self._serialize_single_row(row, medium_size, title, carousel_id)
        except self.expected_errors:
            logger.exception('Can not serialize music row')
            return {}

    def _serialize(self, music_response) -> dict:
        """
        Serialize whole music response as a collection of carousels for Music section
        """
        carousels = []

        for index, item in enumerate(music_response['result']['rows']):
            is_first_row = index == 0  # first row has bigger size squares
            row = self.serialize_single_row(item, medium_size=is_first_row)
            if row:
                carousels.append(row)

        return {'carousels': carousels} if carousels else {}

    def _serialize_single_row(self, row, medium_size=False, title=None, carousel_id=None) -> dict:
        """
        Serialize result.rows[] single item as a carousel

        Row example: {
            "rowId": "wkuHQ52zkshKHK3F",
            "type": "general",
            "typeForFrom": "inf_feed_tag_playlists-tpp",
            "title": "Популярные плейлисты",
            "entities": [...]
        }
        """
        result = {}
        result[self.FIELD_INCLUDES] = []

        # serialize every entity as includes[] element
        for item in row['entities']:
            try:
                semantic_frame_from = row['typeForFrom']
                result[self.FIELD_INCLUDES].append(self.serialize_entity(item, semantic_frame_from))
            except self.expected_errors:
                logger.exception('Can not serialize music entity')

        if not result[self.FIELD_INCLUDES]:
            # carousel is not valid without items
            return {}

        result[self.FIELD_CAROUSEL_ID] = carousel_id if carousel_id else row['rowId']
        result[self.FIELD_CAROUSEL_TYPE] = self.SQUARE_MEDIUM if medium_size else self.SQUARE_SMALL
        result[self.FIELD_TITLE] = title if title else row['title']
        return result

    def serialize_entity(self, item: dict, frame_from: str):
        """
        Entity example: {
            "type": "playlist",
            "data": {
                "uid": 103372440,
                "kind": 2070,
                "title": "Русский поп: открытия",
                "modified": "2021-04-01T15:59:24+00:00",
                "likesCount": 44286,
                "cover": {
                    "type": "pic",
                    "dir": "/get-music-user-playlist/27701/qquadx1TRQCtaW/",
                    "version": "1617203735672",
                    "uri": "avatars.yandex.net/get-music-user-playlist/71140/qquadybyrbRoZo/%%?1617203735672",
                    "custom": true
                }
            }
        }
        """
        result = {}
        result[self.FIELD_CONTENT_ID] = f'{self.object_type(item)}-{self.object_id(item)}'
        result[self.FIELD_TITLE] = self.title(item)
        result[self.FIELD_CONTENT_TYPE] = 'music'
        result[self.FIELD_ALICE_CALLBACK] = self.get_music_play_semantic_frame(item, frame_from)
        self.fill_thumbnails(result, item)

        return result

    def title(self, item):
        if item['type'] == self.ARTIST_TYPE:
            return item['data']['name']
        else:
            return item['data']['title']

    def fill_thumbnails(self, result, item):
        try:
            if item['type'] in self.PLAYLIST_TYPES:
                uri = item['data']['cover']['uri']
            elif item['type'] == self.ARTIST_TYPE:
                uri = item['data']['uri']
            else:
                uri = item['data']['coverUri']
        except KeyError as err:
            logger.debug('Music item has no cover, using default. Error: %s, item: %s', err, item)
            uri = settings.MUSIC_DEFAULT_COVER_PATTERN

        fill_images_from_pattern(uri, self.FIELD_THUMBNAILS, result, self.AVATARS_CONFIG_KEY)

    def get_music_play_semantic_frame(self, item, frame_from):
        data = {
            "type": "server_action",
            "name": "@@mm_semantic_frame",
            "payload": {
                "typed_semantic_frame": {
                    "music_play_semantic_frame": {
                        "object_type": {
                            "enum_value": self.object_type(item)
                        },
                        "object_id": {
                            "string_value": self.object_id(item)
                        },
                        "from": {
                            "string_value": frame_from,
                        },
                        "disable_nlg": {
                            "bool_value": True
                        }
                    }
                },
                "analytics": {
                    "origin": "SmartSpeaker",
                    "purpose": self.get_purpose(item),
                }
            }
        }

        return json.dumps(data, ensure_ascii=False).encode('utf8')

    def object_type(self, item):
        if item['type'] in self.PLAYLIST_TYPES:
            return 'Playlist'

        return item['type'].capitalize()

    def object_id(self, item):
        if item['type'] in self.PLAYLIST_TYPES:
            return f"{item['data']['uid']}:{item['data']['kind']}"
        else:
            return item['data']['id']

    def get_purpose(self, item):
        return 'play_radio' if item['type'] == 'radio' else 'play_music'
