from rest_framework import fields

from smarttv.droideka.proxy.common import fix_schema, join_string_items


class AutoAddUrlSchemaField(fields.CharField):
    """
    Many urls comes from VH frontend without schema(http / https), for example:
    '//yandex.ru/...'
    In this case 'https:' schema must be added
    """

    def to_internal_value(self, data):
        return fix_schema(super().to_internal_value(data))

    def to_representation(self, value):
        return fix_schema(super().to_representation(value))


class JoinedStringField(fields.ListField):
    """
    Some fields like genres, actors, directors comes as string array
    This field converts string array to string, separated by specified separator
    For example:
    ["str1", "str2", "str3"] will be converted to string "str1, str2, str3" if separator is ', '
    """

    def __init__(self):
        super().__init__()
        self.required = False
        self.allow_empty = True
        self.allow_null = True
        self.child = fields.CharField()

    def to_representation(self, value):
        return join_string_items(value)


class OptionalListField(fields.ListField):
    """
    List field which may not be presented or may not contain any item
    """

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.required = False
        self.allow_null = True
        self.allow_empty = True


class PhantomIntegerField(fields.Field):
    """
    Represents fields which may be string number and may be integer
    Returns always integer or 0
    """

    def to_representation(self, value):
        try:
            return int(value)
        except (TypeError, ValueError):
            return value if value else 0

    def to_internal_value(self, data):
        try:
            return int(data)
        except (TypeError, ValueError):
            return data if data else 0


class CommaStringMultipleChoiceField(fields.MultipleChoiceField):
    """
    Borrowed from https://gist.github.com/sivy/1096283
    """

    def to_python(self, value):
        if isinstance(value, str):
            return [val.strip() for val in value.split(',')]
        else:
            return []

    def prepare_value(self, value):
        return self.to_python(value)

    def clean(self, value):
        self.run_validators(value)
        if isinstance(value, (list, tuple)):
            return ','.join([val.strip() for val in value])
        else:
            return ''
