from abc import ABC, abstractmethod
from copy import copy
from enum import Enum
import logging
import re
from urllib import parse
from collections import defaultdict
from typing import Optional, List
import html
from alice.protos.data.search_result.search_result_pb2 import TSearchResultData, TSearchResultGallery, TSearchResultItem
from alice.protos.data.video.video_pb2 import TVideoItem, TLightVideoItem, TAvatarMdsImage, \
    TVHLicenceInfo, TCollectionItem, TPersonItem
from alice.protos.data.video.content_details_pb2 import TContentDetailsItem
from alice.nlu.py_libs.request_normalizer.request_normalizer import RequestNormalizer
from alice.nlu.py_libs.request_normalizer.lang import Lang
from datetime import datetime
from google.protobuf.internal.containers import RepeatedCompositeFieldContainer
from cache_memoize import cache_memoize

from smarttv.droideka.proxy import cache
from smarttv.droideka.proxy.common import fix_schema, join_string_items
from smarttv.droideka.proxy.constants.orientation import Orientation
from smarttv.droideka.proxy.avatars_image_generator import get_proto_images_from_url, get_url_info, \
    GENERAL_AVATARS_MDS_REGEX
from smarttv.droideka.proxy.player_detection import PlayerDetector
from smarttv.droideka.utils import PlatformInfo
from smarttv.droideka.utils.decorators import ignore_errors
from smarttv.droideka.proxy.vh.constants import FIELD_LEGAL, FIELD_VH_LICENSES, FIELD_UUID
from smarttv.droideka.unistat.metrics import CacheType, LocalCachePlace
from smarttv.droideka.proxy.licenses import get_licenses
from smarttv.droideka.proxy.mem_cache import onto_id_mapping


logger = logging.getLogger(__name__)

ignore_key_and_type_error = ignore_errors((KeyError, TypeError))


FIELD_OBJECT = 'object'
FIELD_PARENT_COLLECTION = 'parent_collection'
FIELD_ENTITY_DATA = 'entity_data'
FIELD_THUMB = 'thumb'
FIELD_TYPE = 'type'
FIELD_OTT_CONTENT_TYPE = 'contentType'
FIELD_FILMS_WITH_ACTOR_ROLE = 'filmsWithActorRole'
FIELD_FILMS_WITH_DIRECTOR_ROLE = 'filmsWithDirectorRole'
FIELD_MONETIZATIONS = 'monetizations'
FIELD_MONETIZATION_MODEL = 'monetizationModel'
FIELD_PURCHASED = 'purchased'
FIELD_WATCHING_OPTION = 'watchingOption'
FIELD_POSTER_URL = 'posterUrl'
FIELD_HORIZONTAL_POSTER = 'horizontalPoster'
FIELD_FILM_ID = 'filmId'
FIELD_TV_SHOW = 'tv_show'

CTX_PLATFORM_INFO = 'platform_info'
CTX_BASE_INFO_HORIZONTAL_POSTER = 'base_info_horizontal_poster'
CTX_IS_BASE_INFO = 'is_base_info'
CTX_TITLE = 'title'
TV_SERIES = 'tv-series'
OTT_SERIES = 'ott-series'

FIELD_FILM = 'Film'
FIELD_SERIES = 'Series'

PROVIDER_KINOPOISK = 'kinopoisk'


OTT_CONTENT_TYPE_MAPPING = defaultdict(lambda: 'MOVIE', {
    TV_SERIES: 'TV_SERIES',
    OTT_SERIES: 'TV_SERIES',
    'ott-movie': 'MOVIE'
})


class ItemType(Enum):
    SERIES = FIELD_TV_SHOW
    MOVIE = 'movie'


class RelatedObjectItemType(Enum):
    FILM = FIELD_FILM
    PERSON = 'Hum'
    COLLECTION = 'List'
    MUSIC = 'Music'
    BAND = 'Band'
    UNKNOWN = 'unknown'

    @staticmethod
    def value_of(raw_value: Optional[str]) -> 'RelatedObjectItemType':
        try:
            return RelatedObjectItemType(raw_value)
        except ValueError:
            return RelatedObjectItemType.UNKNOWN


item_type_mapping = {
    FIELD_FILM: ItemType.MOVIE,
    'vod-episode': ItemType.MOVIE,
    FIELD_SERIES: ItemType.SERIES,
    'vod-library': ItemType.SERIES,
}


PERSON_KP_ID_REGEX = re.compile(r'name/(?P<kpid>[0-9]+)')


RELATED_OBJECT_CONTENT_TYPE_MAPPING = defaultdict(lambda: TSearchResultItem.EItemType.association, {
    RelatedObjectItemType.PERSON: TSearchResultItem.EItemType.actor,
    RelatedObjectItemType.COLLECTION: TSearchResultItem.EItemType.collection,
})


@cache_memoize(timeout=None, cache_alias='local',
               **cache.get_cache_memoize_callables(LocalCachePlace.NORMALIZED_TITLE.value, CacheType.LOCAL))
def get_normalized_title(title: str) -> str:
    return RequestNormalizer.normalize(Lang.RUS, title)


class BaseProtobufSerializer(ABC):
    @abstractmethod
    def serialize(self, raw_object: dict, result_protobuf, context: dict):
        pass


class VhLicenseInfoSerializer(BaseProtobufSerializer):
    def fill_avod(self, vh_licenses: dict, vh_license_info: TVHLicenceInfo):
        vh_license_info.Avod = 1 if 'avod' in vh_licenses else 0

    def fill_svod(self, vh_licenses: dict, vh_license_info: TVHLicenceInfo):
        try:
            vh_license_info.Svod.extend(vh_licenses['svod']['subscriptions'])
        except KeyError:
            pass

    def fill_fixed_price_subscriptions_info(
            self,
            vh_licenses: dict,
            vh_license_info: TVHLicenceInfo,
            subscription_name: str,
            has_subscription_origin_field_name: str,
            has_subscription_result_field_name: str,
            subscription_flag_origin_field_name: str,
            subscription_flag_result_field_name: str):
        setattr(vh_license_info, has_subscription_result_field_name, int(subscription_name in vh_licenses and
                has_subscription_origin_field_name in vh_licenses[subscription_name]))
        setattr(vh_license_info, subscription_flag_result_field_name, int(subscription_name in vh_licenses and
                subscription_flag_origin_field_name in vh_licenses[subscription_name]))

    def fill_tvod(self, vh_licenses: dict, vh_license_info: TVHLicenceInfo):
        if 'tvod' not in vh_licenses:
            vh_license_info.UserHasTvod = 0
            vh_license_info.Tvod = 0
            return
        tvod = vh_licenses['tvod']
        vh_license_info.UserHasTvod = int(tvod.get('TVOD', False))
        vh_license_info.Tvod = int('price' in tvod)

    def fill_est(self, vh_licenses: dict, vh_license_info: TVHLicenceInfo):
        if 'est' not in vh_licenses:
            vh_license_info.UserHasEst = 0
            vh_license_info.Est = 0
            return
        est = vh_licenses['est']
        vh_license_info.UserHasEst = int(est.get('EST', False))
        vh_license_info.Est = int('price' in est)

    def serialize(self, vh_licenses: dict, vh_license_info: TVHLicenceInfo, context: Optional[dict] = None):
        self.fill_avod(vh_licenses, vh_license_info)
        self.fill_svod(vh_licenses, vh_license_info)
        self.fill_tvod(vh_licenses, vh_license_info)
        self.fill_est(vh_licenses, vh_license_info)
        vh_license_info.ContentType = vh_licenses.get('content_type')


class BaseVideoItemSerializer:
    vh_license_info_serializer = VhLicenseInfoSerializer()

    @classmethod
    def _fill_type(cls, result: TVideoItem, lite_video_item: TLightVideoItem, subtypes: Optional[List[str]]):
        if not subtypes:
            logger.error('Can not detect type due to empty subtype list')
            return
        for subtype in subtypes:
            if FIELD_FILM in subtype:
                result.Type = 'movie'
            elif FIELD_SERIES in subtype:
                result.Type = FIELD_TV_SHOW
        lite_video_item.Type = result.Type
        if not result.Type:
            logger.error('Unknown wsubtypes: %s', subtypes)

    @classmethod
    def _fill_vh_uuid(cls, raw_object: dict, result: TVideoItem,
                      lite_video_item: TLightVideoItem):
        try:
            vh_uuid = raw_object[FIELD_LEGAL][FIELD_VH_LICENSES][FIELD_UUID]
            result.ProviderItemId = vh_uuid
            lite_video_item.ProviderItemId = vh_uuid
        except KeyError:
            pass

    @classmethod
    def _fill_kp_rating(cls, raw_object: dict, result: TVideoItem):
        if 'rating' not in raw_object:
            return
        for rating in raw_object['rating']:
            if rating['type'] == 'kinopoisk':
                # taken from
                # https://a.yandex-team.ru/arc/trunk/arcadia/smart_devices/android/tv/home-app/app/src/main/java/com/yandex/tv/home/search/model/AssociationsItem.kt?rev=r8120003#L180
                result.Rating = rating['original_rating_value'] * (10 / rating['original_best_rating'])
                return

    @classmethod
    def _fill_poster(cls, raw_object: dict, result: TAvatarMdsImage, platform_info: PlatformInfo):
        try:
            url = raw_object[FIELD_LEGAL][FIELD_VH_LICENSES]['svod']['import_poster']
        except KeyError:
            try:
                url = raw_object[FIELD_LEGAL][FIELD_VH_LICENSES]['import_poster']
            except KeyError:
                try:
                    url = raw_object['image']['original']
                except KeyError:
                    return
        get_proto_images_from_url(fix_schema(url), platform_info, Orientation.VERTICAL, result)

    @classmethod
    def _fill_base_info_horizontal_thumbnail(
            cls,
            result: TAvatarMdsImage,
            platform_info: PlatformInfo,
            horizontal_thumbnail: Optional[str]):
        get_proto_images_from_url(horizontal_thumbnail, platform_info, Orientation.HORIZONTAL, result)

    @classmethod
    def _fill_thumbnail(cls, raw_object: dict, result: TAvatarMdsImage, platform_info: PlatformInfo):
        try:
            get_proto_images_from_url(raw_object[FIELD_THUMB], platform_info, Orientation.HORIZONTAL, result)
        except KeyError:
            return

    @classmethod
    def _fill_assoc_image(cls,
                          image_object: dict,
                          result: RepeatedCompositeFieldContainer,
                          platform_info: PlatformInfo):
        try:
            url = image_object['original']
        except (KeyError, TypeError):
            return
        get_proto_images_from_url(fix_schema(url), platform_info, Orientation.VERTICAL, result.add())

    @classmethod
    def _fill_assoc_images(cls,
                           raw_object: dict,
                           result: RepeatedCompositeFieldContainer,
                           platform_info: PlatformInfo):
        collection_images = []
        try:
            collection_images.extend(raw_object['collection_images'])
        except (KeyError, TypeError):
            try:
                collection_images.append(raw_object['image'])
            except (KeyError, TypeError):
                return
        for image in collection_images:
            cls._fill_assoc_image(image, result, platform_info)

    @classmethod
    def _fill_vh_licenses(cls, raw_object: dict, result: TVideoItem):
        try:
            vh_licenses = raw_object[FIELD_LEGAL][FIELD_VH_LICENSES]
            cls.vh_license_info_serializer.serialize(vh_licenses, result.VhLicences)
        except KeyError:
            pass

    @classmethod
    def _fill_person_kp_id(cls, raw_object: dict, result: TVideoItem):
        try:
            raw_kp_id = raw_object['ids']['kinopoisk']
        except KeyError:
            logger.info('No KpId for person: %s', raw_object['id'])
            return
        match = PERSON_KP_ID_REGEX.match(raw_kp_id)
        if not match:
            logger.info('Can not extract KpId from "%s"', raw_kp_id)
            return
        result.KpId = match.group('kpid')

    @classmethod
    def fill_person_info(cls, raw_object: dict, person_item: TPersonItem, context: dict):
        person_item.Name = raw_object['name']
        person_item.Description = raw_object.get('description') or raw_object.get('subtitle') or ''
        if 'subtitle' in raw_object:
            person_item.Subtitle = raw_object['subtitle']
        person_item.Entref = raw_object['entref']
        person_item.SearchQuery = raw_object['search_request']
        cls._fill_person_kp_id(raw_object, person_item)
        cls._fill_poster(raw_object, person_item.Image, context[CTX_PLATFORM_INFO])

    @classmethod
    def fill_person_from_base_info(cls, result: TSearchResultItem, raw_object: dict, context: dict):
        result.ContentType = TSearchResultItem.EItemType.actor
        cls.fill_person_info(raw_object, result.PersonItem, context)

    @classmethod
    def fill_video_item_general_info(cls, raw_object: dict, result: TVideoItem, context: dict):
        platform_info = context[CTX_PLATFORM_INFO]
        lite_video_item = result.ProviderInfo.add()
        cls._fill_type(result, lite_video_item, raw_object['wsubtype'])
        result.ProviderName = PROVIDER_KINOPOISK
        lite_video_item.ProviderName = result.ProviderName
        cls._fill_vh_uuid(raw_object, result, lite_video_item)
        result.MiscIds.OntoId = raw_object['id']
        result.Available = 1
        lite_video_item.Available = result.Available
        result.Name = raw_object['title']
        result.NormalizedName = get_normalized_title(raw_object['title'])
        result.Description = raw_object.get('description') or raw_object.get('short_description') or ''
        cls._fill_kp_rating(raw_object, result)
        result.ReleaseYear = raw_object['release_year']
        age_limit = raw_object.get('age_limit')
        if age_limit:
            result.MinAge = int(age_limit)
            result.AgeLimit = age_limit
        result.Entref = raw_object['entref']
        cls._fill_poster(raw_object, result.Poster, platform_info)

        is_base_info = context.get(CTX_IS_BASE_INFO)
        base_info_horizontal_poster = context.get(CTX_BASE_INFO_HORIZONTAL_POSTER)
        if is_base_info and base_info_horizontal_poster:
            cls._fill_base_info_horizontal_thumbnail(result.Thumbnail, platform_info, base_info_horizontal_poster)
        else:
            cls._fill_thumbnail(raw_object, result.Thumbnail, platform_info)
        result.SearchQuery = raw_object['search_request']
        hint_description = raw_object.get('hint_description')
        if hint_description:
            result.HintDescription = hint_description
        cls._fill_vh_licenses(raw_object, result)


class ParentCollectionSerializer(BaseVideoItemSerializer, BaseProtobufSerializer):
    vh_license_info_serializer = VhLicenseInfoSerializer()

    def _get_parent_collection(self, obj: dict) -> Optional[dict]:
        try:
            return obj[FIELD_ENTITY_DATA][FIELD_PARENT_COLLECTION]
        except (KeyError, TypeError):
            return None

    def _serialize_parent_collection_object(
            self, result: TSearchResultItem, raw_object: dict, context: dict):
        result.ContentType = TSearchResultItem.EItemType.association
        self.fill_video_item_general_info(raw_object, result.VideoItem, context)

    def _fill_parent_collection(self, raw_parent_collection: dict, parent_collection: TSearchResultGallery,
                                context: dict):
        objects = raw_parent_collection[FIELD_OBJECT]
        for obj in objects:
            item = parent_collection.Items.add()
            self._serialize_parent_collection_object(item, obj, context)

    def serialize(self, obj: dict, result: TSearchResultData, context: dict):
        raw_parent_collection = self._get_parent_collection(obj)
        if not raw_parent_collection or FIELD_OBJECT not in raw_parent_collection:
            return None
        self._fill_parent_collection(raw_parent_collection, result.Galleries.add(), context)


class ClipsSerializer(BaseVideoItemSerializer, BaseProtobufSerializer):
    STRM_PLAYER_IDS = (PlayerDetector.VH_PLAYER, PlayerDetector.OTT_PLAYER, PlayerDetector.KINOPOISK_PLAYER)
    YOUTUBE_VIDEO_ID_REGEX = re.compile(r'https?://www\.youtube\.com/watch\?v=(?P<video_id>[\w\-]+)')
    CLIPS_MTIME_PATTERN = '%Y-%m-%dT%H:%M:%S%z'
    # Borrowed from https://a.yandex-team.ru/arc/trunk/arcadia/smart_devices/android/tv/home-app/app/src/main/java/com/yandex/tv/home/search/model/SearchVideoItem.kt?rev=r8226104#L71-75
    # TODO(alex-garmash): https://st.yandex-team.ru/SMARTTVBACKEND-860
    VIDEO_SRC_PATTERN = re.compile(r'.*src="(?P<embedded_url>[^"]+)"')

    def _make_base_info_context(self, context):
        context = copy(context)
        context[CTX_IS_BASE_INFO] = True
        return context

    def _fill_base_info(self, base_info: dict, context: dict, result: TSearchResultGallery):
        context = self._make_base_info_context(context)
        base_info_type = base_info['type']
        item: TSearchResultItem = result.Items.add()
        if base_info_type == 'Hum':
            self.fill_person_from_base_info(item, base_info, context)
            item.ContentType = TSearchResultItem.EItemType.actor
        else:
            self.fill_video_item_general_info(base_info, item.VideoItem, context)
            item.ContentType = TSearchResultItem.EItemType.entity

    def _extract_youtube_video_id(self, youtube_url: str) -> Optional[str]:
        match = self.YOUTUBE_VIDEO_ID_REGEX.match(youtube_url)
        if not match:
            logger.warning('Can not extract youtube video id from url: %s', youtube_url)
            return None
        return match.group('video_id')

    def _fill_clip_type(self, clip: dict, video_item: TVideoItem, lite_video_item: TLightVideoItem):
        video_item.Type = 'video'
        lite_video_item.Type = 'video'

        video_item.Available = 1
        lite_video_item.Available = 1

    def _fill_clip_provider_info(self, clip: dict, video_item: TVideoItem, lite_video_item: TLightVideoItem):
        uri = None
        if 'youtube' in clip['VisibleHost'].lower():
            provider_name = 'youtube'
            provider_id = self._extract_youtube_video_id(clip['url'])
            uri = clip['url']
        elif clip['player_id'] in self.STRM_PLAYER_IDS:
            provider_name = 'strm'
            provider_id = clip['vh_uuid']
            uri = clip['url']
        elif 'qproxy' in clip:
            provider_name = 'yavideo_proxy'
            provider_id = clip['url']
        else:
            provider_name = 'yavideo'
            provider_id = clip['url']

        if uri:
            video_item.PlayUri = uri
        video_item.ProviderName = provider_name
        lite_video_item.ProviderName = provider_name

        # TODO(alex-garmah): provider_id not available
        if provider_id:
            video_item.ProviderItemId = provider_id
            lite_video_item.ProviderItemId = provider_id

    def _fill_clip_thumbnail(self, raw_object: dict, result: TVideoItem, platform_info: PlatformInfo):
        try:
            get_proto_images_from_url(f'https:{raw_object["thmb_href"]}/orig',
                                      platform_info,
                                      Orientation.HORIZONTAL,
                                      result.Thumbnail)
        except KeyError:
            logger.info('Can not generate thumbnail for %s', raw_object['id'])

    def _fill_embedded_uri(self, clip: dict, video_item: TVideoItem):
        try:
            iframe_embedded_code = clip['players']['autoplay']['html']
        except (KeyError, TypeError):
            return
        if not iframe_embedded_code:
            return
        iframe_embedded_code = html.unescape(iframe_embedded_code)
        match = self.VIDEO_SRC_PATTERN.match(iframe_embedded_code)
        if not match:
            logger.info('Can not determine embedded url for clip: %s', clip['url'])
        embedded_url = match['embedded_url']
        parse_result: parse.ParseResult = parse.urlparse(embedded_url)
        video_item.EmbedUri = parse.urlunparse((
            'https',
            parse_result.netloc,
            parse_result.path,
            parse_result.params,
            parse_result.query,
            parse_result.fragment,
        ))

    def _fill_clip(self, clip: dict, result: TSearchResultItem, context: dict):
        result.ContentType = TSearchResultItem.EItemType.search
        video_item = result.VideoItem
        lite_video_item = video_item.ProviderInfo.add()
        self._fill_clip_provider_info(clip, video_item, lite_video_item)
        self._fill_clip_type(clip, video_item, lite_video_item)
        video_item.Name = clip['title']
        video_item.Duration = clip['duration']
        video_item.SourceHost = clip['VisibleHost']
        # TODO(alex-garmash): I don't see view count
        video_item.ContentUri = clip['url']
        self._fill_embedded_uri(clip, video_item)
        video_item.PlayerId = clip['player_id']
        if clip.get('mtime'):
            time = int(datetime.strptime(clip['mtime'], self.CLIPS_MTIME_PATTERN).timestamp())
            if time >= 0:  # TODO(alex-garmash): SMARTTVBACKEND-1024
                video_item.Mtime = time
        self._fill_clip_thumbnail(clip, video_item, context[CTX_PLATFORM_INFO])

    def _fill_clips(self, obj: dict, context: dict, result: TSearchResultGallery):
        for clip in obj['clips']:
            try:
                self._fill_clip(clip, result.Items.add(), context)
            except KeyError:
                pass

    def serialize(self, obj: dict, result: TSearchResultData, context: dict):
        gallery = None
        try:
            base_info = obj['entity_data']['base_info']
            gallery: TSearchResultGallery = result.Galleries.add()
            self._fill_base_info(base_info, context, gallery)
        except (KeyError, TypeError):
            pass
        if obj.get('clips'):
            if not gallery:
                gallery: TSearchResultGallery = result.Galleries.add()
            self._fill_clips(obj, context, gallery)


class RelatedObjectSerializer(BaseVideoItemSerializer, BaseProtobufSerializer):

    RELATED_OBJECT_ALLOW_LIST_TYPES = (
        RelatedObjectItemType.FILM,
        RelatedObjectItemType.COLLECTION,
        RelatedObjectItemType.PERSON,
    )

    @classmethod
    def serialize_list_related_object(cls, obj: dict, result: TCollectionItem, context: dict):
        result.Id = obj['id']
        result.Title = obj['title']
        result.Entref = obj['entref']
        result.SearchQuery = obj['search_request']

        # TODO(alex-garmash): use method below for generating sizes, when they become available
        # cls._fill_assoc_images(obj, result.Images, context[CTX_PLATFORM_INFO])
        for raw_image in obj['collection_images']:
            url_info = get_url_info(raw_image['original'], context[CTX_PLATFORM_INFO], GENERAL_AVATARS_MDS_REGEX)
            if not url_info:
                logger.info('Can not process url: %s', raw_image['original'])
                continue
            image: TAvatarMdsImage = result.Images.add()
            image.Sizes.append('orig')
            image.BaseUrl = url_info['base_url']

    def serialize(self, obj: dict, result: TSearchResultGallery, context: dict):
        result.Title = obj['list_name']
        for item in obj[FIELD_OBJECT]:
            result_item: TSearchResultItem = result.Items.add()
            item_type: RelatedObjectItemType = RelatedObjectItemType.value_of(item[FIELD_TYPE])

            result_item.ContentType = RELATED_OBJECT_CONTENT_TYPE_MAPPING[item_type]
            if item_type not in self.RELATED_OBJECT_ALLOW_LIST_TYPES:
                logger.debug('This item type %s can not be handled', item_type)
                return
            if item_type == RelatedObjectItemType.PERSON:
                RelatedObjectSerializer.fill_person_info(item, result_item.PersonItem, context)
            elif item_type == RelatedObjectItemType.COLLECTION:
                RelatedObjectSerializer.serialize_list_related_object(item, result_item.CollectionItem, context)
            elif item_type == RelatedObjectItemType.FILM:
                RelatedObjectSerializer.fill_video_item_general_info(item, result_item.VideoItem, context)
            else:
                raise NotImplementedError(f'Handling of type {item_type} is not implemented')


class SearchResultSerializer(BaseProtobufSerializer):
    parent_collection_serializer = ParentCollectionSerializer()
    clips_serializer = ClipsSerializer()
    related_object_serializer = RelatedObjectSerializer()

    RELATED_OBJECT_TYPES_ALLOW_LIST = (
        RelatedObjectItemType.FILM,
        RelatedObjectItemType.PERSON,
        RelatedObjectItemType.COLLECTION,
    )

    def _fill_related_object(self, obj: dict, context: dict, result: TSearchResultData):
        try:
            related_objects: list = obj[FIELD_ENTITY_DATA]['related_object']
        except (KeyError, TypeError):
            return
        for related_object in related_objects:
            if not related_object[FIELD_OBJECT]:
                return
            item_type = RelatedObjectItemType.value_of(related_object[FIELD_OBJECT][0][FIELD_TYPE])
            if item_type not in self.RELATED_OBJECT_TYPES_ALLOW_LIST:
                logger.debug('Item with type %s is no allowed', item_type)
                return
            self.related_object_serializer.serialize(related_object, result.Galleries.add(), context)

    def serialize(self, obj: dict, result: TSearchResultData, context: dict):
        self.parent_collection_serializer.serialize(obj, result, context)
        self.clips_serializer.serialize(obj, result, context)
        self._fill_related_object(obj, context, result)

        return result


class ContentDetailsDeviceStateSerializer(BaseProtobufSerializer):
    FIELD_CONTENT_ID = 'content_id'
    FIELD_CONTENT_TYPE = 'content_type'
    FIELD_RESTRICTION_AGE = 'restriction_age'

    def _get_item_type(self, content_type: str) -> Optional[str]:
        item_type: Optional[ItemType] = item_type_mapping.get(content_type)
        if not item_type:
            logger.warning('Unknown content type: %s', content_type)
            return None
        return item_type.value

    def serialize(self, raw_object: dict, result: TContentDetailsItem, context: dict = None):
        result.ProviderItemId = raw_object[self.FIELD_CONTENT_ID]
        result.ProviderName = PROVIDER_KINOPOISK
        result.ItemType = self._get_item_type(raw_object[self.FIELD_CONTENT_TYPE])
        result.AgeLimit = raw_object.get(self.FIELD_RESTRICTION_AGE)
        # TODO(alex-garmash): there is no such parameter here
        # result.SearchQuery = ...


def get_release_year(years_string):
    if '-' in years_string:
        return int(years_string.split('-')[0])
    return int(years_string)


class KpDocumentSerializer(BaseProtobufSerializer):

    def _fill_constant_fields(self, result: TVideoItem, provider_info: TLightVideoItem):
        result.ProviderName = PROVIDER_KINOPOISK
        provider_info.ProviderName = PROVIDER_KINOPOISK
        result.Available = 1

    @ignore_key_and_type_error
    def _fill_provider_item_id(self, raw_object: dict, result: TVideoItem, provider_info: TLightVideoItem):
        uuid = raw_object[FIELD_FILM_ID]
        result.ProviderItemId = uuid
        provider_info.ProviderItemId = uuid

    @ignore_key_and_type_error
    def _fill_type(self, raw_object: dict, result: TVideoItem, provider_info: TLightVideoItem):
        content_type = ItemType.SERIES if raw_object[FIELD_OTT_CONTENT_TYPE] in [TV_SERIES, OTT_SERIES] \
            else ItemType.MOVIE
        result.Type = content_type.value
        provider_info.Type = content_type.value

    @ignore_key_and_type_error
    def _fill_age_limit(self, raw_object: dict, result: TVideoItem):
        restriction_age = raw_object['restrictionAge']
        result.MinAge = restriction_age
        result.AgeLimit = str(restriction_age)

    @ignore_key_and_type_error
    def _fill_onto_id(self, raw_object: dict, result: TVideoItem):
        result.MiscIds.OntoId = onto_id_mapping[raw_object[FIELD_FILM_ID]]

    @ignore_key_and_type_error
    def _fill_name(self, raw_object: dict, result: TVideoItem):
        title = raw_object['title']
        result.Name = title
        result.NormalizedName = get_normalized_title(title)

    @ignore_key_and_type_error
    def _fill_duration(self, raw_object: dict, result: TVideoItem):
        result.Duration = raw_object['duration']

    @ignore_key_and_type_error
    def _fill_description(self, raw_object: dict, result: TVideoItem):
        result.Description = raw_object['shortDescription']

    @ignore_key_and_type_error
    def _fill_rating(self, raw_object: dict, result: TVideoItem):
        result.Rating = raw_object['kpRating']

    @ignore_key_and_type_error
    def _fill_genre(self, raw_object: dict, result: TVideoItem):
        result.Genre = join_string_items(raw_object['genres'])

    @ignore_key_and_type_error
    def _fill_release_year(self, raw_object: dict, result: TVideoItem):
        result.ReleaseYear = get_release_year(raw_object['years'])

    @ignore_key_and_type_error
    def _fill_vh_license_type(self, raw_object: dict, result: TVHLicenceInfo):
        result.ContentType = OTT_CONTENT_TYPE_MAPPING[raw_object[FIELD_OTT_CONTENT_TYPE]]

    def _fill_license_info(self, raw_object: dict, result: TVHLicenceInfo):
        if FIELD_WATCHING_OPTION not in raw_object:
            return
        watching_option = raw_object[FIELD_WATCHING_OPTION]
        monetizations = watching_option[FIELD_MONETIZATIONS]

        result.Avod = 1 if 'AVOD' in monetizations else 0

        result.Tvod = 1 if 'TVOD' in monetizations else 0
        result.UserHasTvod = 1 if result.Tvod and watching_option.get(FIELD_PURCHASED) else 0

        result.Est = 1 if 'EST' in monetizations else 0
        result.UserHasEst = 1 if result.Est and watching_option.get(FIELD_PURCHASED) else 0

        if 'SVOD' in monetizations and 'subscription' in watching_option:
            result.Svod.extend([watching_option['subscription']])

        self._fill_vh_license_type(raw_object, result)

    def _fill_player_id(self, raw_object: dict, result: TVideoItem):
        result.PlayerId = PlayerDetector.get_player_id_for_kp_carousel_item(raw_object)

    @ignore_key_and_type_error
    def _fill_poster(self, raw_object: dict, result: TVideoItem, context: dict):
        get_proto_images_from_url(raw_object[FIELD_POSTER_URL],
                                  context[CTX_PLATFORM_INFO], Orientation.VERTICAL, result.Poster)

    @ignore_key_and_type_error
    def _fill_thumbnail(self, raw_object: dict, result: TVideoItem, context: dict):
        get_proto_images_from_url(raw_object[FIELD_HORIZONTAL_POSTER],
                                  context[CTX_PLATFORM_INFO], Orientation.HORIZONTAL, result.Thumbnail)

    def _fill_hint_description(self, raw_object: dict, result: TVideoItem):
        hint_description = ''
        if raw_object.get('years'):
            hint_description += raw_object['years']
        if raw_object.get('genres'):
            if result:
                hint_description += ', '
            hint_description += ", ".join(raw_object['genres'])
        result.HintDescription = hint_description

    def serialize(self, raw_object: dict, result: TSearchResultItem, context: dict):
        logger.info(raw_object)
        result.ContentType = TSearchResultItem.EItemType.association

        video_item: TVideoItem = result.VideoItem
        provider_info = video_item.ProviderInfo.add()

        self._fill_type(raw_object, video_item, provider_info)
        self._fill_provider_item_id(raw_object, video_item, provider_info)
        self._fill_age_limit(raw_object, video_item)
        self._fill_onto_id(raw_object, video_item)
        self._fill_name(raw_object, video_item)
        self._fill_duration(raw_object, video_item)
        self._fill_description(raw_object, video_item)
        self._fill_rating(raw_object, video_item)
        self._fill_genre(raw_object, video_item)
        self._fill_release_year(raw_object, video_item)
        self._fill_license_info(raw_object, video_item.VhLicences)
        self._fill_player_id(raw_object, video_item)
        self._fill_poster(raw_object, video_item, context)
        self._fill_thumbnail(raw_object, video_item, context)
        self._fill_constant_fields(video_item, provider_info)
        self._fill_hint_description(raw_object, video_item)


class FilmographyResultSerializer:
    @staticmethod
    def merge_films(films_with_actor_role, films_with_director_role):
        films = films_with_actor_role
        for film in films_with_director_role:
            if film not in films:
                films.append(film)
        return films

    def serialize(self, raw_object: dict, result: TSearchResultData, context: dict):
        gallery: TSearchResultGallery = result.Galleries.add()
        for film in self.merge_films(raw_object[FIELD_FILMS_WITH_ACTOR_ROLE],
                                     raw_object[FIELD_FILMS_WITH_DIRECTOR_ROLE]):
            item: TSearchResultItem = gallery.Items.add()
            KpDocumentSerializer().serialize(film, item, context)


class VhCarouselAsSearchResultSerializer(BaseVideoItemSerializer):

    def _fill_type(self, _type: ItemType, video_item: TVideoItem, provider_info: TLightVideoItem):
        video_item.Type = str(_type.value)
        provider_info.Type = str(_type.value)

    def _fill_provider_name(self, video_item: TVideoItem, provider_info: TLightVideoItem):
        video_item.ProviderName = PROVIDER_KINOPOISK
        provider_info.ProviderName = PROVIDER_KINOPOISK

    def _fill_provider_item_id(self, raw_object: dict, video_item: TVideoItem, provider_info: TLightVideoItem):
        video_item.ProviderItemId = provider_info.ProviderItemId = raw_object['content_id']

    def _fill_available(self, video_item: TVideoItem, provider_info: TLightVideoItem):
        video_item.Available = 1
        provider_info.Available = 1

    def _fill_name(self, raw_object: dict, video_item: TVideoItem):
        video_item.Name = raw_object['title']
        video_item.NormalizedName = get_normalized_title(raw_object['title'])

    @ignore_key_and_type_error
    def _fill_description(self, raw_object: dict, video_item: TVideoItem):
        video_item.Description = raw_object['description']

    def _fill_release_year(self, raw_object: dict, video_item: TVideoItem):
        video_item.ReleaseYear = int(raw_object['release_year'])

    def _fill_onto_id(self, raw_object: dict, video_item: TVideoItem):
        video_item.MiscIds.OntoId = raw_object['onto_id']

    def _fill_age_limit(self, raw_object: dict, video_item: TVideoItem):
        age_limit = raw_object.get('restriction_age')
        if age_limit:
            video_item.MinAge = int(age_limit)
            video_item.AgeLimit = str(age_limit)

    def _fill_rating(self, raw_object: dict, video_item: TVideoItem):
        video_item.Rating = raw_object['rating_kp']

    def _fill_thumbnail(self, raw_object: dict, thumbnail: TAvatarMdsImage, platform_info: PlatformInfo):
        get_proto_images_from_url(fix_schema(raw_object['thumbnail']), platform_info, Orientation.HORIZONTAL, thumbnail)

    @ignore_key_and_type_error
    def _fill_poster(self, raw_object: dict, poster: TAvatarMdsImage, platform_info: PlatformInfo):
        get_proto_images_from_url(fix_schema(raw_object['onto_poster']), platform_info, Orientation.VERTICAL, poster)

    def _fill_vh_licenses(self, raw_object: dict, vh_licenses: TVHLicenceInfo):
        # defaults
        vh_licenses.Avod = 0
        vh_licenses.UserHasTvod = 0
        vh_licenses.Tvod = 0
        vh_licenses.UserHasEst = 0
        vh_licenses.Est = 0

        license_info = get_licenses(raw_object)
        primary = license_info.get_primary()
        if not primary:
            logger.error('Licenses for item not found: %s', raw_object)
            return

        if primary[FIELD_MONETIZATION_MODEL] == 'SVOD':
            vh_licenses.Svod.extend(license_info.get_legacy_ya_plus_array())
        elif primary[FIELD_MONETIZATION_MODEL] == 'EST':
            vh_licenses.UserHasEst = 1
        elif primary[FIELD_MONETIZATION_MODEL] == 'TVOD':
            vh_licenses.UserHasTvod = 1
        elif primary[FIELD_MONETIZATION_MODEL] == 'AVOD':
            vh_licenses.Avod = 1

        if raw_object['content_type_name'] == 'vod-episode':
            vh_licenses.ContentType = 'MOVIE'
        elif raw_object['content_type_name'] == 'vod-library':
            vh_licenses.ContentType = 'TV_SERIES'
        else:
            logger.error(f"Unknown vh item content_type: {raw_object['content_type_name']}")

    def _fill_hint_description(self, raw_object: dict, result: TVideoItem):
        try:
            parts = [str(raw_object['release_year'])]
            parts.extend(raw_object['genres'])
            result.HintDescription = ', '.join(parts)
        except (KeyError, ValueError):
            logger.warning('Can not generate hint description')
            result.HintDescription = ''

    def serialize_item(self, raw_object: dict, sri: TSearchResultItem, _type: ItemType, context: dict):
        platform_info = context[CTX_PLATFORM_INFO]

        # горизонтальная карточка
        sri.ContentType = TSearchResultItem.EItemType.entity

        video_item = sri.VideoItem
        provider_info = video_item.ProviderInfo.add()

        self._fill_type(_type, video_item, provider_info)
        self._fill_provider_item_id(raw_object, video_item, provider_info)
        self._fill_provider_name(video_item, provider_info)
        self._fill_available(video_item, provider_info)
        self._fill_name(raw_object, video_item)
        self._fill_description(raw_object, video_item)
        self._fill_release_year(raw_object, video_item)
        self._fill_onto_id(raw_object, video_item)
        self._fill_age_limit(raw_object, video_item)
        self._fill_rating(raw_object, video_item)
        self._fill_thumbnail(raw_object, video_item.Thumbnail, platform_info)
        self._fill_poster(raw_object, video_item.Poster, platform_info)
        self._fill_vh_licenses(raw_object, video_item.VhLicences)
        self._fill_hint_description(raw_object, video_item)

    def get_item_type(self, vh_item: dict):
        if not isinstance(vh_item, dict):
            return

        vh_type = vh_item.get('onto_otype', '')
        if vh_type.startswith('Film/Series'):
            return ItemType.SERIES
        elif vh_type.startswith('Film/Film'):
            return ItemType.MOVIE

        logger.error('Unknown VH response item type %r, assuming Film', vh_type)
        return ItemType.MOVIE

    def serialize(self, vh_response: dict, result: TSearchResultData, context: dict):
        gallery = result.Galleries.add()

        if CTX_TITLE in context:
            gallery.Title = context[CTX_TITLE]

        for raw_item in vh_response.get('includes', []):
            _type = self.get_item_type(raw_item)

            if _type in (ItemType.MOVIE, ItemType.SERIES):
                try:
                    sritem = TSearchResultItem()
                    self.serialize_item(raw_item, sritem, _type, context)
                    gallery.Items.append(sritem)
                except (KeyError, ValueError, AttributeError, TypeError):
                    logger.exception('Can not serialize vh response item, skipping')
            else:
                logger.error('Unsupported item type in vh carousel, skipping. Item: %r', raw_item)
                continue

        return result
