"""
Contains serializers for validating responses and forming result objects from raw responses from services like
videohosting, etc.
"""
from rest_framework import fields

from smarttv.droideka.proxy.common import join_string_items
from smarttv.droideka.proxy.models import Category2, ScreenSaver, Promo
from smarttv.droideka.proxy.s3mds import s3_client
from smarttv.droideka.proxy.serializers.base import BaseModelSerializer
from smarttv.droideka.proxy.categories_provider import KEY_PERSISTENT_CLIENT_CATEGORY_ID
from smarttv.droideka.proxy.constants.home_app_versions import VER_1_87_1_3467

from yaphone.utils.parsed_version import ParsedVersion


def get_payload_info_from_object_response(data):
    """
    Object response returns structure with huge nesting:
    {
      cards: [
        {... payload data ...}
      ]
    }

    root object of object response always attribute cards
    cards contains object which was requested
    Since, for now, we always query only 1 object, access object at 0 is fine

    obj.get('object_response', {}).get('cards', [])
    """
    cards = data.get('object_response', {}).get('cards', [])
    if cards:
        return cards[0]
    else:
        return None


def get_directors(data):
    """
    'item' - list of items with different information like directors, countries, genres    The structure is:
    object_response: {
      ...
      'wiki_snippet': {
        ...
        'item': [
          {'key': 'Director@role', 'value':[...]}
        ],
        ...
      },
      ...
    }    :return: joined with ', ' director names
    """
    additional_info_items = data.get('object_response', {}).get('wiki_snippet', {}).get('item', [])
    # filtered items will contain only directors or will be empty list
    filtered_items = [item for item in additional_info_items if item['key'] == 'Director@role']
    if not filtered_items:
        return None
    director_list = filtered_items[0].get('value')
    if not director_list:
        return None
    # extract the actual names
    director_name_list = [director.get('text') for director in director_list]
    return join_string_items(director_name_list)


def get_actors(data):
    """
    The structure in the source json:
    object_response: [
      {
        ...,
        type: "team",
        ...,
        object: [
          {
            ...,
            'title': 'Some Actor1',
            ...
          },
          {
            ...,
            'title': 'Some Actor2',
            ...
          },
          ...
        ],
        ...
      }
    ]
    :return: joined with ', ' actor names
    """
    related_object_items = data.get('object_response', {}).get('related_object', [])
    # filtered items will be only actors or empty list
    actors_items = [item for item in related_object_items if item.get('type') == 'team']
    # get first or None
    actors_container = next(iter(actors_items), None)
    if not actors_container:
        return None
    actors_list = actors_container.get('object')
    if not actors_list:
        return None
    actors_names_list = [actor.get('title') for actor in actors_list]
    return join_string_items(actors_names_list)


class CategorySerializer(BaseModelSerializer):
    hidden_fields = ('icon_s3_key',)
    KEY_CLIENT_ID = KEY_PERSISTENT_CLIENT_CATEGORY_ID

    icon = fields.CharField(required=False, allow_null=True, allow_blank=True)

    def to_representation(self, instance):
        data = super().to_representation(instance)
        if instance.icon_s3_key:
            data['icon'] = s3_client.get_url(instance.icon_s3_key)
        client_category_id_allowed = \
            ParsedVersion(self.context['platform_info'].app_version) >= VER_1_87_1_3467
        if not client_category_id_allowed or self.KEY_CLIENT_ID in data and not data.get(self.KEY_CLIENT_ID):
            del data[self.KEY_CLIENT_ID]
        return data

    class Meta:
        model = Category2
        fields = ('category_id', 'title', 'rank', 'icon_s3_key', 'parent_category_id', 'icon',
                  'persistent_client_category_id')


class ScreenSaverSerializer(BaseModelSerializer):
    hidden_fields = ('s3_key',)

    url = fields.CharField(required=False, allow_null=True, allow_blank=True)

    def to_representation(self, instance):
        data = super().to_representation(instance)
        if instance.s3_key:
            data['url'] = s3_client.get_url(instance.s3_key)
        return data

    class Meta:
        model = ScreenSaver
        fields = ('s3_key', 'url')


class PromoSerializer(BaseModelSerializer):
    hidden_fields = ('promo_s3_key', 'promo_id', 'promo_type', )

    class Meta:
        model = Promo
        fields = ('content_id', 'action', 'content_type', 'thumbnail', 'title', 'subtitle', 'fallback_action')
