import logging

from django.core.exceptions import ValidationError
from rest_framework import fields, serializers

from smarttv.droideka.proxy.models import ScreenSaver
from smarttv.droideka.proxy.serializers import fields as custom_fields
from smarttv.droideka.proxy.constants.carousels import CarouselsExternal, DroidekaCarousel, FILTERABLE_CAROUSEL_ID
from smarttv.droideka.utils import caching_date
from smarttv.droideka.proxy.models import ValidIdentifier
from smarttv.utils.machelpers import normalize_mac

logger = logging.getLogger(__name__)

MAX_LIMIT = 20


class CommaSeparatedListField(fields.Field):
    separator = ','

    def to_representation(self, obj):
        return self.separator.join(obj)

    def to_internal_value(self, data):
        return [item for item in data.split(self.separator) if len(item) > 0]


class PaginationMixin(metaclass=serializers.SerializerMetaclass):
    limit = fields.IntegerField(required=False, default=10, max_value=MAX_LIMIT)
    offset = fields.IntegerField(required=False, default=0)


class KidModeValidator(serializers.Serializer):
    valid_ages = (0, 6, 12, 16, 18)

    kid_mode = fields.BooleanField(required=False, help_text='Флаг, указывающий, что включен детский режим')
    restriction_age = fields.IntegerField(required=False, help_text='Возрастной порог возвращаемых фильмов. '
                                                                    'Учитывается, только если `kid_mode=True`, '
                                                                    'иначе - игнорируется')

    def validate(self, attrs):
        if attrs.get('kid_mode'):
            restriction_age = attrs.get('restriction_age')
            if restriction_age is None or restriction_age not in self.valid_ages:
                raise ValidationError(f'Wrong \'restriction_age\'({restriction_age}) must be one of {self.valid_ages}',
                                      code=400)
        else:
            attrs['restriction_age'] = None
        return attrs


# noinspection PyAbstractClass
class CarouselsValidator(PaginationMixin, KidModeValidator, serializers.Serializer):
    category_id = fields.CharField(required=True, help_text='ID категории')
    max_items_count = fields.IntegerField(required=False, default=10, max_value=MAX_LIMIT,
                                          help_text='Максимальное количество элементов в 1 карусели')
    cache_hash = fields.CharField(required=False, default=None, help_text='ID выдачи. Нужен для предотвращения дублей. '
                                                                          'Обязателен, если был в предыдущей пачке')
    purchases_available_only = fields.BooleanField(
        required=False, help_text='Вернет только доступные фильмы в карусели "Покупки". Имеет смысл, '
                                  'только для категорий, содержащих карусель "Покупки". Для остальных - игнорируется')


class CarouselsV7Validator(CarouselsValidator):
    external_carousel_offset = fields.IntegerField(
        required=False,
        default=0,
        max_value=MAX_LIMIT,
        help_text='Общее количество external каруселей, которые были подмешаны в предыдущие пачки каруселей'
    )


# noinspection PyAbstractClass
class CarouselValidator(PaginationMixin, KidModeValidator, serializers.Serializer):
    carousel_id = fields.CharField(required=True, help_text='ID карусели')
    docs_cache_hash = fields.CharField(required=False, default=None,
                                       help_text='ID выдачи для предотвращения дублей. Обязателен для передачи и был '
                                                 'передан клиенту в предыдущем запросе')
    carousel_type = fields.ChoiceField(required=False, choices=CarouselsExternal.TYPES, allow_null=True,
                                       help_text='Тип карусели. Указывает, в какой источник нужно идти за каруспелью'
                                                 '(VH / OTT). Не требуется указывать самостоятельно. Он автоматически'
                                                 ' добавляется в `more_url`')
    limit = fields.IntegerField(required=False, default=10, max_value=36, help_text='Количество элементов в карусели')
    filter = fields.CharField(required=False, help_text='Ключ по которому надо отфильтровать элементы карусели. '
                                                        'Может быть использован, только если перед запросом для этой '
                                                        'карусели был получен объект `filters`(как правило, это '
                                                        'происходит в ручке `carousels`)')
    tag = fields.CharField(required=False, help_text='Используется для фильтрации каруселей. '
                                                     'Автоматически проставляется при создании `base_url`. '
                                                     'Самостоятельно проставлять не требуется.')
    available_only = fields.BooleanField(
        required=False, help_text='Если `True` - Для карусели покупок вернется доступные для просмотра фильмы. Покупки '
                                  '- это в т.ч. и аренда. Срок аренды может истеч.')
    more_url_limit = fields.IntegerField(
        required=False, help_text='Перезаписывает limit для последующих запросов. (Т.е. для `more_url` будет '
                                  'игнорироваться лимит текущего запроса, а ьудет браться `more_url_limit`)\n'
                                  'В ответе ручки `carousels` могут приходить "вложенные карусели". '
                                  'Или по-другому - ссылка на карусель. Т.е. приходит просто объект с title, id и т.д. '
                                  'как у обычной карусели, но без элементов. Когда пользователь кликает на такой объект'
                                  ' - он переходит в grid. Чтобы заполнить все возможное пространство - для первого '
                                  'запроса такого "grid" выставляется `limit=36`, и выставляется `more_url_limit=12`. '
                                  'Таким образом, в 1ю пачку клиент получит 36 элементов, но дольше уже не будет '
                                  'запрашивать так много.')

    def validate(self, attrs):
        validated_data = super().validate(attrs)
        if (validated_data[DroidekaCarousel.FIELD_CAROUSEL_ID] != FILTERABLE_CAROUSEL_ID and
                (DroidekaCarousel.FIELD_TAG in validated_data or DroidekaCarousel.FIELD_FILTER in validated_data)):
            raise ValidationError(message=f"'filter' and 'tag' are not allowed for carousel "
                                          f"'{validated_data[DroidekaCarousel.FIELD_CAROUSEL_ID]}'")
        return validated_data


# noinspection PyAbstractClass
class ProgramsValidator(serializers.Serializer):
    parent_id = fields.CharField(
        required=True,
        help_text='ID канала, для которого запрашивается расписание. '
                  'Можно передавать несколько parent_id через запятую',
    )
    from_time = fields.IntegerField(
        required=False,
        default=caching_date.yesterday_start,
        help_text='Начало временного отрезка, за который запрашиваются программы. '
                  'Испольузет end_date__from в API VH (VideoHosting). '
                  'По умолчанию берётся один день назад.',
    )
    to_time = fields.IntegerField(
        required=False,
        default=caching_date.tomorrow_end,
        help_text='Конец временного отрезка, за который запрашиваются программы. '
                  'Использует start_date__to в API VH (VideoHosting). '
                  'По умолчанию берётся три дня вперёд.',
    )
    limit = fields.IntegerField(required=False, default=20)
    offset = fields.IntegerField(required=False, default=0)

    def to_internal_value(self, data):
        data = super().to_internal_value(data)
        data['from_time'] = caching_date.reduce_accuracy(data['from_time'])
        data['to_time'] = caching_date.reduce_accuracy(data['to_time'])
        return data


# noinspection PyAbstractClass
class Doc2DocRequestValidator(PaginationMixin, KidModeValidator):
    """
    Attributes:
        content_id(str): id of the content for which recommendations are required
        limit(int): how much objects will be returned(at max) in the response. Maximum is 20, default is 10
        offset(int): how much objects should be skipped from the start of ranked recommendation list. Default is 0
        rvb(str): token which must be used in case of requesting recommendations
            for the video from response of this hook. In other words, first request must be done without `rvb`,
            but futher requests for recommendation for content which was returned from this hook,
            must use corresponding `rvb` value
            Usually, this parameter is a very big string with random characters

    """
    content_id = fields.CharField(required=True, help_text='Id документа')
    rvb = fields.CharField(
        required=False,
        default=None,
        help_text='Данные для предотвращения циклов проигрывания. Значение берётся из rvb_mapping предыдущих ответов ручки',
    )


# noinspection PyAbstractClass
class CardDetailValidator(serializers.Serializer):
    content_id = fields.CharField(required=True)


# noinspection PyAbstractClass
class CardDetailV4Validator(KidModeValidator):
    content_id = fields.CharField(required=True, allow_blank=False, allow_null=False)
    onto_id = fields.CharField(required=False, allow_blank=False, allow_null=False)
    content_type = fields.CharField(required=False, allow_blank=False, allow_null=False)

    def validate(self, attrs):
        validated_data = super().validate(attrs)
        content_type = attrs.get('content_type')
        if 'onto_id' in validated_data and not content_type:
            raise ValidationError(message='\'content-type\' is required if \'onto_id\'')
        return validated_data


class CardDetailV9Validator(KidModeValidator):
    content_id = fields.CharField(required=False, allow_blank=False, allow_null=False)
    onto_id = fields.CharField(required=False, allow_blank=False, allow_null=False)
    content_type = fields.CharField(required=False, allow_blank=False, allow_null=False)

    def validate(self, attrs):
        validated_data = super().validate(attrs)
        if 'onto_id' not in validated_data and 'content_id' not in validated_data:
            raise ValidationError(message="either 'content_id' or 'onto_id' is required")
        return validated_data


# noinspection PyAbstractClass
class ScreenSaverValidator(PaginationMixin, serializers.Serializer):
    SCREEN_SAVER_MAX_LIMIT = 50
    SCREEN_SAVER_DEFAULT_LIMIT = 50

    limit = fields.IntegerField(required=False, default=SCREEN_SAVER_DEFAULT_LIMIT, max_value=SCREEN_SAVER_MAX_LIMIT)
    type = fields.ChoiceField(required=True, choices=ScreenSaver.TYPE_CHOICES)
    resolutions = custom_fields.CommaStringMultipleChoiceField(required=True, choices=ScreenSaver.RESOLUTION_CHOICES,
                                                               allow_empty=False, allow_null=False, allow_blank=False)


# noinspection PyAbstractClass
class AndroidPlatformSerializer(serializers.Serializer):
    platform_version = fields.CharField(allow_null=False, required=True)
    app_version = fields.CharField(allow_null=False, required=True)
    device_manufacturer = fields.CharField(allow_null=True, required=False)
    device_model = fields.CharField(allow_null=True, required=False)
    quasar_platform = fields.CharField(allow_null=True, required=False)


# noinspection PyAbstractClass
class ChannelsValidator(serializers.Serializer):
    show_hidden = fields.BooleanField(required=False, default=False)
    project_alias = fields.CharField(required=False)


# noinspection PyAbstractClass
class SearchValidator(serializers.Serializer):
    RESTRICTION_MODES = ('moderate', 'family', 'kids')
    RESTRICTION_AGES = (0, 6, 12)

    text = fields.CharField(required=True, help_text='Поисковый запрос')
    entref = fields.CharField(required=False,
                              help_text='Идентификатор для получения детальной информации о конкретном контенте')
    restriction_mode = fields.ChoiceField(
        choices=RESTRICTION_MODES,
        required=False,
        error_messages={
            'invalid_choice': f"Invalid value '{{input}}', must be one of {RESTRICTION_MODES}",
        },
        help_text='Режим фильтрации контента',
    )
    restriction_age = fields.ChoiceField(
        choices=RESTRICTION_AGES,
        required=False,
        error_messages={
            'invalid_choice': f"Invalid value '{{input}}', must be one of {RESTRICTION_AGES}",
        },
        help_text='Возраст для ограничения. Только для режима kids',
    )

    def validate(self, attrs):
        if 'restriction_mode' in attrs and attrs['restriction_mode'] == 'kids':
            if 'restriction_age' not in attrs:
                raise ValidationError('restriction_mode=kids, but restriction_age is not set')
        elif 'restriction_age' in attrs:
            raise ValidationError('restriction_age in set while restriction_mode is not kids')
        return attrs


# noinspection PyAbstractClass
class SuggestHistoryValidator(serializers.Serializer):
    text = fields.CharField(required=True, help_text='Поисковый запрос')


# noinspection PyAbstractClass
class BaseSetupWizardValidator(serializers.Serializer):
    serial_number = fields.CharField(required=False)


# noinspection PyAbstractClass
class SeriesSeasonsValidator(PaginationMixin, serializers.Serializer):
    series_id = fields.CharField(
        required=True,
        help_text="'content_id' сериала. Не эпизода, не сезона а именно сериала.")
    more_episodes_limit = fields.IntegerField(
        required=False,
        default=5,
        max_value=MAX_LIMIT,
        help_text='Лимит для more url эпизодов'
    )
    limit = fields.IntegerField(required=False, default=100, max_value=100)


# noinspection PyAbstractClass
class SeriesEpisodesValidator(PaginationMixin, serializers.Serializer):
    season_id = fields.CharField(
        required=True,
        help_text="'content_id' сезона сериала. Не эпизода, не сериала целиком а именно сезона."
    )


# noinspection PyAbstractClass
class UnboundSeriesEpisodesValidator(serializers.Serializer):
    series_id = fields.CharField(
        required=True,
        help_text="'content_id' сериала. Не эпизода, не сезона самого сериала."
    )
    season_number = fields.IntegerField(
        required=True,
        help_text="Номер сезона",
        min_value=1
    )
    pivot_episode_number = fields.IntegerField(
        required=True,
        help_text="Номер 'опорного' эпизода",
        min_value=1
    )
    backward_count = fields.IntegerField(
        required=False,
        default=5,
        help_text="Количество эпизодов до 'опорного'",
        max_value=10
    )
    forward_count = fields.IntegerField(
        required=False,
        default=5,
        help_text="Количество эпизодов после 'опорного'",
        max_value=10
    )
    stop_at_season_boundaries = fields.BooleanField(
        required=False,
        default=True,
        help_text="Ограничивать ли выборку эпизодами из параметра 'номер сезона'"
    )


# noinspection PyAbstractClass
class KpGiftsGivenValidator(serializers.Serializer):
    kp_gifts_id = fields.UUIDField(required=True)


# noinspection PyAbstractClass
class ExperimentsValidator(serializers.Serializer):
    test_ids = CommaSeparatedListField(required=False)


# noinspection PyAbstractClass
class IdentifierValidator(serializers.Serializer):
    type = fields.ChoiceField(choices=ValidIdentifier.TYPES, required=True)
    value = fields.RegexField(regex=r'([0-9a-f]{2}:){5}([0-9a-f]{2})', required=True)

    def to_representation(self, instance):
        return ValidIdentifier(type=instance['type'], value=instance['value'])


# noinspection PyAbstractClass
class IdentifierSerializer(serializers.Serializer):
    type = fields.CharField(required=False)
    value = fields.CharField(required=False)

    def to_representation(self, instance):
        result = super().to_representation(instance)

        if 'type' in result:
            result['type'] = result['type'].lower()

        if 'value' in result:
            result['value'] = normalize_mac(result['value'])

        return result


# noinspection PyAbstractClass
class ParentCollectionValidator(serializers.Serializer):
    ento = fields.CharField(required=True)


# noinspection PyAbstractClass
class ProfilingControlValidator(serializers.Serializer):
    state = fields.BooleanField(required=True)


# noinspection PyAbstractClass
class FilmographyValidator(serializers.Serializer):
    person_id = fields.CharField(required=True)


# noinspection PyAbstractClass
class SmotreshkaTvPlaybackBeginRequestValidator(serializers.Serializer):
    PURPOSES = ['change-media', 'resume', 'change-program', 'seek', 'restart-after-error']

    content_id = fields.CharField(required=True)
    purpose = fields.ChoiceField(required=False, allow_null=True, choices=PURPOSES, default='change-media')
    playback_session_id = fields.CharField(required=False, allow_null=True, allow_blank=True)
    target_timestamp = fields.CharField(required=False, allow_null=True)
    device_ip = fields.CharField(required=False, allow_null=True, allow_blank=True)
    device_name = fields.CharField(required=False, allow_null=True, allow_blank=True)
