import secrets
import string
import hashlib
import logging

from django.conf import settings
from cache_memoize import cache_memoize

from smarttv.droideka.protos.profile.profile_pb2 import TSmotreshkaProfile
from smarttv.droideka.proxy.api.smotreshka import client as smotreshka_api
from smarttv.droideka.proxy.api.smotreshka import billing_client
from smarttv.droideka.proxy.cache import smotreshka_profile
from smarttv.droideka.proxy.models import SmotreshkaChannel
from smarttv.droideka.proxy.player_detection import PlayerDetector

logger = logging.getLogger(__name__)


def begin(media_id: str, quasar_device_id: str, user_agent: str, device_ip: str = '', device_type: str = '',
          purpose: str = '', playback_session_id: str = ''):
    """
    Начать воспроизведение канала с заданным media_id для устройства
    """
    account = account_manager.get_or_register_account(quasar_device_id)
    if not account:
        raise NoAccountError(f'No account for device {quasar_device_id}')

    data = {
        'accountId': account.smotreshka_id,
        'mediaId': media_id,
        'purpose': purpose or 'change-media',
        'deviceInfo': {
            'deviceId': account.smotreshka_device_id,
            'deviceIp': device_ip,
            'deviceType': device_type or 'smarttv',
            'deviceUserAgent': user_agent,
            'deviceName': '',
        },
    }

    if playback_session_id:
        data['recyclePbsId'] = playback_session_id

    response = smotreshka_api.begin(data)

    return response


def get_medias(device_id: str, timezone_offset: str = ''):
    """
    Запрашивает медии для пользователя в часовом поясе Х
    """
    timezone = ''
    if timezone_offset:
        timezone = timezone_offset_to_timezone(timezone_offset)
    logger.info('Using timezone "%s"', timezone)

    # автоматически регистрирует аккаунт, если его нет
    account = account_manager.get_or_register_account(device_id)

    token = get_token(account.smotreshka_id, timezone)
    response = smotreshka_api.get_medias(token)

    return response['medias']


def get_token(account_id: str, region: str = ''):
    response = smotreshka_api.get_tv_asset_token_for_account(account_id, region)
    logger.info('Token response %s', response)

    return response['tvAssetToken']


class CreateAccountError(Exception):
    pass


class NoAccountError(Exception):
    """
    У этого пользователя нет созданного аккаунта в смотрешке
    """


class InvalidResponseError(ValueError):
    """
    Смотрешка отвечает в формате, который мы не понимаем
    """


class Account:
    smotreshka_id: str
    smotreshka_device_id: str

    def __init__(self, smotreshka_id, smotreshka_device_id):
        self.smotreshka_id = smotreshka_id
        self.smotreshka_device_id = smotreshka_device_id


class AccountManager:
    def __init__(self):
        self.billing_api = billing_client

    def get_account(self, quasar_device_id: str):
        profile = smotreshka_profile.get(quasar_device_id)
        if profile:
            return Account(profile.SmotreshkaAccountId, profile.SmotreshkaDeviceId)

    def generate_password(self, length=20):
        password_alphabet = string.ascii_letters + string.digits
        return ''.join(secrets.choice(password_alphabet) for i in range(length))

    def register_account(self, quasar_device_id: str):
        smotreshka_device_id = hashlib.md5(quasar_device_id.encode('utf-8')).hexdigest()
        kwargs = {
            'username': f'yatv_{smotreshka_device_id}',
            'email': f'{smotreshka_device_id}@yandex-tv.users-mail.lfstm.tv',
            'purchases': ['100'],
            'password': self.generate_password(),
        }

        logger.debug('registering account: %s', kwargs)

        try:
            result = self.billing_api.create_account(**kwargs)
        except self.billing_api.ApiError as err:
            exception = CreateAccountError(f'Can not register account: {err}')
            logger.error(exception)
            raise exception

        logger.info('result: %s', result)

        if not result or 'id' not in result:
            raise CreateAccountError(f'Incorrect answer: {result}')

        return smotreshka_device_id, result['id']

    def get_or_register_account(self, quasar_device_id: str):
        profile = smotreshka_profile.get(quasar_device_id)
        if not profile:
            logger.info('Profile for device %s not found', quasar_device_id)

            profile = TSmotreshkaProfile()
            profile.QuasarDeviceId = quasar_device_id

            smotreshka_device_id, account_id = self.register_account(quasar_device_id)
            profile.SmotreshkaDeviceId = smotreshka_device_id
            profile.SmotreshkaAccountId = account_id

            logger.info('Saving profile data: %s', profile)
            smotreshka_profile.set_many({
                profile.QuasarDeviceId: profile,
                profile.SmotreshkaDeviceId: profile,
            })
            logger.info('Profile data saved in redis')
        else:
            logger.info('Profile loaded %s', profile)

        return Account(profile.SmotreshkaAccountId, profile.SmotreshkaDeviceId)


@cache_memoize(timeout=settings.DEFAULT_RESPONSE_CACHE_TIME)
def get_saved_channels_data():
    """
    Возвращает данные для каналов Смотрешки, которые хранятся у нас
    и обогащаются потом от смотрешки только айдишниками каналов/медиа
    """

    query = SmotreshkaChannel.objects.filter(enabled=True).order_by('-number')

    result = []
    for channel in query:
        result.append({
            'channel_id': channel.channel_id,
            'smotreshka_channel_id': channel.smotreshka_channel_id,
            'title': channel.title,
            'has_cachup': 0,
            'catchup_age': 0,
            'channel_category': [channel.category],
            'main_color': channel.main_color,
            'description': '',
            'logo': '',
            'thumbnail': channel.thumbnail,
            'content_type': 'channel',
            'smarttv_icon': channel.smarttv_icon,
            'auto_fields': {
                'channel_smarttv_number': channel.number,
            },
            'player_id': PlayerDetector.SMOTRESHKA_PLAYER,
            'channel_provider': 'smotreshka',
        })

    return result


def timezone_offset_to_timezone(offset) -> str:
    if not offset:
        return ''

    try:
        offset_int = round(int(offset) / 60)
    except ValueError:
        return ''

    return f'special.timezone.utc-{offset_int:02}' if 1 <= offset_int <= 12 else ''

account_manager = AccountManager()
