import logging

from drf_yasg import openapi
from drf_yasg.utils import swagger_auto_schema

from smarttv.utils import headers
from yaphone.utils.parsers import parsers

from smarttv.droideka.proxy.serializers.serializers import KidModeValidator

logger = logging.getLogger(__name__)


def smarttv_swagger_schema(spec, description, summary=None):
    if summary is None:
        summary = description
    return swagger_auto_schema(manual_parameters=spec.get_request_params(),
                               responses=spec.response_schema,
                               operation_description=description,
                               operation_summary=summary,)


def swagger_schema(spec):
    return swagger_auto_schema(**spec)


class PlatformParamsSchemaProvider:
    """
    Returns openapi schema required for platform API docs generation
    """

    @staticmethod
    def get():
        return [
            openapi.Parameter(
                name=headers.DEVICE_ID_HEADER,
                in_=openapi.IN_HEADER,
                description='Device id из апп метрики',
                type=openapi.TYPE_STRING,
                required=True
            ),
            openapi.Parameter(
                name=headers.UUID_HEADER,
                in_=openapi.IN_HEADER,
                description='UUID из апп метрики',
                type=openapi.TYPE_STRING,
                required=True
            ),
            openapi.Parameter(
                name='User-Agent',
                in_=openapi.IN_HEADER,
                description='Примеры валидных значений User-Agent:\n'
                            '"com.yandex.tv.input.efir/1.2.616 (Realtek SmartTV; Android 7.1.1)"\n'
                            '"com.yandex.tv.services/1.2.2147483647 (Realtek SmartTV; Android 7.1.1)"\n'
                            '"com.yandex.tv.home/1.2.2147483647 (Realtek SmartTV; Android 7.1.1)"\n'
                            '"com.yandex.tv.videoplayer/1.2.616 (Realtek SmartTV; Android 7.1.1)"\n'
                            'В случае передачи некорректного User-Agent бэкэнд будет себя некорректно.'
                            'По вопросам изменения структуры User-Agent обращайтесь к разработчикам бэкэнда.',
                type=openapi.TYPE_STRING,
                required=True,
                pattern=parsers.USER_AGENT_STR_PATTERN
            ),
        ]


class KidModeParamsSchemaProvider:
    """
    Returns openapi schema required for platform API docs generation
    """

    @staticmethod
    def get():
        return [
            openapi.Parameter(
                name='kid_mode',
                in_=openapi.IN_QUERY,
                description='Флаг включения детского режима',
                type=openapi.TYPE_BOOLEAN,
                required=False
            ),
            openapi.Parameter(
                name='restriction_age',
                in_=openapi.IN_QUERY,
                description='Минимально допустимый возраст',
                type=openapi.TYPE_INTEGER,
                required=False,
                enum=KidModeValidator.valid_ages
            )
        ]


class PaginationParamSchemaProvider:
    @staticmethod
    def get():
        return [
            openapi.Parameter(
                name='offset',
                in_=openapi.IN_QUERY,
                description='Как много элементов нужно пропустить(смещение относительно начала списка).',
                type=openapi.TYPE_INTEGER,
                required=False
            ),
            openapi.Parameter(
                name='limit',
                in_=openapi.IN_QUERY,
                description='Сколько элементов получить в выдаче.'
                            'Максимальный лимит - 20',
                type=openapi.TYPE_INTEGER,
                required=False
            )
        ]
